package com.bcxin.tenant.data.etc.tasks.components.impls;

import com.bcxin.event.core.FlinkConstants;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.job.core.domain.CacheProvider;
import com.bcxin.event.job.core.domain.documents.enums.DocumentType;
import com.bcxin.event.job.core.domain.utils.DocumentTypeUtil;
import com.bcxin.tenant.data.etc.tasks.components.BinlogRawValue;
import com.bcxin.tenant.data.etc.tasks.components.CustomJdbcAcceptPreparedStatementParameter;
import com.bcxin.tenant.data.etc.tasks.components.CustomJdbcOutputFormatParameterWrapper;
import com.bcxin.tenant.data.etc.tasks.components.DataSourceUtil;
import com.bcxin.tenant.data.etc.tasks.components.builder.DataBuilderAbstract;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.flink.connector.jdbc.JdbcConnectionOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.sql.*;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.*;
import java.util.Date;

public class CustomJdbcAcceptPreparedStatementParameterImpl implements CustomJdbcAcceptPreparedStatementParameter {
    private static final Logger logger  = LoggerFactory.getLogger(CustomJdbcAcceptPreparedStatementParameterImpl.class);

    @Override
    public void accept(PreparedStatement statement,
                       BinlogRawValue o,
                       CacheProvider cacheProvider,
                       CustomJdbcOutputFormatParameterWrapper paramWrapper) {
        JsonProvider jsonProvider = new JsonProviderImpl();
        Map<String, Object> sourceParams = null;
        StringBuilder trace = new StringBuilder();
        List<int[]> pIndexes = paramWrapper.getParameterIndexes();
        List<String> pNames = paramWrapper.getParameterNames();
        Boolean hasDependencyTable =
                paramWrapper.getParameterNames().stream()
                        .anyMatch(ix -> !ix.toLowerCase().startsWith("before.") &&
                                !ix.toLowerCase().startsWith("after.")
                        );
        trace.append(String.format("hasDependencyTable=%s;", hasDependencyTable));
        try {
            sourceParams =
                    o.getParameters(
                            jsonProvider,
                            cacheProvider,
                            (dt, id) -> {
                                try {
                                    return getDocMapFromDb(
                                            paramWrapper.getJdbcConnectionOptions(),
                                            dt, statement.getConnection(),
                                            id
                                    );
                                } catch (Exception ex) {
                                    throw new BadEventException(String.format("获取数据(dt=%s;id=%s)发生异常", dt, id), ex);
                                }
                            },
                            hasDependencyTable
                    );
            Object sourceTable = sourceParams.get("source.table");
            DocumentType documentType = DocumentTypeUtil.translate2CacheableDocumentType(String.valueOf(sourceTable));
            if (documentType != null) {
                cacheProvider.expireCache(documentType, String.valueOf(o.getReadyPkId()));
            }
            trace.append(sourceParams == null ? "sourceParams is null " : ("sourceParams 大小=" + String.valueOf(sourceParams.size())));

            for (int index = 0; index < pIndexes.size(); index++) {
                String fieldName = pNames.get(index);
                trace.append(String.format("fieldName=%s;", fieldName));
                Object fieldValue = getFieldValueFromMap(sourceParams, fieldName, null);
                trace.append(String.format("fieldValue=%s;", fieldValue));
                if (sourceTable != null &&
                        (!String.valueOf(sourceTable).equalsIgnoreCase("tenant_user_credentials") &&
                                !String.valueOf(sourceTable).equalsIgnoreCase("tenant_user_credential_details"))
                ) {
                    if (fieldValue != null && fieldValue instanceof Date) {
                        Date date = (Date) fieldValue;
                        Calendar calendar = Calendar.getInstance();
                        calendar.setTime(date);
                        if (calendar.get(Calendar.YEAR) < 1001 || calendar.get(Calendar.YEAR) >= 9990) {
                            try {
                                logger.error("错误数据={}的日期信息:{}", jsonProvider.getJson(sourceParams), date);
                            } catch (Exception ex) {
                                logger.error("时间解析处理异常:{}={}", fieldName, fieldValue, ex);
                            }
                        }
                    }
                }

                statement.setObject(index + 1, fieldValue);
            }
        } catch (Exception ex) {
            logger.error("v3: 执行归集的时候sql={};参数列表:{};跟踪={}",
                    paramWrapper.getSql(),
                    jsonProvider.getJson(pNames),
                    trace,
                    ex
            );

            throw new BadEventException(ex);
        }
    }

    private Object getFieldValueFromMap(Map<String, Object> sourceParams, String fieldName, String prefix) {
        if (StringUtils.hasLength(prefix)) {
            fieldName = String.format("%s.%s", prefix, fieldName);
        }

        String finalFieldName = fieldName;
        Object fieldValue = sourceParams.get(fieldName);
        if (fieldValue == null) {
            Optional<String> fieldNameOptional =
                    sourceParams.keySet().stream()
                            .filter(ix -> ix.equalsIgnoreCase(finalFieldName.trim()))
                            .findFirst();

            if (fieldNameOptional.isPresent()) {
                fieldValue = sourceParams.get(fieldNameOptional.get());
            }
        }
        if (FlinkConstants.isDateTimeField(finalFieldName)) {
            if (fieldValue instanceof Double) {
                fieldValue = Instant.ofEpochMilli((Long) fieldValue);
            } else if (fieldValue instanceof Long) {
                fieldValue = Timestamp.from(Instant.ofEpochMilli((Long) fieldValue));
            } else if (fieldValue instanceof String){
                try {
                    fieldValue =  new Date(DataBuilderAbstract.getTimeValue(fieldName, fieldValue));
//                    /**字符串转时间，需要加上8小时。
//                     * 数据库字段是timestamp，从binlog获取到的是2022-01-20T06:50:20Z的形式，要转成CST时间
//                     */
//                    if(((String) fieldValue).endsWith("Z")){
//                        fieldValue = DateUtils.addHours(parsedDate,8);
//                    }
                } catch (Exception ex) {
                    logger.error("获取字段发生异常:{}={}", fieldName, fieldValue, ex);
                }
            }
        }

        if (fieldValue == null && fieldName.contains(".after.")) {
            String replacedFieldName = fieldName.replace("after.", "");
            return getFieldValueFromMap(sourceParams, replacedFieldName, prefix);
        }

        return fieldValue;
    }

    private Map<String, String> getDocMapFromDb(
            JdbcConnectionOptions connectionOptions,
            DocumentType documentType, Connection connection,
            String id) throws SQLException {
        Map<String, String> map = new HashMap<>();
        for (int retryIndex = 0; retryIndex < 10; retryIndex++) {
            try (PreparedStatement statement = connection.prepareStatement(documentType.getFetchSql())) {
                statement.setString(1, id);
                try (ResultSet resultSet = statement.executeQuery()) {
                    while (resultSet.next()) {
                        if (map.isEmpty()) {
                            map.put("dt", documentType.name());
                            map.put("id", id);
                            map.put("datetime", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
                        }

                        int columnCount = resultSet.getMetaData().getColumnCount();
                        for (int index = 0; index < columnCount; index++) {
                            String columnName = resultSet.getMetaData().getColumnLabel(index + 1);
                            Object value = resultSet.getObject(columnName);
                            if (value != null) {
                                map.put(columnName, String.valueOf(value));
                            }
                        }
                    }
                }

                return map;
            } catch (Exception ex) {
                if (connectionOptions == null) {
                    throw new BadEventException(
                            String.format("找不到连接（数据sql=%s）对象信息:%s;异常:%s",
                                    documentType.getFetchSql(),
                                    ExceptionUtils.getMessage(ex),
                                    ExceptionUtils.getStackTrace(ex))
                    );
                }

                connection = DataSourceUtil.getDataSource(
                        connectionOptions.getDriverName(),
                        connectionOptions.getDbURL(),
                        connectionOptions.getUsername().get(),
                        connectionOptions.getPassword().get()
                ).getConnection();
                if (retryIndex > 5) {
                    throw new BadEventException(String.format("无法获取(sql=%s,dt=%s,id=%s)数据信息:%s",
                            documentType.getFetchSql(),
                            documentType, id,
                            ex
                    )
                    );
                }
            }
        }

        return map;
    }
}
