package com.bcxin.tenant.data.etc.tasks.components.builder;

import com.bcxin.event.core.FlinkConstants;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.utils.DebeziumUtil;
import com.bcxin.event.job.core.domain.CacheProvider;
import com.bcxin.event.job.core.domain.DockMapDbExtractor;
import com.bcxin.event.job.core.domain.documents.enums.DocumentType;
import com.bcxin.tenant.data.etc.tasks.components.JedisPoolFactory;
import lombok.Getter;
import org.apache.commons.lang3.time.DateUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.sql.Connection;
import java.time.Instant;
import java.util.Date;
import java.util.Map;
import java.util.Optional;

public abstract class DataBuilderAbstract implements Serializable {
    private static final Logger logger = LoggerFactory.getLogger(DataBuilderAbstract.class);

    public abstract Map<String, Object> build(CacheProvider cacheProvider, Map<String, Object> data, String tableName, DockMapDbExtractor dbExtractor);

    protected String getRedisKey(String db, String tableName, String domainId, String id) {
        return JedisPoolFactory.getRedisFormatKey(
                String.format(FlinkConstants.REDIS_HMAP_DB_NAME_FORMAT,
                        db, tableName, domainId, id)
        );
    }

    protected static Object getKeyValue(Map<String, Object> pams, String key) {
        if (pams == null || pams.isEmpty()) {
            return null;
        }

        Optional<String> keyOptional =
                pams.keySet().stream().filter(ix -> ix.equalsIgnoreCase(key)).findFirst();
        if (!keyOptional.isPresent()) {
            return null;
        }

        return pams.get(keyOptional.get());
    }

    protected DataCoreMeta getDataMeta(Map<String, Object> data) {
        if (data == null) {
            return null;
        }

        return new DataCoreMeta(data);
    }

    protected Object formatValue(String nodeKey, String nodeValue) {
        if (!StringUtils.hasLength(nodeKey)) {
            return null;
        }
        if (!StringUtils.hasLength(nodeValue)) {
            return null;
        }

        if (nodeValue.contains("0000-00-00") || nodeValue.contains("9999-")) {
            nodeValue = null;
        }

        try {
            if (FlinkConstants.isDateTimeField(nodeKey)) {
                return new Date(getTimeValue(nodeKey, nodeValue));
//                Date date = new Date(getTimeValue(nodeKey, nodeValue));
//                /**
//                 * redis缓存字符串转cst时间需要加8个小时
//                 */
//                return DateUtils.addHours(date,8);
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        return nodeValue;
    }

    public static long getTimeValue(String key, Object value) {
        if (value == null) {
            return Instant.now().getEpochSecond();
        }

        if (value != null && String.valueOf(value).contains("-")) {
            StringBuilder sb = new StringBuilder();
            try {
                String dateValue = String.valueOf(value);
                sb.append("值=" + dateValue);
                try {
                    String timeStr = dateValue.replace("T", " ").replace("Z","");
                    /**如果字符串以毫秒级,"yyyy-MM-dd HH:mm:ss.SSS"，需要转换成"yyyy-MM-dd HH:mm:ss"的格式*/
                    if (timeStr.contains(".")) {
                        timeStr = timeStr.substring(0, timeStr.indexOf("."));
                    }
                    /**比如入职日期，如果最后的秒是00，从数据库获取到的值的格式为2022-07-01T00:01*/
                    if(timeStr.indexOf(":") > 0 && timeStr.indexOf(":") == timeStr.lastIndexOf(":")){
                        timeStr = timeStr + ":00";
                    }
                    Date dateTime =  DateUtils.parseDate(timeStr,"yyyy-MM-dd HH:mm:ss");
                    /**如果是以Z结尾的，表示是utc时区，转成cst时区，需要加8小时*/
                    if(dateValue.endsWith("Z")){
                        dateTime = DateUtils.addHours(dateTime, 8);
                    }
                    return dateTime.getTime();
                } catch (Exception ex) {
                    if (dateValue.contains("T")) {
                        return DateUtils.parseDate(dateValue.substring(0, dateValue.indexOf("T")), "yyyy-MM-dd").getTime();
                    } else {
                        return DateUtils.parseDate(dateValue, "yyyy-MM-dd").getTime();
                    }
                }
            } catch (Exception e) {
                logger.error("【先跳过】类型转换失败:原值={};key={};异常={}", value, key, e.toString() + sb);
                return Instant.now().getEpochSecond();
                // throw new BadEventException(String.format("类型转换失败:原值=%s;key=%s", value, key));
            }
        }

        try {
            String xValue = String.valueOf(value);
            if (xValue.length() < 8) {
                return DebeziumUtil.translate2ValueInMils(Long.parseLong(String.valueOf(value)));
            }

            return Long.parseLong(String.valueOf(value));
        } catch (Exception ex) {
            return (long) Double.parseDouble(String.valueOf(value));
        }
    }

    @Getter
    protected static class DataCoreMeta implements Serializable {
        private final String dbName;
        private final String tableName;
        private final String domainId;
        private final String id;

        protected DataCoreMeta(Map<String, Object> data) {
            this.dbName = (String) data.get("source.db");
            this.tableName = (String) data.get("source.table");

            Object keyValue = getKeyValue(data, "after.id");
            String pId = keyValue == null ? null : keyValue.toString();
            /**
             * 针对非删除操作
             */
            if (StringUtils.hasLength(pId)) {
                String oId = (String) getKeyValue(data, "after.domainid");
                if (!StringUtils.hasLength(oId)) {
                    oId = (String) getKeyValue(data, "after.organization_id");
                }

                this.domainId = oId;
                this.id = String.valueOf(getKeyValue(data, "after.id"));
            } else {
                String oId = (String) getKeyValue(data, "before.domainid");
                if (!StringUtils.hasLength(oId)) {
                    oId = (String) getKeyValue(data, "before.organization_id");
                }

                this.domainId = oId;
                this.id = (String) getKeyValue(data, "before.id");
            }
        }
    }

    protected Map<String, Object> bindUserBusinessData(String userId, Map<String, Object> data, CacheProvider cacheProvider, DockMapDbExtractor dbExtractor) {
        Map<String, String> userBusinessMap =
                cacheProvider.getDocumentFromCache(
                        DocumentType.User_Business_Credentials,
                        userId, dbExtractor, 5 * 24 * 60);

        if (userBusinessMap != null && userBusinessMap.size() > 1) {
            for (String hKey : userBusinessMap.keySet()) {
                Object ucValue = formatValue(hKey, userBusinessMap.get(hKey));
                if (ucValue != null) {
                    String cValue = StringUtils.trimLeadingCharacter(
                            StringUtils.trimTrailingCharacter(
                                    String.valueOf(ucValue).replace(";;", ";"), ';'),
                            ';');
                    if (!cValue.equalsIgnoreCase(";")) {
                        data.put(String.format("userCredential.%s", hKey), cValue);
                    }
                }
            }
        }

        return data;
    }
}