package com.bcxin.tenant.data.etc.tasks.components;

import org.apache.commons.lang3.NotImplementedException;
import org.apache.flink.connector.jdbc.JdbcConnectionOptions;
import org.apache.flink.connector.jdbc.internal.connection.JdbcConnectionProvider;
import org.apache.flink.connector.jdbc.internal.connection.SimpleJdbcConnectionProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Date;
import java.util.Set;

/**
 * 集成连接池的JdbcConnectionProvider
 */
public abstract class SimpleDataSourceJdbcConnectionProvider implements JdbcConnectionProvider, Serializable {
    private static final Logger LOG = LoggerFactory.getLogger(SimpleJdbcConnectionProvider.class);

    private static final long serialVersionUID = 1L;

    private final JdbcConnectionOptions jdbcOptions;
    private transient Connection connection;
    private String prefixMessage = System.getProperty("ENV").concat(".")
            .concat(System.getProperty("CONFIG_FILE"));

    public SimpleDataSourceJdbcConnectionProvider(JdbcConnectionOptions jdbcOptions) {
        this.jdbcOptions = jdbcOptions;
    }

    @Override
    public Connection getConnection() {
        return connection;
    }

    @Override
    public boolean isConnectionValid() throws SQLException {
        return connection != null
                && connection.isValid(jdbcOptions.getConnectionCheckTimeoutSeconds());
    }

    @Override
    public Connection getOrEstablishConnection() throws SQLException, ClassNotFoundException {
        if (connection != null && !connection.isClosed()) {
            return connection;
        }

        connection = getDataSource(jdbcOptions).getConnection();

        return connection;
    }

    @Override
    public void closeConnection() {
        if (connection != null) {
            try {
                connection.close();
            } catch (SQLException e) {
                LOG.warn("JDBC connection close failed.", e);
            } finally {
                connection = null;
            }
        }
    }

    @Override
    public Connection reestablishConnection() throws SQLException, ClassNotFoundException {
        closeConnection();
        return getOrEstablishConnection();
    }

    protected DataSource getDataSource(JdbcConnectionOptions jdbcOptions) {
        DataSource dataSource = DataSourceUtil.getDataSource(
                jdbcOptions.getDriverName(),
                jdbcOptions.getDbURL(),
                jdbcOptions.getUsername().get(),
                jdbcOptions.getPassword().get()
        );

        return dataSource;
    }
}
