package com.bcxin.tenant.data.etc.tasks.components;

import com.bcxin.event.core.KafkaConstants;
import com.bcxin.flink.streaming.cores.utils.KafkaUtils;
import com.bcxin.tenant.data.etc.tasks.dtos.DtqRecordDto;
import org.apache.flink.api.common.io.RichOutputFormat;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.util.concurrent.ExecutorThreadFactory;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

public class KafkaOutputFormat extends RichOutputFormat<DtqRecordDto> {
    private final Queue<DtqRecordDto> batchQueue = new LinkedList<>();
    private final Map<String, Object> kafkaPropertyConfig;
    private final int batchIntervalMs;
    private final int batchSize;
    private final String topic;
    private transient KafkaProducer<String, String> kafkaProducer;
    private transient ScheduledFuture<?> scheduledFuture;
    private final String bootstrapServer;

    public KafkaOutputFormat(
            String bootstrapServer,
            String topic,
            int batchSize,int batchIntervalMs) {
        this.batchSize = batchSize;
        this.batchIntervalMs = batchIntervalMs;
        this.bootstrapServer = bootstrapServer;

        this.kafkaPropertyConfig = new HashMap<>();
        this.kafkaPropertyConfig.put(
                ProducerConfig.BOOTSTRAP_SERVERS_CONFIG,
                bootstrapServer);
        this.kafkaPropertyConfig.put(
                ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG,
                "org.apache.kafka.common.serialization.StringSerializer");
        this.kafkaPropertyConfig.put(
                ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG,
                "org.apache.kafka.common.serialization.StringSerializer");

        this.topic = topic;
    }

    @Override
    public void configure(Configuration parameters) {

    }

    @Override
    public void open(int taskNumber, int numTasks) throws IOException {
        ScheduledExecutorService scheduler =
                Executors.newScheduledThreadPool(
                        1, new ExecutorThreadFactory("kafka-dlq-output-format"));
        this.scheduledFuture =
                scheduler.scheduleWithFixedDelay(
                        () -> {
                            synchronized (KafkaOutputFormat.class) {
                                flush();
                            }
                        },
                        this.batchIntervalMs,
                        this.batchIntervalMs,
                        TimeUnit.MILLISECONDS);

    }

    @Override
    public void writeRecord(DtqRecordDto record) throws IOException {
        batchQueue.add(record);

        if (batchQueue.size() >= this.batchSize) {
            synchronized (KafkaOutputFormat.class) {
                flush();
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (this.kafkaProducer != null) {
            this.kafkaProducer.close();
        }
    }

    private void flush() {
        if (!this.batchQueue.isEmpty()) {
            synchronized (KafkaOutputFormat.class) {
                while (!this.batchQueue.isEmpty()) {
                    if(kafkaProducer==null) {
                        synchronized (KafkaUtils.class) {
                            if (kafkaProducer == null) {
                                this.kafkaProducer = new KafkaProducer<String, String>(this.kafkaPropertyConfig);

                                Properties properties = new Properties();
                                properties.setProperty(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServer);
                                properties.setProperty(ProducerConfig.CLIENT_ID_CONFIG, String.format("admin-client-ensure-topic-%s", Thread.currentThread().getId()));
                                KafkaUtils.ensureTopic(this.topic, properties);
                            }
                        }
                    }

                    DtqRecordDto value = this.batchQueue.poll();
                    String key = String.format("%s.%s#%s", value.getDbName(), value.getTableName(), value.getId());
                    int partition =
                            Math.abs(key.hashCode()) % KafkaConstants.DEFAULT_PARTITION_COUNT;

                    ProducerRecord record = new ProducerRecord(this.topic, partition, key, new String(value.getValue()));

                    this.kafkaProducer.send(record);
                }

                this.batchQueue.clear();
            }
        }
    }
}
