package com.bcxin.tenant.data.etc.tasks;

import com.bcxin.event.core.FlinkJobAbstract;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.flink.streaming.cores.SystemPropertyUtil;
import com.bcxin.flink.streaming.cores.TableConstant;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigProperty;
import com.bcxin.flink.streaming.cores.properties.impls.DiskCheckpointConfigPropertyImpl;
import com.bcxin.flink.streaming.cores.utils.StorageUtil;
import com.bcxin.tenant.data.etc.tasks.jobs.DataEtcJob;
import com.bcxin.tenant.data.etc.tasks.properties.DataEtcConfigProperty;
import org.apache.commons.io.Charsets;
import org.apache.commons.io.IOUtils;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.runtime.entrypoint.StandaloneSessionClusterEntrypoint;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.Properties;
import java.util.stream.Collectors;

public class DataEtcApp {
    private static final Logger logger = LoggerFactory.getLogger(DataEtcApp.class);

    public static void main(String[] args) throws Exception {
        if (args.length < 2) {
            throw new BadEventException("必须传递环境信息和轨迹配置文件包名称(configs/环境/底下的文件名)");
        }

        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        /**
         * https://blog.csdn.net/magic_kid_2010/article/details/103902067
         * --env te --configFile 369
         */
        String env = parameterTool.get("env"); //args[0];//parameterTool.get("env");
        String configFile = parameterTool.get("configFile"); //args[1];//parameterTool.get("configFile");
        boolean isDebug = false;
        CheckpointConfigProperty checkpointConfigProperty = getCheckpointConfigProperty(env, configFile,parameterTool, isDebug);
        Collection<DataEtcConfigProperty> configProperties = getDataExcConfigs(env, configFile);

        String invalidConfigTopics = configProperties.stream()
                .flatMap(ix -> ix.getTopicSubscribers().stream())
                .filter(ix -> !StringUtils.hasLength(ix.getUid()))
                .map(ix -> ix.getTopic())
                .collect(Collectors.joining(";"));
        if (StringUtils.hasLength(invalidConfigTopics)) {
            throw new BadEventException(String.format("uid不能为空; 无效的主题(topic=%s)配置", invalidConfigTopics));
        }

        if (StringUtils.hasLength(invalidConfigTopics)) {
            String invalidConfigTopicsSubscriberContents = configProperties.stream()
                    .flatMap(ix -> ix.getTopicSubscribers().stream())
                    .flatMap(ix -> ix.getSubscriberContents().stream())
                    .filter(ix -> !StringUtils.hasLength(ix.getUid()))
                    .map(ix -> ix.getTitle())
                    .collect(Collectors.joining(";"));

            throw new BadEventException(String.format("uid不能为空; 无效的订阅(title=%s)配置", invalidConfigTopicsSubscriberContents));
        }


        FlinkJobAbstract job = new DataEtcJob(
                checkpointConfigProperty, configProperties, env, configFile, isDebug
        );

        job.execute();
    }


    private static Collection<DataEtcConfigProperty> getDataExcConfigs(String env,String configFile) throws IOException {
        StringBuilder sb = new StringBuilder();
        try {
            JsonProvider jsonProvider = new JsonProviderImpl();

            ClassLoader classLoader = DataEtcJob.class.getClassLoader();

            Collection<DataEtcConfigProperty> configProperties = new HashSet<>();
            sb.append(String.format("正在加载配置文件:%s;", configFile));
            try (InputStream coreStream = classLoader.getResourceAsStream(
                    String.format("configs/%s/%s.json", env, configFile))) {
                if (coreStream == null) {
                    throw new BadEventException(String.format("v2.找不到该环境配置:%s 及其文件:%s", env, configFile));
                }

                String content = IOUtils.toString(coreStream, Charsets.toCharset("UTF-8"));
                DataEtcConfigProperty configProperty =
                        jsonProvider.toObject(DataEtcConfigProperty.class, content);
                configProperties.add(configProperty);
            }

            System.setProperty("ENV",env);
            System.setProperty("CONFIG_FILE",configFile);

            return configProperties;
        } finally {
            logger.error("正在启动加载配置信息:环境={};消息={}", env, sb);
        }
    }

    private static CheckpointConfigProperty getCheckpointConfigProperty(
            String env,String businessSubPath,
            ParameterTool parameterTool,
            boolean isDebug) throws IOException {
        Properties properties = System.getProperties();

        ClassLoader classLoader = SystemPropertyUtil.class.getClassLoader();
        try (InputStream coreStream = classLoader.getResourceAsStream(
                String.format("configs/%s/flink-point.properties", env))) {
            if (coreStream == null) {
                throw new BadEventException(String.format("找不到该环境配置:%s", env));
            }

            properties.load(coreStream);
        }

        properties.put(TableConstant.PARAM_ENV, env);

        //String path = properties.getProperty(StreamingConfigConstants.DISK_CHECKPOINT_LOCATION);
        /**
         * 加businessSubPath子目录的目的是确保每个job独立

        if (OperatingSystem.isWindows()) {
            path = "e:\\checkpoints\\" + businessSubPath;
        } else {
            path += path.concat("/" + businessSubPath);
        }  */

        CheckpointConfigProperty configProperty = DiskCheckpointConfigPropertyImpl.create(
                businessSubPath,parameterTool

        );

        return configProperty;
    }
}
