package com.bcxin.flink.cdc.kafka.source.task.jobs;

import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.job.core.domain.dtos.RedisConfig;
import com.bcxin.flink.cdc.kafka.source.task.JobContext;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogCdcValue;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogOffsetValue;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.CdcSourceMeta;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.interceptors.CustomProducerInterceptor;
import com.bcxin.flink.cdc.kafka.source.task.compnents.JdbcDbReaderComponent;
import com.bcxin.flink.cdc.kafka.source.task.proerpties.CdcDatabaseSourceProperty;
import com.bcxin.flink.cdc.kafka.source.task.utils.RecordHeadUtil;
import com.bcxin.flink.streaming.cores.utils.KafkaUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.serialization.SerializationSchema;
import org.apache.flink.connector.kafka.sink.KafkaRecordSerializationSchema;
import org.apache.flink.connector.kafka.sink.KafkaSink;
import org.apache.flink.connector.kafka.sink.KafkaSinkBuilder;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.header.Headers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.util.*;

/**
 * https://ververica.github.io/flink-cdc-connectors/release-2.0/content/connectors/mysql-cdc.html
 */
public class DbStreamCdcKafkaJob extends DbStreamCdcJobAbstract {
    private final static Logger logger = LoggerFactory.getLogger(DbStreamCdcKafkaJob.class);
    private final JdbcDbReaderComponent jdbcDbReaderComponent ;

    public DbStreamCdcKafkaJob(JdbcDbReaderComponent jdbcDbReaderComponent) {
        this.jdbcDbReaderComponent = jdbcDbReaderComponent;
    }

    @Override
    protected void executeCoreAction(StreamExecutionEnvironment env, SingleOutputStreamOperator<BinlogCdcValue> dataStreamSource,
                                     BinlogOffsetValue binlogOffsetValue) {
        executeTargetKafkaSink(dataStreamSource);
    }

    /**
     * https://blog.csdn.net/boling_cavalry/article/details/105598224
     *
     * @param sourceStream
     */
    private void executeTargetKafkaSink(SingleOutputStreamOperator<BinlogCdcValue> sourceStream) {
        JobContext jobContext = JobContext.getInstance();
        Properties properties = new Properties();
        properties.setProperty(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG,
                jobContext.getKafkaConfigProperty().getBootstrapServer());

        if (!jobContext.isSkipBinlogRedisCalculated()) {
            RedisConfig redisConfig = RedisConfig.getDefaultFromMainThread();
            properties.setProperty(CustomProducerInterceptor.REDIS_CONFIG_KEY_HOST, redisConfig.getHost());
            properties.setProperty(CustomProducerInterceptor.REDIS_CONFIG_KEY_PORT, String.valueOf(redisConfig.getPort()));
            properties.setProperty(CustomProducerInterceptor.REDIS_CONFIG_KEY_PASSWORD, redisConfig.getPassword());

            properties.setProperty(ProducerConfig.INTERCEPTOR_CLASSES_CONFIG,
                    com.bcxin.flink.cdc.kafka.source.task.cdcs.interceptors.CustomProducerInterceptor.class.getName()
            );
        }

        properties.setProperty(ProducerConfig.CLIENT_ID_CONFIG,
                String.format("%s_%s", jobContext.getKafkaConfigProperty().getConsumerGroupId(), UUID.randomUUID())
        );
        properties.setProperty("group.id", jobContext.getKafkaConfigProperty().getConsumerGroupId());
        KafkaSinkBuilder<BinlogCdcValue> kafkaSinkBuilder = KafkaSink.<BinlogCdcValue>builder().setKafkaProducerConfig(properties);


        //KafkaRecordEmitter
        //KafkaPartitionSplitReader
        KafkaSink<BinlogCdcValue> kafkaSink = kafkaSinkBuilder
                .setRecordSerializer(new KafkaRecordSerializationSchema<BinlogCdcValue>() {
                    private volatile Map<String, String> topicMap;

                    @Override
                    public void open(SerializationSchema.InitializationContext context, KafkaSinkContext sinkContext) throws Exception {
                        KafkaRecordSerializationSchema.super.open(context, sinkContext);
                        this.topicMap = new HashMap<>();
                    }

                    @Override
                    public ProducerRecord<byte[], byte[]> serialize(BinlogCdcValue cdcValue, KafkaSinkContext context, Long timestamp) {
                        JsonProvider jsonProvider = new JsonProviderImpl();
                        String element = cdcValue.getValue();
                        CdcSourceMeta schema = jsonProvider.toObject(CdcSourceMeta.class, element);
                        if (schema == null || schema.getDbName() == null) {
                            logger.error("当前无效数据:{}", element);
                            throw new IllegalArgumentException(element);
                        }

                        String topicKey = jobContext.getKafkaConfigProperty()
                                .getActualTopicName(schema.getDbName(), schema.getTableName());
                        String key = schema.getId();
                        if (StringUtils.isEmpty(key)) {
                            key = element;
                        }

                        Headers headers = RecordHeadUtil.extractHeaders(schema, jdbcDbReaderComponent);
                        /**
                         * 临时禁用分区: todo:
                         */
                        if (!topicKey.contains("single-partition")) {
                            if (!topicMap.containsKey(topicKey)) {
                                KafkaUtils.ensureTopic(topicKey, properties);
                                topicMap.put(topicKey, Instant.now().toString());
                            }

                            return new ProducerRecord<>(
                                    topicKey,
                                    schema.getPartition(),
                                    key.getBytes(),
                                    element.getBytes(), headers
                            );
                        }


                        return new ProducerRecord<>(topicKey, 0, key.getBytes(), element.getBytes(), headers);
                    }
                })
                .build();

        CdcDatabaseSourceProperty databaseProperty = jobContext.getDatabaseProperty();

        sourceStream
                .sinkTo(kafkaSink)
                .uid(String.format("cdc-sink-%s-%s", databaseProperty.getServerId(), databaseProperty.getJobId()))
                .setParallelism(4);
    }

    @Override
    protected String getJobPrefixTitle() {
        return "2kafka";
    }
}