package com.bcxin.flink.cdc.kafka.source.task.jobs;

import com.bcxin.event.core.FlinkJobAbstract;
import com.bcxin.event.job.core.domain.dtos.RedisConfig;
import com.bcxin.flink.cdc.kafka.source.task.JobContext;
import com.bcxin.flink.cdc.kafka.source.task.StartupOptionUtil;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogCdcValue;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogOffsetValue;
import com.bcxin.flink.cdc.kafka.source.task.compnents.BinlogCheckpointProcessFunction;
import com.bcxin.flink.cdc.kafka.source.task.compnents.JsonBinlogMetaDebeziumDeserializationSchema;
import com.bcxin.flink.cdc.kafka.source.task.proerpties.CdcDatabaseSourceProperty;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigProperty;
import com.bcxin.flink.streaming.cores.utils.StorageUtil;
import com.ververica.cdc.connectors.mysql.source.MySqlSource;
import com.ververica.cdc.connectors.mysql.source.MySqlSourceBuilder;
import com.ververica.cdc.connectors.mysql.table.StartupOptions;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.api.common.eventtime.WatermarkStrategy;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.configuration.BlobServerOptions;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.contrib.streaming.state.EmbeddedRocksDBStateBackend;
import org.apache.flink.contrib.streaming.state.PredefinedOptions;
import org.apache.flink.streaming.api.CheckpointingMode;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.environment.CheckpointConfig;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.kafka.connect.json.DecimalFormat;
import org.apache.kafka.connect.json.JsonConverterConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.*;

public abstract class DbStreamCdcJobAbstract extends FlinkJobAbstract {
    private static Logger logger = LoggerFactory.getLogger(DbStreamCdcJobAbstract.class);

    /**
     * 默认采用flink cdc就不会造成数据库锁的问题
     *
     * @param env
     * @return
     */
    protected final SingleOutputStreamOperator<BinlogCdcValue> initDatabaseBinlog(StreamExecutionEnvironment env,
                                                                                  BinlogOffsetValue binlogOffsetValue) {
        JobContext jobContext = JobContext.getInstance();
        CdcDatabaseSourceProperty databaseProperty = jobContext.getDatabaseProperty();
        Properties debeziumProperties = new Properties();
        debeziumProperties.put("snapshot.locking.mode", "none");
        debeziumProperties.put("database.history.store.only.monitored.tables.ddl", "false");
        /**
         * The connector is trying to read binlog starting at Struct{version=1.6.4.Final,connector=mysql,name=mysql_binlog_source,ts_ms=1682067191784,db=,server_id=0,file=mysql-bin.001505,pos=50666530,row=0}, but this is no longer available on the server
         */

        //todo 230405 debeziumProperties
        String[] dbLists = Arrays.stream(databaseProperty.getDbList().split(";"))
                .filter(ii -> !StringUtils.isEmpty(ii)).toArray(String[]::new);

        String[] tableList = Arrays.stream(databaseProperty.getTableList().split(";"))
                .filter(ii -> !StringUtils.isEmpty(ii)).toArray(size -> new String[size]);
        logger.info("当前设置的serverId为:{}", databaseProperty.getServerId());

        logger.error("捕获的DB binlog的配置信息为:{}-{}", jobContext.getDatabaseProperty().getHostName(),
                databaseProperty.getUserName()
        );

        Map<String, Object> config = new HashMap();
        config.put(JsonConverterConfig.DECIMAL_FORMAT_CONFIG, DecimalFormat.NUMERIC.name());
        JsonBinlogMetaDebeziumDeserializationSchema jsonBinlogMetaDebeziumDeserializationSchema
                = new JsonBinlogMetaDebeziumDeserializationSchema(false, config);

        StartupOptions startupOptions = StartupOptionUtil.calculateStartupOption(binlogOffsetValue);
        logger.error("v3-当前的开始选项为={};offset={}", startupOptions.startupMode, startupOptions.binlogOffset);

        MySqlSourceBuilder<BinlogCdcValue> mySqlSourceBuilder = MySqlSource.<BinlogCdcValue>builder()
                .hostname(jobContext.getDatabaseProperty().getHostName())
                .port(jobContext.getDatabaseProperty().getPort())
                .databaseList(dbLists)
                .tableList(tableList)
                .username(databaseProperty.getUserName())
                .password(databaseProperty.getPassword())
                /**
                 * 降低每次获取的压力
                 */
                .fetchSize(1500)
                .serverId(databaseProperty.getServerId())
                /**
                 * 暂时从最新的拉取
                 */
                //.startupOptions(StartupOptions.specificOffset("mysql-bin.001655", 213076343))
                //.startupOptions(StartupOptions.initial())
                .startupOptions(startupOptions)
                /**
                 * 预发布
                 */
                //.startupOptions(StartupOptions.specificOffset("mysql-bin.003074",324574))
                .includeSchemaChanges(false)
                .connectionPoolSize(200)
                .connectTimeout(
                        Duration.of(databaseProperty.getConnectTimeout(),
                                ChronoUnit.MILLIS))
                //.deserializer(jsonDebeziumDeserializationSchema)
                .deserializer(jsonBinlogMetaDebeziumDeserializationSchema)
                .debeziumProperties(debeziumProperties);

        if (!StringUtils.isEmpty(databaseProperty.getConnectionTimeZone())) {
            mySqlSourceBuilder = mySqlSourceBuilder.serverTimeZone(databaseProperty.getConnectionTimeZone());
        }

        if (!StringUtils.isEmpty(databaseProperty.getServerId())) {
            mySqlSourceBuilder = mySqlSourceBuilder.serverId(databaseProperty.getServerId());
        }

        MySqlSource<BinlogCdcValue> mySqlSource =
                mySqlSourceBuilder.build();

        RedisConfig redisConfig = RedisConfig.getDefaultFromMainThread();
        SingleOutputStreamOperator<BinlogCdcValue> outputStreamOperator =
                env.fromSource(mySqlSource,
                                WatermarkStrategy.forBoundedOutOfOrderness(Duration.of(200, ChronoUnit.MILLIS)),
                                databaseProperty.getName()
                        )
                        .setParallelism(4)
                        .keyBy(ix -> ix.getFullTable())
                        .process(new BinlogCheckpointProcessFunction(redisConfig, binlogOffsetValue))
                        //.map(ix -> translate2CdcValue(ix))
                        //.map(new RichMapProcessFunction(redisConfig))
                        .uid(String.format("binlog-cdc-mp-%s-%s", databaseProperty.getServerId(), databaseProperty.getJobId()))
                        .name(String.format("数据源:%s", databaseProperty.getName()))
                        .setParallelism(4);

        return outputStreamOperator;
    }

    @Override
    protected void coreExecute() throws Exception {
        JobContext jobContext = JobContext.getInstance();
        Configuration configuration = new Configuration();
        /**
         * 每120秒清除一次, 临时日志
         */
        configuration.set(BlobServerOptions.CLEANUP_INTERVAL,120l);
        configuration.set(BlobServerOptions.STORAGE_DIRECTORY,StorageUtil.getTmpPath());
        StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment(configuration);
        //com.ververica.cdc.connectors.mysql.debezium.task.context.StatefulTaskContext
        /**
         * https://nightlies.apache.org/flink/flink-docs-release-1.17/docs/ops/state/state_backends/#incremental-checkpoints
         * 启用增量式的checkpoint
         */
        EmbeddedRocksDBStateBackend rocksDBStateBackend = new EmbeddedRocksDBStateBackend(true);
        /**
         * https://nightlies.apache.org/flink/flink-docs-release-1.17/docs/ops/state/state_backends/#predefined-per-columnfamily-options
         */
        rocksDBStateBackend.setPredefinedOptions(PredefinedOptions.SPINNING_DISK_OPTIMIZED_HIGH_MEM);
        env.setStateBackend(rocksDBStateBackend);

        CheckpointConfigProperty configProperty = jobContext.getConfigProperty();
        String checkpointPath = configProperty.getCheckpointPath();

        env.getCheckpointConfig().setCheckpointStorage(checkpointPath);
        logger.error("cdc.checkpoint的PointStorage位置={};", configProperty.getCheckpointPath());
        env.enableCheckpointing(150_000);
        env.setRestartStrategy(RestartStrategies.fallBackRestart());
        /**
         * cp之间, 至少间隔20秒
         */
        env.getCheckpointConfig().setMinPauseBetweenCheckpoints(20_000);
        env.getCheckpointConfig().setCheckpointingMode(CheckpointingMode.EXACTLY_ONCE);
        env.getCheckpointConfig().setMaxConcurrentCheckpoints(1);
        env.getCheckpointConfig().setExternalizedCheckpointCleanup(CheckpointConfig.ExternalizedCheckpointCleanup.RETAIN_ON_CANCELLATION);
        env.getCheckpointConfig().setTolerableCheckpointFailureNumber(10);

        BinlogOffsetValue binlogOffsetValue = StartupOptionUtil.getBinlogOffsetValue(
                RedisConfig.getDefaultFromMainThread(),
                jobContext.getDatabaseProperty()
        );
        SingleOutputStreamOperator<BinlogCdcValue> dataStreamSource = initDatabaseBinlog(env, binlogOffsetValue);
        executeCoreAction(env, dataStreamSource, binlogOffsetValue);
        env.execute(String.format("flink-v10-%s-%s-%s",
                getJobPrefixTitle(),
                jobContext.getEnv(),
                jobContext.getName(),
                jobContext.getDatabaseProperty().getJobId()));
    }

    protected abstract void executeCoreAction(StreamExecutionEnvironment env,
                                              SingleOutputStreamOperator<BinlogCdcValue> dataStreamSource,
                                              BinlogOffsetValue binlogOffsetValue
    );

    protected abstract String getJobPrefixTitle();
}
