package com.bcxin.flink.cdc.kafka.source.task.compnents;

import com.alibaba.fastjson.JSONObject;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.BinlogCdcValue;
import com.bcxin.flink.streaming.cores.utils.DebeziumJsonNodeDtoUtils;
import com.ververica.cdc.debezium.DebeziumDeserializationSchema;
import org.apache.flink.api.common.typeinfo.BasicTypeInfo;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.runtime.state.FunctionInitializationContext;
import org.apache.flink.runtime.state.FunctionSnapshotContext;
import org.apache.flink.streaming.api.checkpoint.CheckpointedFunction;
import org.apache.flink.util.Collector;
import org.apache.kafka.connect.json.JsonConverter;
import org.apache.kafka.connect.json.JsonConverterConfig;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.storage.ConverterConfig;
import org.apache.kafka.connect.storage.ConverterType;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * 加这个的目的
 */
public class JsonBinlogMetaDebeziumDeserializationSchema implements DebeziumDeserializationSchema<BinlogCdcValue> {
    private static final long serialVersionUID = 1L;

    private transient JsonConverter jsonConverter;

    /**
     * Configuration whether to enable {@link JsonConverterConfig#SCHEMAS_ENABLE_CONFIG} to include
     * schema in messages.
     */
    private final Boolean includeSchema;

    /**
     * The custom configurations for {@link JsonConverter}.
     */
    private Map<String, Object> customConverterConfigs;

    private final JsonProvider jsonProvider;

    public JsonBinlogMetaDebeziumDeserializationSchema() {
        this(false);
    }

    public JsonBinlogMetaDebeziumDeserializationSchema(Boolean includeSchema) {
        this.includeSchema = includeSchema;

        this.jsonProvider = new JsonProviderImpl();
    }

    public JsonBinlogMetaDebeziumDeserializationSchema(
            Boolean includeSchema, Map<String, Object> customConverterConfigs) {
        this.includeSchema = includeSchema;
        this.customConverterConfigs = customConverterConfigs;
        this.jsonProvider = new JsonProviderImpl();
    }

    @Override
    public void deserialize(SourceRecord record, Collector<BinlogCdcValue> out) throws Exception {
        if (jsonConverter == null) {
            initializeJsonConverter();
        }
        byte[] bytes =
                jsonConverter.fromConnectData(record.topic(), record.valueSchema(), record.value());

        BinlogCdcValue cdcValue = this.translate2BinlogCdcValue(record, bytes);

        out.collect(cdcValue);
    }

    /**
     * Initialize {@link JsonConverter} with given configs.
     */
    private void initializeJsonConverter() {
        jsonConverter = new JsonConverter();
        final HashMap<String, Object> configs = new HashMap<>(2);
        configs.put(ConverterConfig.TYPE_CONFIG, ConverterType.VALUE.getName());
        configs.put(JsonConverterConfig.SCHEMAS_ENABLE_CONFIG, includeSchema);
        if (customConverterConfigs != null) {
            configs.putAll(customConverterConfigs);
        }
        jsonConverter.configure(configs);
    }

    @Override
    public TypeInformation<BinlogCdcValue> getProducedType() {
        return BasicTypeInfo.of(BinlogCdcValue.class);
    }

    private BinlogCdcValue translate2BinlogCdcValue(SourceRecord record, byte[] value) {
        String ix = new String(value);
        JSONObject jsonObject = jsonProvider.toObject(JSONObject.class, ix);
        if (jsonObject == null) {
            throw new BadEventException(String.format("无效的数据:%s", ix));
        }

        JSONObject sourceNode = jsonObject.getJSONObject("source");
        if (sourceNode == null) {
            throw new BadEventException(String.format("无效的数据:%s", ix));
        }

        JSONObject dataNode = jsonObject.getJSONObject("after");
        if (dataNode == null) {
            dataNode = jsonObject.getJSONObject("before");
        }

        Optional<String> keyOptional =
                dataNode.keySet().stream().filter(ii -> ii.replace("_", "")
                        .equalsIgnoreCase("pkId")).findFirst();

        if (!keyOptional.isPresent()) {
            keyOptional =
                    dataNode.keySet().stream().filter(ii -> ii.equalsIgnoreCase("id")).findFirst();
        }

        if (!keyOptional.isPresent()) {
            throw new BadEventException(String.format("找不到对应的主键:%s", ix));
        }

        String id = dataNode.getString(keyOptional.get());

        Date lastSyncTime = DebeziumJsonNodeDtoUtils.getLastSyncTimeValue(jsonObject);
        BinlogCdcValue cdcValue = BinlogCdcValue.create(
                id, sourceNode.getString("db"),
                sourceNode.getString("table"), ix,
                lastSyncTime);
        Map<String, Object> sourceMeta = new HashMap<>();
        for (String key : record.sourcePartition().keySet()) {
            sourceMeta.put(key, record.sourcePartition().get(key));
        }
        for (String key : record.sourceOffset().keySet()) {
            sourceMeta.put(key, record.sourceOffset().get(key));
        }

        cdcValue.changeBinlog(sourceMeta);

        return cdcValue;
    }
}
