package com.bcxin.flink.cdc.kafka.source.task.cdcs.dynamic;

import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoClients;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.Indexes;
import com.mongodb.client.model.WriteModel;
import org.apache.flink.connector.base.DeliveryGuarantee;
import org.apache.flink.connector.mongodb.common.config.MongoConnectionOptions;
import org.apache.flink.connector.mongodb.sink.MongoSink;
import org.apache.flink.connector.mongodb.sink.config.MongoWriteOptions;
import org.apache.flink.connector.mongodb.sink.writer.serializer.MongoSerializationSchema;
import org.apache.flink.util.InstantiationUtil;
import org.bson.Document;
import org.bson.conversions.Bson;

import static org.apache.flink.util.Preconditions.checkNotNull;
import static org.apache.flink.util.Preconditions.checkState;

/**
 * 从MongoSinkBuilder拷贝过来
 * @param <IN>
 */
public class DynamicDbCollectionMongoSinkBuilder<IN> {
    private final MongoConnectionOptions.MongoConnectionOptionsBuilder connectionOptionsBuilder;
    private final MongoWriteOptions.MongoWriteOptionsBuilder writeOptionsBuilder;

    private MongoSerializationSchema<IN> serializationSchema;

    public DynamicDbCollectionMongoSinkBuilder() {
        this.connectionOptionsBuilder = MongoConnectionOptions.builder();
        this.writeOptionsBuilder = MongoWriteOptions.builder();
    }

    /**
     * Sets the connection string of MongoDB.
     *
     * @param uri connection string of MongoDB
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setUri(String uri) {
        connectionOptionsBuilder.setUri(uri);
        return this;
    }

    /**
     * Sets the database to sink of MongoDB.
     *
     * @param database the database to sink of MongoDB.
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setDatabase(String database) {
        connectionOptionsBuilder.setDatabase(database);
        return this;
    }

    /**
     * Sets the collection to sink of MongoDB.
     *
     * @param collection the collection to sink of MongoDB.
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setCollection(String collection) {
        connectionOptionsBuilder.setCollection(collection);
        return this;
    }

    /**
     * Sets the maximum number of actions to buffer for each batch request. You can pass -1 to
     * disable batching.
     *
     * @param batchSize the maximum number of actions to buffer for each batch request.
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setBatchSize(int batchSize) {
        writeOptionsBuilder.setBatchSize(batchSize);
        return this;
    }

    /**
     * Sets the batch flush interval, in milliseconds. You can pass -1 to disable it.
     *
     * @param batchIntervalMs the batch flush interval, in milliseconds.
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setBatchIntervalMs(long batchIntervalMs) {
        writeOptionsBuilder.setBatchIntervalMs(batchIntervalMs);
        return this;
    }

    /**
     * Sets the max retry times if writing records failed.
     *
     * @param maxRetries the max retry times.
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setMaxRetries(int maxRetries) {
        writeOptionsBuilder.setMaxRetries(maxRetries);
        return this;
    }

    /**
     * Sets the wanted {@link DeliveryGuarantee}. The default delivery guarantee is {@link
     * DeliveryGuarantee#AT_LEAST_ONCE}
     *
     * @param deliveryGuarantee which describes the record emission behaviour
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setDeliveryGuarantee(DeliveryGuarantee deliveryGuarantee) {
        writeOptionsBuilder.setDeliveryGuarantee(deliveryGuarantee);
        return this;
    }

    /**
     * Sets the serialization schema which is invoked on every record to convert it to MongoDB bulk
     * request.
     *
     * @param serializationSchema to process records into MongoDB bulk {@link WriteModel}.
     * @return this builder
     */
    public DynamicDbCollectionMongoSinkBuilder<IN> setSerializationSchema(
            MongoSerializationSchema<IN> serializationSchema) {
        checkNotNull(serializationSchema);
        checkState(
                InstantiationUtil.isSerializable(serializationSchema),
                "The mongo serialization schema must be serializable.");
        this.serializationSchema = serializationSchema;
        return this;
    }

    /**
     * Constructs the {@link MongoSink} with the properties configured this builder.
     *
     * @return {@link MongoSink}
     */
    public DynamicDbCollectionMongoSink<IN> build(String[] tables) {
        checkNotNull(serializationSchema, "The serialization schema must be supplied");
        MongoConnectionOptions connectionOptions = connectionOptionsBuilder.build();

        ensurePkIndex(connectionOptions, tables);
        return new DynamicDbCollectionMongoSink<>(
                connectionOptions,
                writeOptionsBuilder.build(),
                serializationSchema);
    }

    private void ensurePkIndex(MongoConnectionOptions connectionOptions,String[] tables) {
        /*
        MongoClient mongoClient = MongoClients.create(connectionOptions.getUri());

        for (String dbTable : tables) {
            String[] dbTableArray = dbTable.split("\\.");
            MongoDatabase database = mongoClient.getDatabase(dbTableArray[0]);
            MongoCollection<Document> documentMongoCollection = database.getCollection(dbTableArray[1]);


         */
            /*
            IndexOptions indexOptions = new IndexOptions().unique(true)
                    .background(false).name("id");
            Bson keys = new Document("")
            documentMongoCollection.createIndex()


        } */
    }
}
