package com.zbkj.admin.controller.merchant;

import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.context.AnalysisContext;
import com.alibaba.excel.event.AnalysisEventListener;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.model.admin.SystemAdmin;
import com.zbkj.common.model.bcx.BcxOrderDeviceNo;
import com.zbkj.common.model.order.Order;
import com.zbkj.common.model.order.OrderDetail;
import com.zbkj.common.model.product.Product;
import com.zbkj.common.model.user.User;
import com.zbkj.common.page.CommonPage;
import com.zbkj.common.request.PageParamRequest;
import com.zbkj.common.request.bcx.BcxOrderDeviceNoRequest;
import com.zbkj.common.request.bcx.BcxOrderNoImportRequest;
import com.zbkj.common.response.bcx.BcxOrderDeviceNoImportResponse;
import com.zbkj.common.response.bcx.BcxOrderDeviceNoResponse;
import com.zbkj.common.result.CommonResult;
import com.zbkj.common.utils.SecurityUtil;
import com.zbkj.service.service.OrderDetailService;
import com.zbkj.service.service.OrderService;
import com.zbkj.service.service.ProductService;
import com.zbkj.service.service.UserService;
import com.zbkj.service.service.bcx.BcxOrderDeviceNoService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 订单设备编号 控制层
 */
@Slf4j
@Api(tags = "订单设备编号控制器")
@RestController
@RequestMapping("/api/admin/merchant/order/device")
public class BcxOrderDeviceNoController {

    @Autowired
    private BcxOrderDeviceNoService bcxOrderDeviceNoService;
    @Resource
    private OrderService orderService;
    @Resource
    private OrderDetailService orderDetailService;
    @Resource
    private TransactionTemplate transactionTemplate;
    @Resource
    private UserService userService;
    @Resource
    private ProductService productService;

    @ApiOperation("查询单条订单设备编号")
    @RequestMapping(value = "/info/{id}", method = RequestMethod.GET)
    public CommonResult<BcxOrderDeviceNoResponse> queryById(@PathVariable Integer id) {
        return CommonResult.success(bcxOrderDeviceNoService.queryById(id));
    }

    @ApiOperation("查询分页订单设备编号")
    @RequestMapping(value = "/list", method = RequestMethod.GET)
    public CommonResult<CommonPage<BcxOrderDeviceNoResponse>> queryPagedList(@Validated BcxOrderDeviceNoRequest request,
                                                                             @Validated PageParamRequest pageParamRequest) {
        return CommonResult.success(CommonPage.restPage(bcxOrderDeviceNoService.queryPagedList(request, pageParamRequest)));
    }

    @ApiOperation("新增订单设备编号")
    @RequestMapping(value = "/create", method = RequestMethod.POST)
    public CommonResult<BcxOrderDeviceNo> create(@RequestBody @Validated BcxOrderDeviceNo bcxOrderDeviceNo) {
        if (bcxOrderDeviceNoService.save(bcxOrderDeviceNo)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }

    @ApiOperation("修改订单设备编号")
    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public CommonResult<BcxOrderDeviceNo> update(@RequestBody @Validated BcxOrderDeviceNo bcxOrderDeviceNo) {
        if (bcxOrderDeviceNoService.updateById(bcxOrderDeviceNo)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }

    @ApiOperation("删除订单设备编号")
    @RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
    public CommonResult<String> delete(@PathVariable Integer id) {
        if (bcxOrderDeviceNoService.removeById(id)) {
            return CommonResult.success();
        }
        return CommonResult.failed();
    }

    @ApiOperation(value = "导入订单设备编号-上传excel")
    @RequestMapping(value = "/upload", method = RequestMethod.POST)
    public CommonResult<String> file(@RequestParam(value = "file") MultipartFile file, @RequestParam String orderNo) {
        SystemAdmin systemAdmin = SecurityUtil.getLoginUserVo().getUser();
        List<BcxOrderDeviceNoImportResponse> resultList = new ArrayList<>();
        AnalysisEventListener<BcxOrderNoImportRequest> listener = new AnalysisEventListener<BcxOrderNoImportRequest>() {
            private final List<BcxOrderNoImportRequest> recordList = new ArrayList<>();

            @Override
            public void invoke(BcxOrderNoImportRequest record, AnalysisContext analysisContext) {
                if (StrUtil.isBlank(record.getOrderNo()) || StrUtil.isBlank(record.getDeviceNo())) {
                    throw new CrmebException("第" + analysisContext.readRowHolder().getRowIndex() + "行记录不能为空");
                }
                if (StrUtil.isBlank(orderNo) || !orderNo.equalsIgnoreCase(record.getOrderNo())) {
                    throw new CrmebException("订单号不匹配：" + orderNo + " ->" + record.getOrderNo());
                }
                recordList.add(record);
            }

            @Override
            public void doAfterAllAnalysed(AnalysisContext analysisContext) {
                log.debug("全部解析完成");
                if (recordList.stream().map(BcxOrderNoImportRequest::getOrderNo).distinct().count() > 1) {
                    throw new CrmebException("只能是同一个订单的数据，请检查订单号");
                }
                String orderNo = recordList.get(0).getOrderNo();
                Order order = orderService.getByOrderNo(orderNo);
                User user = userService.getById(order.getUid());
                List<OrderDetail> orderDetails = orderDetailService.getByOrderNo(orderNo);
                Integer total = orderDetails.stream().map(OrderDetail::getPayNum).reduce(0, Integer::sum);
                Integer totalRowNumber = recordList.size();
                if (!total.equals(totalRowNumber)) {
                    String eMsg = "excel条数：" + totalRowNumber + " 订单商品数：" + total;
                    log.error(eMsg);
                    throw new CrmebException("请检查数据，记录条数要跟订单数量一致，" + eMsg);
                }
                Integer productId = orderDetails.get(0).getProductId();
                Integer skuId = orderDetails.get(0).getAttrValueId();
                Product product = productService.getById(productId);
                List<BcxOrderDeviceNo> deviceNos = recordList.stream().map(r ->
                                new BcxOrderDeviceNo()
                                        .setDeviceNo(r.getDeviceNo())
                                        .setOrderNo(orderNo)
                                        .setSkuId(skuId)
                                        .setProductId(productId)
                                        .setBizSubType(product.getBizSubType())
                                        .setDeviceModel(product.getDeviceModel())
                                        .setAccount(user.getAccount()))
                        .collect(Collectors.toList());
                LambdaUpdateWrapper<BcxOrderDeviceNo> existsWrapper = Wrappers.lambdaUpdate();
                existsWrapper.eq(BcxOrderDeviceNo::getOrderNo, orderNo);
                transactionTemplate.execute(t -> {
                    bcxOrderDeviceNoService.remove(existsWrapper);
                    bcxOrderDeviceNoService.saveBatch(deviceNos);
                    return true;
                });
            }
        };

        readExcel(file, listener);

        return CommonResult.success("完成");
    }

    public void readExcel(MultipartFile file, AnalysisEventListener<BcxOrderNoImportRequest> listener) {
        try {
            log.debug("订单设备号excel解析开始");
            EasyExcel.read(file.getInputStream())
                    .autoCloseStream(false)
                    .autoTrim(true)
                    .sheet(0)
                    .head(BcxOrderNoImportRequest.class)
                    .headRowNumber(1)
                    .registerReadListener(listener)
                    .doReadSync();
        } catch (IOException e) {
            log.error("读取excel失败", e);
            throw new CrmebException("上传失败，请检查excel格式");
        }
    }
}
