package com.bcxin.obpm.service.impl;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.IoUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.auth.common.config.BMSConfig;
import com.bcxin.auth.common.constant.Constants;
import com.bcxin.auth.common.core.domain.AjaxResult;
import com.bcxin.auth.common.exception.V5BusinessException;
import com.bcxin.auth.common.utils.DateUtils;
import com.bcxin.auth.common.utils.StringUtils;
import com.bcxin.auth.system.domain.SysInterfaceLog;
import com.bcxin.auth.system.mapper.SysInterfaceLogMapper;
import com.bcxin.auth.system.util.ConfigUtil;
import com.bcxin.obpm.dto.*;
import com.bcxin.obpm.service.BackgroundService;
import com.bcxin.obpm.util.AuthConstants;
import com.bcxin.obpm.util.FaceUtil;
import com.bcxin.obpm.util.FtpUtils;
import com.google.common.collect.Maps;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 陕西
 *
 * @author linqinglin
 * @date 2021/08/19 0019 10:25
 */
@Service("backgroundService_610000")
public class SNBackgroundServiceImpl implements BackgroundService {
    private static Logger logger = LoggerFactory.getLogger(SNBackgroundServiceImpl.class);

    @Autowired
    private FaceUtil faceUtil;

    @Resource
    private ConfigUtil configUtil;

    @Autowired
    private SysInterfaceLogMapper sysInterfaceLogMapper;

    /**
     * 该类实现了人员审查的背景服务。它检查一个人是否有任何背景信息，如吸毒、在逃或有犯罪记录。
     * 检查是使用SN背景审查API完成的。
     */
    @Override
    public AjaxResult personnelReview(String idNumber, String realName) {
        if (StringUtils.isEmpty(idNumber)) {
            return AjaxResult.error("参数不正确");
        }
        List<BackgroundInfo> backgroundInfoList = new ArrayList<>();
        List<String> unRemind = configUtil.getUnRemindList();
        BackgroundResult backgroundResult = new BackgroundResult();
        //checkPersonInfo(backgroundInfoList, idNumber, unRemind);
        backgroundResult.setIdNumber(idNumber);
        backgroundResult.setCensorResult(backgroundInfoList);
        logger.info("身份证{}筛查结果:{}", idNumber, backgroundResult);
        // 请求
        return AjaxResult.success("请求成功", backgroundResult);
    }

    /**
     * 此方法检查人员的背景信息，如吸毒、在逃或有犯罪记录。
     * 详细信息是使用SN背景审查API获取的，并添加到列表中。
     *
     * @param list     存储背景信息的列表。
     * @param idNumber 人员的身份证号码。
     * @param unRemind 未提醒事项的列表。
     */
    private void checkPersonInfo(List<BackgroundInfo> list, String idNumber, List<String> unRemind) {
        BackgroundInfo backgroundInfo = new BackgroundInfo();
        String wfStr = StringUtils.EMPTY;
        try {
            Map<String, Object> wfMap = new HashMap<>();

            // 检查是否吸毒
            boolean isDrugUser = getSNCensorResult(idNumber, Constants.LIBRARYTYPE_XD);
            if (isDrugUser) {
                wfStr += "吸毒人员,";
            }

            // 检查是否在逃
            boolean isFugitive = getSNCensorResult(idNumber, Constants.LIBRARYTYPE_DT);
            if (isFugitive) {
                wfStr += "在逃人员,";
            }

            // 检查是否有违法犯罪记录
            boolean hasCriminalRecord = getSNCensorResult(idNumber, Constants.LIBRARYTYPE_WF);
            if (hasCriminalRecord) {
                wfStr += "有违法犯罪记录,";
            }

            if (!wfStr.isEmpty()) {
                wfStr = wfStr.substring(0, wfStr.length() - 1);
                wfMap.put("detail", wfStr);
                backgroundInfo.setCode(200);
                backgroundInfo.setData(wfMap);
                backgroundInfo.setResult(1);
                backgroundInfo.setCxxt("陕西四大库接口");
                list.add(backgroundInfo);
            }
        } catch (Exception e) {
            throw new V5BusinessException(e.getMessage());
        }
    }

    public AjaxResult checkPersonInfo(String idNumber) {
        List<BackgroundInfo> backgroundInfoList = new ArrayList<>();
        checkPersonInfo(backgroundInfoList, idNumber, null);

        String result =
                backgroundInfoList.stream().map(ii -> {
                    Map<String, Object> rt = ii.getData();
                    if (rt == null) {
                        return null;
                    }

                    Object dtl = rt.get("detail");
                    if (dtl == null) {
                        return null;
                    }

                    return String.valueOf(dtl);
                }).filter(ii -> ii != null).collect(Collectors.joining(","));

        Map<String, Object> objectMap = new HashMap<>();
        objectMap.put("problemDetails", result);

        return AjaxResult.success("请求成功", objectMap);
    }

    /**
     * 此方法查询SN背景审查API，根据给定的类型检查人员是否有任何背景信息。
     *
     * @param idNumber 人员的身份证号码。
     * @param type     要检查的背景信息类型（如吸毒、在逃、犯罪记录）。
     * @return 如果人员具有指定的背景信息，则返回true；否则返回false。
     */
    private boolean getSNCensorResult(String idNumber, String type) {
        String requestUrl = Constants.SN_BACKGROUND_CENSOR_PER_INFO_URL;

        // 创建接口日志对象
        SysInterfaceLog interfaceLog = new SysInterfaceLog();
        interfaceLog.setCreateTime(new Date());
        interfaceLog.setActive(true);
        interfaceLog.setUrl(Constants.SN_BACKGROUND_CENSOR_PER_INFO_URL);
        interfaceLog.setParameters("请求人口库接口：" + idNumber);

        // 创建请求参数对象
        Map<String, Object> params = Maps.newHashMap();
        switch (type) {
            case Constants.LIBRARYTYPE_DT:
                params.put("resId", "38");
                params.put("sql", "(`ZJHM`=? and `ryzt_jyqk`=0)");
                break;
            case Constants.LIBRARYTYPE_WF:
                params.put("resId", "62");
                params.put("sql", "(`ZJHM`=? and `ryjbxxbj` in ('0','1','5'))");
                break;
            case Constants.LIBRARYTYPE_XD:
                params.put("resId", "57");
                params.put("sql", "(`gmsfhm`=? and `yxx`=1)");
                break;
        }
        params.put("accessId", "ba3146d4");
        params.put("accessKey", "99565732e1654334859d52ae98b52c77");
        params.put("reqId", "40");
        params.put("queryValues", String.format("[{\"type\":\"string\",\"value\":\"%s\"}]", idNumber));

        logger.error("CensorRequestUrl:{}", requestUrl);
        logger.error("censorParams:{}", JSON.toJSONString(params));

        int errorCount = 0;
        String responseContent = null;
        SNCensorResult snCensorResult = null;
        List<SNPerInfoDto> data = null;
        for (int i = 0; i < Constants.ZAFLAG_COUNT; i++) {
            responseContent = HttpUtil.post(requestUrl, params);
            logger.error("陕西背景筛查新库查询{}返回:{}", idNumber, responseContent);
            snCensorResult = JSON.parseObject(responseContent, SNCensorResult.class);
            if (snCensorResult != null && snCensorResult.getCode() == 0) {
                data = snCensorResult.getData();
                if (data.size() > 0) {
                    return true;
                } else {
                    return false;
                }
            } else {
                errorCount++;
            }

            // 如果错误计数达到3次，则记录接口日志
            if (errorCount == 3) {
                break;
            }
        }
        // 记录接口日志
        if (errorCount == 3) {
            interfaceLog.setResult("背景筛查新库查询异常!!");
            interfaceLog.setUpdateTime(new Date());
            interfaceLog.setSuccessful(false);
            // 异步插入接口日志
            new Thread(() -> {
                sysInterfaceLogMapper.insert(interfaceLog);
            }).start();
            throw new V5BusinessException("背景筛查新库查询异常！！");
        }

        return false;
    }

    @Override
    public AjaxResult getPersonnelInformation(String idNumber) {
        if (StringUtils.isEmpty(idNumber)) {
            return AjaxResult.error("参数不正确");
        }
        AuthenticationResult authenticationResult = new AuthenticationResult();
        authenticationResult.setIdNumber(idNumber);
        try {
            PerPhotoDto perPhotoDto = getPopulationInfo(idNumber);
            authenticationResult.setPhoto(perPhotoDto.getPhoto());
            authenticationResult.setRealName(perPhotoDto.getName());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            throw new V5BusinessException("北京人员基本信息接口调用异常！！");
        }
        logger.info("{}人员信息服务返回:{}", idNumber, authenticationResult);
        return AjaxResult.success("请求成功", authenticationResult);
    }

    /**
     * 获取人口库信息
     *
     * @param idnum
     * @return
     */
    private PerPhotoDto getPopulationInfo(String idnum) {
        PerPhotoDto person = new PerPhotoDto();
        SNPerInfoDto snPerInfoDto = getSnPerInfo(idnum);
        if (snPerInfoDto != null) {
            person.setPhoto(snPerInfoDto.getZp());
            person.setName(snPerInfoDto.getXm());
            person.setIdNum(snPerInfoDto.getGmsfhm());
            return person;
        }
        return person;
    }

    /**
     * 根据身份证号码获取人口信息
     *
     * @param idNumber 身份证号码(或许是集合, 示例:`身份证1,身份证2`)
     * @return {@link PerPhotoDto} 人员列表
     */
    private SNPerInfoDto getSnPerInfo(String idNumber) {
        // 创建请求参数对象
        Map<String, Object> snPersonParams = new HashMap<>();
        // 创建接口日志对象
        SysInterfaceLog interfaceLog = new SysInterfaceLog();
        // 设置接口日志属性
        interfaceLog.setCreateTime(new Date());
        interfaceLog.setActive(true);
        interfaceLog.setUrl(Constants.SN_BACKGROUND_CENSOR_PER_INFO_URL);
        interfaceLog.setParameters("请求人口库接口：" + idNumber);
        // 设置请求参数
        snPersonParams.put("resId", "49");
        snPersonParams.put("sql", " (`GMSFHM`=?)");
        snPersonParams.put("accessId", "ba3146d4");
        snPersonParams.put("accessKey", "99565732e1654334859d52ae98b52c77");
        snPersonParams.put("reqId", "40");
        snPersonParams.put("queryValues", String.format("[{\"type\":\"string\",\"value\":\"%s\"}]", idNumber));
        // 打印日志
        logger.error("陕西实名认证url:{}", Constants.SN_BACKGROUND_CENSOR_PER_INFO_URL);
        logger.error("陕西实名认证params:{}", snPersonParams);
        // 初始化错误计数
        int errorCount = 0;
        // 发起请求，最多尝试 Constants.ZAFLAG_COUNT 次
        for (int i = 0; i < Constants.ZAFLAG_COUNT; i++) {
            // 发起 POST 请求
            String responseContent = HttpUtil.post(Constants.SN_BACKGROUND_CENSOR_PER_INFO_URL, snPersonParams);
            // 解析返回结果
            SNCensorResult snCensorResult = JSON.parseObject(responseContent, SNCensorResult.class);
            // 打印日志
            logger.error("陕西实名认证snCensorResult:{}", snCensorResult);
            // 判断返回结果是否成功
            if (snCensorResult != null && snCensorResult.getCode() == 0) {
                // 获取人员列表
                List<SNPerInfoDto> data = snCensorResult.getData();
                if (data.size() > 0) {
                    // 获取第一个人员信息
                    SNPerInfoDto person = data.get(0);
                    // 将返回的照片转为base64码
                    String xp = person.getZp();
                    String path = xp.substring(0, xp.lastIndexOf("/"));
                    String fileName = xp.substring(xp.lastIndexOf("/") + 1);
                    // 下载照片并转为字节数组
                    InputStream photoInputStream = FtpUtils.downloadFile("92.1.31.25", "xyyh", "xyyh@123", 12100, path, fileName);
                    if (photoInputStream != null) {
                        byte[] bytes = IoUtil.readBytes(photoInputStream);
                        String base64Code = Base64.getEncoder().encodeToString(bytes);
                        // 将base64设置回去
                        person.setZp(base64Code);
                    }
                    return person;
                } else {
                    errorCount++;
                    logger.error("人口库获取异常,没有返回值,身份证号：" + idNumber);
                }
            } else {
                errorCount++;
            }
            // 如果错误计数达到3次，则记录接口日志
            if (errorCount == 3) {
                interfaceLog.setResult("获取人口库基本信息异常!!");
                interfaceLog.setUpdateTime(new Date());
                interfaceLog.setSuccessful(false);
                new Thread(() -> {
                    sysInterfaceLogMapper.insert(interfaceLog);
                }).start();
            }
        }
        return null;
    }

    @Override
    public List<AuthLog> validateResult(List<AuthLog> list) {
        for (AuthLog authLog : list) {
            try {
                this.matchResult(authLog);
                authLog.setAuthDate(DateUtils.getNowDate());
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
            }
        }
        return list.stream().filter(((authLog) -> StrUtil.isNotEmpty(authLog.getAuthStatus()))).collect(Collectors.toList());
    }

    @Override
    public void getPersonnelAddress(String idNumber) {
    }

    /**
     * 实名认证比对
     *
     * @param authLog
     * @throws IOException
     */
    public void matchResult(AuthLog authLog) throws IOException {
        try {
            /***2.2 获取人口库照片，base64，如果是互联网则获取本地图片，以身份证命名******/
            PerPhotoDto person = getPopulationInfo(authLog.getIdNumber());
            String xp = "";
            /**
             * 2.2.1获取姓名
             */
            String name = person.getName();
            /**
             * 2.2.2 判断姓名是否为空，如果为空，则人口库没有信息
             */
            if (StrUtil.isEmpty(name)) {
                // 姓名和身份证不匹配
                authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_CARD_IDEMSG);
            } else if (name.equals(authLog.getSecurityName())) {
                /**
                 * 2.2.3 匹配姓名是否正常，如果匹配则获取照片信息，不匹配则返回
                 */
                // 获取人口库照片
                xp = person.getPhoto();
                /***2.2.4 没有照片代码身份证和姓名不匹配**/
                if (StrUtil.isEmpty(xp)) {
                    authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                    authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_PHOTO_IDEMSG);
                }
            } else {
                /***
                 * 2.2.4 姓名和身份证不匹配
                 */
                authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_NAME_IDEMSG);
            }
            /***
             * 2.3 如果有照片进行对比
             */
            if (StrUtil.isNotEmpty(xp)) {
                /***2.4 人口库和认证头像比对 **/
                if (xp.trim().startsWith("http")) {
                    logger.error("在线地址转base64");
                    String suffix = ".jpg";
                    if (!xp.matches(".*access_token.*")) {
                        suffix = xp.substring(xp.lastIndexOf("."));
                    }
                    String filePath = BMSConfig.getTempPath() + UUID.randomUUID().toString() + suffix;
                    faceUtil.downloadFile(xp, filePath);
                    xp = faceUtil.ImageToBase64(filePath);
                    FileUtil.del(filePath);
                }
                if (faceUtil.matchResult(xp, faceUtil.ImageToBase64(authLog.getImgPath()))) {
                    authLog.setAuthStatus(AuthConstants.AUTHRESULT_YES);
                    authLog.setAuthResult(AuthConstants.AUTHRESULT_YES_MESSAGE);
                } else {
                    authLog.setAuthStatus(AuthConstants.AUTHRESULT_NO);
                    authLog.setAuthResult(AuthConstants.AUTHRESULT_NO_IMGMSG);
                }
            }
            authLog.setAuthDate(DateUtils.getNowDate());
        } catch (Exception e) {
            authLog.setAuthStatus(AuthConstants.AUTHRESULT_ZHONG);
            authLog.setAuthResult("比对接口异常！！");
            e.printStackTrace();
        }
    }
}