package com.bcxin.statistics.common.utils;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.support.ExcelTypeEnum;
import com.alibaba.excel.write.handler.WriteHandler;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.BorderStyle;
import org.apache.poi.ss.usermodel.HorizontalAlignment;

import javax.servlet.http.HttpServletResponse;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;

/**
 * description：导出工具类
 * author：linchunpeng
 * date：2025/4/28
 */
@Slf4j
public class ExportUtil<T> {


    /**
     * description：导出excel
     * author：linchunpeng
     * date：2023/8/28
     */
    public void export(String fileName, List<T> resultList, HttpServletResponse response, WriteHandler writeHandler) {
        try {
            //设置导出excel的response
            setExcelResponse(fileName, response);
            if (CollectionUtil.isNotEmpty(resultList)) {
                //执行导出
                EasyExcel.write(response.getOutputStream(), resultList.get(0).getClass())
                        .autoCloseStream(false)
                        .registerWriteHandler(writeHandler)
                        .excelType(ExcelTypeEnum.XLS)
                        .sheet(fileName)
                        .registerWriteHandler(getWriteCellStyle())
                        .doWrite(resultList);
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.error("[导出excel-{}]异常，{}", fileName, e.getMessage(), e);
        }
    }

    /**
     * description：设置导出excel的response
     * author：linchunpeng
     * date：2023/8/29
     */
    private void setExcelResponse(String name, HttpServletResponse response) throws UnsupportedEncodingException {
        String fileName = String.format("%s-%s", name, DateUtils.dateTimeNow());
        String disposition = String.format("attachment;filename=%s.xls", URLEncoder.encode(fileName, "UTF-8"));
        response.setHeader("Access-Control-Expose-Headers", "Content-Disposition");
        response.setHeader("Content-Disposition", disposition);
        response.setContentType("application/vnd.ms-excel;charset=UTF-8");// 设置文本内省
        response.setCharacterEncoding("UTF-8");// 设置字符编码
        //设置跨域支持 这里如果不设置，swagger和postman可以正常访问，但是前端vue页面无法访问
        response.setHeader("Access-Control-Allow-Origin","*");
        response.setHeader("Access-Control-Allow-Method","POST,GET,OPTIONS");
        response.setHeader("Access-Control-Allow-Headers","*");
    }

    /**
     * 获取内容样式
     */
    private HorizontalCellStyleStrategy getWriteCellStyle() {
        //内容样式
        WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
        contentWriteCellStyle.setBorderLeft(BorderStyle.THIN);//细实线
        contentWriteCellStyle.setBorderTop(BorderStyle.THIN);
        contentWriteCellStyle.setBorderRight(BorderStyle.THIN);
        contentWriteCellStyle.setBorderBottom(BorderStyle.THIN);
        //设置头部样式
        WriteCellStyle headWriteCellStyle = new WriteCellStyle();
        //设置头部标题居中
        headWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
        //这个策略是 头是头的样式 内容是内容的样式 其他的策略可以自己实现
        return new HorizontalCellStyleStrategy(headWriteCellStyle, contentWriteCellStyle);
    }
}
