package com.wlos.app.utils;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONArray;
import com.alibaba.fastjson2.JSONObject;
import com.alibaba.fastjson2.JSONWriter;
import com.google.common.primitives.Ints;
import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import java.net.URLEncoder;
import com.wlos.app.exception.BusinessException;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import javax.crypto.*;
import javax.crypto.spec.DESKeySpec;
import javax.crypto.spec.DESedeKeySpec;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import java.io.*;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.Key;
import java.security.KeyFactory;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * 流程工具类
 * @author
 * @date 2024-08-30 17:02:53
 */
public class ProcessUtil {

    protected static final Logger log = LoggerFactory.getLogger(ProcessUtil.class);
    private static final Configuration configuration = Configuration.builder().options(Option.DEFAULT_PATH_LEAF_TO_NULL).build();

    /**
     *  新增参数 路径存在为修改，路径不存在为新增
     * @param paramsJson
     * @param path
     * @param value
     */
    public static String jsonParamsAdd(String paramsJson, String path, String key, Object value) {
        return JsonPath.parse(paramsJson).put(path, key, value).jsonString();
    }

    /**
    *  新增参数 路径存在为修改，路径不存在为新增
    * @param paramsJsonContext
    * @param path
    * @param value
    */
    public static String jsonParamsAdd(DocumentContext paramsJsonContext, String path, String key, Object value) {
        return paramsJsonContext.put(path, key, value).jsonString();
    }

    //转换request中的请求参数
    public static Map getParameterMap(Map properties) {
        // 返回值Map
        Map returnMap = new HashMap();
        Iterator entries = properties.entrySet().iterator();
        Map.Entry entry;
        String name = "";
        String value = "";
        while (entries.hasNext()) {
            entry = (Map.Entry) entries.next();
            name = (String) entry.getKey();
            Object valueObj = entry.getValue();
            if (null == valueObj) {
                value = "";
            } else if (valueObj instanceof String[]) {
                String[] values = (String[]) valueObj;
                for (int i = 0; i < values.length; i++) {
                    if (values[i].equals("null")) {
                        value = ",";
                    } else {
                        value = values[i] + ",";
                    }
                }
                value = value.substring(0, value.length() - 1);
            } else {
                value = valueObj.toString();
                if (value.equals("null")) {
                    value = "";
                }
            }
            returnMap.put(name, value);
        }
        return returnMap;
    }

    /**
     * 校验参数是否允许为空
     *
     * @param nullable    --是否必填：1-必填 0-非必填
     * @param paramsDesc  --参数描述
     * @param paramsValue --参数值
     * @param taskName    --流程节点名称
     * @return void
     * @author
     * @date
     */
    public static void checkParamNullable(String nullable, String paramsDesc, Object paramsValue, String taskName) {

        if ("1".equals(nullable)) {
            //必填
            if (null == paramsValue || "".equals(paramsValue)) {
                if (StringUtils.isNotBlank(taskName)) {
                    throw new BusinessException("操作异常：参数" + paramsDesc + "不能为空" + "[" + taskName + "]");
                } else {
                    throw new BusinessException("操作异常：参数" + paramsDesc + "不能为空");
                }
            }
        }
    }

    /**
     * 校验参数类型是否与参数值匹配
     *
     * @param dataType    --参数的数据类型：见参数展示类型文件
     * @param paramsDesc  --参数描述
     * @param paramsValue --参数值
     * @param taskName    --流程节点名称
     * @return void
     * @author
     * @date
     */
    public static void checkParamDataType(String dataType, String paramsDesc, Object paramsValue, String taskName) throws Exception {

        //校验参数类型
        if (!StringUtils.isEmpty(dataType) && null != paramsValue && !"".equals(paramsValue)) {

            switch (dataType) {
                case "Mobile": {
                    //校验手机号类型
                    //boolean flag = BeeUtils.checkMobileNumber(String.valueOf(paramsValue));
                    boolean flag = true;
                    if (!flag) {
                        String errorMsg = "";
                        if (!StringUtils.isEmpty(taskName)) {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确" + "[" + taskName + "]";
                        } else {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确";
                        }
                        throw new Exception(errorMsg);
                    }
                    break;
                }
                case "Email": {
                    //校验邮箱类型
                    //boolean flag = BeeUtils.checkEmail(String.valueOf(paramsValue));
                    boolean flag = true;
                    if (!flag) {
                        String errorMsg = "";
                        if (!StringUtils.isEmpty(taskName)) {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确" + "[" + taskName + "]";
                        } else {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确";
                        }
                        throw new Exception(errorMsg);
                    }
                    break;
                }

                case "Integer": {
                    //校验邮箱类型
                    Pattern pattern = Pattern
                            .compile("^(([1-9]{0}\\d{0,12})(\\.(\\d){1,6})?)|(0\\.0[1-9]{1,5})|(0\\.[1-9]{1,5}(\\d{1,5})?)$"); // 判断小数点后2位的数字的正则表达式
                    Matcher match = pattern.matcher(String.valueOf(paramsValue));
                    boolean flag = match.matches();
                    if (!flag) {
                        String errorMsg = "";
                        if (!StringUtils.isEmpty(taskName)) {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确" + "[" + taskName + "]";
                        } else {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确";
                        }
                        throw new Exception(errorMsg);
                    }
                    break;
                }

                case "Amount": {
                    //校验邮箱类型
                    Pattern pattern = Pattern
                            .compile("^(([1-9]{0}\\d{0,12})(\\.(\\d){1,6})?)|(0\\.0[1-9]{1,5})|(0\\.[1-9]{1,5}(\\d{1,5})?)$"); // 判断小数点后2位的数字的正则表达式
                    Matcher match = pattern.matcher(String.valueOf(paramsValue));
                    boolean flag = match.matches();
                    if (!flag) {
                        String errorMsg = "";
                        if (!StringUtils.isEmpty(taskName)) {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确" + "[" + taskName + "]";
                        } else {
                            errorMsg = "操作异常：参数" + paramsDesc + "格式不正确";
                        }
                        throw new Exception(errorMsg);
                    }
                    break;
                }
                default: {
                    break;
                }
            }

        }
    }

    /**
     * 校验流程入参
     *
     * @param params
     * @param pageJsonRule
     * @throws BusinessException
     */
    public static void checkProcessParams(String params, String pageJsonRule) throws BusinessException {
        try {
            //获取请求参数列表
            List<Map> paramList = JsonPath.parse(pageJsonRule).read("$.request.*");
            //判断是否有请求参数：如果有请求参数，则判断是否为空，判断参数类型是否匹配
            if (null != paramList && paramList.size() > 0) {

                Configuration conf = Configuration.defaultConfiguration().addOptions(Option.DEFAULT_PATH_LEAF_TO_NULL);
                DocumentContext dataJsonDocument = JsonPath.using(conf).parse(params);
                for (Map map : paramList) {

                    //获取参数名
                    String paramsName = String.valueOf(map.get("paramsName"));
                    //判断参数名是否为空，如果为空，则不处理，不为空，则处理
                    if (org.apache.commons.lang3.StringUtils.isNotBlank(paramsName)) {
                        //校验参数是否必填：1-必填 0-非必填
                        String requireFlag = String.valueOf(map.get("requireFlag"));
                        //参数描述
                        String paramsExplain = String.valueOf(map.get("paramsExplain"));
                        //获取参数值
                        Object value = dataJsonDocument.read("$.data." + paramsName);

                        ProcessUtil.checkParamNullable(requireFlag, paramsExplain, value, null);

                        //参数类型
                        String paramsType = String.valueOf(map.get("paramsType"));
                        ProcessUtil.checkParamDataType(paramsType, paramsExplain, value, null);
                    }
                }
            }

        } catch (BusinessException e) {
            log.error("校验异常", e);
            throw e;
        } catch (Exception e) {
            log.error("校验异常", e);
            throw new BusinessException("请求参数校验失败", e);
        }
    }

    /**
     * 转换类型
     *
     * @param value
     * @param type
     * @return java.lang.Object
     * @author
     * @date
     */
    public static Object transValueType(Object value, String type) {
        if (null == value || "".equals(value)) {
            return null;
        }

        if (Constants.DM_DATA_TYPE_AMOUNT.equals(type)
                || Constants.DM_DATA_TYPE_DECIMAL.equals(type)
                || Constants.DM_DATA_TYPE_DOUBLE.equals(type)
        ) {
            if (null != value && !"".equals(value)) {

                if (value instanceof List || value.getClass().isArray()) {

                    if (value.getClass().isArray()) {
                        value = Arrays.asList(value);
                    }

                    if (value instanceof List) {
                        List temp = new ArrayList();
                        ((List) value).forEach(a -> {
                            if (null != a) {
                                temp.add(getBigDecimal(a));
                            }
                        });
                        value = temp;
                    }

                } else {

                    value = getBigDecimal(value);
                }
            }

        } else if (Constants.DM_DATA_TYPE_INTEGER.equals(type)) {
            if (null != value && !"".equals(value)) {
                //整形类型转换
                if (value instanceof List || value.getClass().isArray()) {
                    if (value.getClass().isArray()) {
                        value = Arrays.asList(value);
                    }

                    if (value instanceof List) {
                        List temp = new ArrayList();
                        ((List) value).forEach(a -> {
                            if (null != a) {
                                temp.add(Double.valueOf(String.valueOf(a)).intValue());
                            }
                        });
                        value = temp;
                    }
                } else {
                    if (!(value instanceof Integer)) {

                        value = Double.valueOf(String.valueOf(value)).intValue();
                    }
                }
            }
        } else if (Constants.DM_DATA_TYPE_BOOLEAN.equals(type)) {
            value = Boolean.valueOf(value.toString());
        }
        return value;
    }

    public static String getTokenByRequest(HttpServletRequest request) {
        String token = null;
        //先从header中获取token
        token = request.getHeader("TOKEN");
        //如果header中不存在，则从请求参数中获取
        if (StringUtils.isBlank(token)) {
            token = request.getParameter("TOKEN");
        }
        return token;
    }

    public static BigDecimal getBigDecimal(Object value) {
        BigDecimal ret = null;
        if (value != null && !"".equals(value)) {
            if (value instanceof BigDecimal transformed) {
                ret = transformed;
            } else if (value instanceof String transformed) {
                ret = new BigDecimal(transformed);
            } else if (value instanceof BigInteger transformed) {
                ret = new BigDecimal(transformed);
            } else if (value instanceof Number transformed) {
                ret = new BigDecimal(transformed.toString());
            } else {
                throw new ClassCastException("Not possible to coerce [" + value + "] from class " + value.getClass()
                        + " into a BigDecimal.");
            }
        }
        return ret;
    }

    /**
     * 判断字符串是不是json格式
     *
     * @param content
     * @return
     */
    public static boolean isJson(Object content) {
        try {
            if (null == content) {
                return false;
            }
            boolean isJsonObject = false;
            try {
                isJsonObject = true;
            } catch (Exception e) {

            }
            JSON.toJSON(content);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 通过url地址获取文件的byte格式
     *
     * @param path
     * @return
     */
    public static Map<String, Object> readNetFile(String path) {
        InputStream inputFile = null;
        try {
            URL url = new URL(path); // 创建URL
            URLConnection urlconn = url.openConnection(); // 试图连接并取得返回状态码
            urlconn.setReadTimeout(50000);
            urlconn.connect();
            HttpURLConnection httpconn = (HttpURLConnection) urlconn;
            int httpResult = httpconn.getResponseCode();
            if (httpResult != HttpURLConnection.HTTP_OK) // 不等于HTTP_OK说明连接不成功
                log.error("无法连接到");
            else {
                inputFile = urlconn.getInputStream();
                ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                byte[] buffer = new byte[1024];

                int len = 0;
                while ((len = inputFile.read(buffer)) != -1) {
                    outStream.write(buffer, 0, len);
                }
                inputFile.close();
                byte[] fileData = outStream.toByteArray();
                outStream.flush();
                outStream.close();

                Map<String, Object> map = new HashMap<String, Object>();
                map.put("contentType", httpconn.getContentType());
                map.put("byteFile", fileData);

                return map;
            }
        } catch (Exception e) {
            log.error("解析异常", e);
            return null;

        } finally {
            if (inputFile != null)
                try {
                    inputFile.close();
                } catch (IOException e) {
                    log.error("解析异常", e);
                    return null;

                }

        }
        return null;
    }

    /**
     * 判断返回值是否为文件输出
     *
     * @param value
     * @return
     */
    public static boolean checkReturnFile(Object value) {
        boolean isReturnFile = false;
        try {
            //判断存储的文件是否为文件下载节点输出的数据：文件下载节点输出的数据格式为json格式，且存在特殊标识符 DOWN_FILE_FLAG_KEY
            if (null != value && !"".equals(value) && value instanceof Map) {
            JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(value));
                if (jsonObject.containsKey(Constants.DOWN_FILE_FLAG_KEY)
                        && jsonObject.get(Constants.DOWN_FILE_FLAG_KEY).equals(Constants.DOWN_FILE_FLAG_VALUE_1)) {
                    isReturnFile = true;
                }
            }
        } catch (Exception e) {
            log.error("操作异常", e);
        }
        return isReturnFile;
    }

    public static String getFileUrl(Object filePath, String errorSuffix) throws BusinessException {
        try {
            String uploadFile = null;
            //判断上传文件地址的类型：单个字符串格式或者数组-[{"url":"https:...","title":"file name"}]
            if (filePath instanceof List) {
                //如果是数组格式，则获取第一个上传的文件地址
                List<Map> uploadFileList = (List<Map>) filePath;
                if (null != uploadFileList && uploadFileList.size() > 0) {
                    uploadFile = String.valueOf(uploadFileList.get(0).get("url"));
                }
            } else if (filePath instanceof Map) {
                uploadFile = String.valueOf(((Map) filePath).get("url"));
            } else if (filePath instanceof String) {
                //如果不是数组格式，则判断是否为数组格式字符串或者单纯的url字符串
                boolean isJson = false;
                try {
                    Object file = JSON.parse(String.valueOf(filePath));
                    if (file instanceof List) {
                        //如果是数组格式，则获取第一个上传的文件地址
                        List<Map> uploadFileList = (List<Map>) file;
                        if (null != uploadFileList && uploadFileList.size() > 0) {
                            uploadFile = String.valueOf(uploadFileList.get(0).get("url"));
                        }
                        isJson = true;
                    } else if (file instanceof Map) {
                        uploadFile = String.valueOf(((Map) file).get("url"));
                        isJson = true;
                    }
                } catch (Exception e) {

                }
                if (!isJson) {
                    uploadFile = String.valueOf(filePath);
                }
            } else {
                uploadFile = String.valueOf(filePath);
            }
            return uploadFile;
        } catch (Exception e) {
            log.error("处理失败", e);
            throw new BusinessException("文件解析失败" + errorSuffix, e);
        }
    }

    public static byte[] getFileByteByUrl(String fileUrl, String errorSuffix) throws BusinessException {
        try {
            fileUrl = urlEncodeChinese(fileUrl);
            URL url = new URL(fileUrl);
            DataInputStream inputStream = new DataInputStream(url.openStream());
            byte[] buffer = new byte[1024];
            int len;
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            while ((len = inputStream.read(buffer)) > 0) {
                bos.write(buffer, 0, len);
            }
            byte b[] = bos.toByteArray();
            return b;
        } catch (Exception e) {
            log.error("处理失败", e);
            throw new BusinessException("文件解析失败" + errorSuffix, e);
        }
    }
    public static String urlEncodeChinese(String url) {
        try {
            Matcher matcher = Pattern.compile("[\\u4e00-\\u9fa5]").matcher(url);
            String tmp = "";
            while (matcher.find()) {
                tmp = matcher.group();
                url = url.replaceAll(tmp, URLEncoder.encode(tmp, "UTF-8"));
            }
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        }
        return url.replace(" ", "%20");
    }

    public static Object handleAnalysisJsonByJson(Object analysisContent, JSONArray bindParams, String errorSuffix) throws BusinessException {
        try {
            Object getJson = null;
            try {
                if (analysisContent instanceof String) {
                    boolean isJson = false;
                    try {
                        getJson = JSON.parseObject((String) analysisContent);
                        isJson = true;
                    } catch (Exception e) {
                        log.debug("捕获异常");
                    }
                    //兼容获取到的数据为数组的
                    try {
                        getJson = JSON.parseArray((String) analysisContent);
                        isJson = true;
                    } catch (Exception e) {
                        log.debug("捕获异常");
                    }
                    if (!isJson) {
                        throw new BusinessException("操作异常：解析的数据非json格式" + errorSuffix);
                    }
                } else if (analysisContent instanceof Map) {
                    getJson = new JSONObject((Map) analysisContent);
                } else if (analysisContent instanceof List) {
                    getJson = new JSONArray((List<Object>) analysisContent);
                } else {
                    getJson = (JSONObject) analysisContent;
                }
            } catch (BusinessException e) {
                throw e;
            } catch (Exception e) {
                throw new BusinessException("操作异常：解析的数据非json格式" + errorSuffix);
            }

            return getJson;
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new BusinessException("解析json处理失败" + errorSuffix, e);
        }
    }

    /**
     * 根据字符长度解析文本，组装成json格式
     * @param bindParams
     * @param analysisContent
     * @return
     */
    public static JSONObject handleAnalysisJsonByIndex(JSONArray bindParams, String analysisContent) {
        if (StringUtils.isBlank(analysisContent) || CollectionUtils.isEmpty(bindParams) || bindParams.size() < 1) {
            return null;
        }
        JSONObject jsonObject = new JSONObject();
        for (int i = 0; i < bindParams.size(); i++) {
            JSONObject bindParam = bindParams.getJSONObject(i);
            String bindKey = bindParam.getString("bindKey");
            if (StringUtils.isBlank(bindKey)) {
                continue;
            }
            Object startIndex = bindParam.getString("startIndex");
            Object endIndex = bindParam.getString("endIndex");
            if (null == startIndex) {
                startIndex = 0;
            }
            Object bindValue = null;
            int start = Integer.parseInt((String) startIndex);
            if (null != endIndex) {
                int end = Integer.parseInt((String) endIndex);
                bindValue = analysisContent.substring(start, end);
            } else {
                bindValue = analysisContent.substring((int) startIndex);
            }
            log.info("bindKey===" + bindKey + "=====bindValue====" + bindValue);
            //本身有值，则存储当前绑定值
            jsonObject.put(bindKey, bindValue);
        }
        return jsonObject;
    }

    public static Object handleDownDataByDataType(Object columnValue, String data_type) {
        try {
            if (null == columnValue || "".equals(columnValue)) {
                return null;
            }
            if (Constants.DM_DATA_TYPE_FILE.equals(data_type)
                    || Constants.DM_DATA_TYPE_IMAGE.equals(data_type)) {
                //文件类型返回数据处理
                if (columnValue instanceof List) {
                    List<Map> uploadFileList = (List<Map>) JSON.toJSON(columnValue);
                    if (null != uploadFileList && uploadFileList.size() > 0) {
                        String urls = "";
                        for (int i = 0; i < uploadFileList.size(); i++) {

                            Map urlMap = uploadFileList.get(i);
                            urls += urlMap.get("url") + "";
                            if (i + 1 != uploadFileList.size()) {
                                urls = urls + ",";
                            }
                        }
                        columnValue = urls;
                    }
                }
            } else if (Constants.DM_DATA_TYPE_OPTION.equals(data_type)) {
                //选项集类型返回数据处理
                //如果是选项集类型的，则特殊处理
                try {
                    //如果是集合数据的话，特殊处理：lookup获取到的数据可能存在多条数据
                    if (columnValue instanceof List) {
                        columnValue = ((List<?>) columnValue)
                                .stream()
                                .map(mm -> (JSON.parseObject(JSONObject.toJSONString(mm))).getString("fields"))
                                .collect(Collectors.toList());
                    } else {
                        //非数组的话，则当一条数据来处理
                        if (columnValue instanceof Map) {
                            JSONObject vjson = JSON.parseObject(JSON.toJSONString(columnValue));
                            columnValue = vjson.get("fields");
                        }
                    }
                } catch (Exception e) {
                    //log.info("variableValue兼容获取选项集类型值", e);
                }
            } else if (Constants.DM_DATA_TYPE_ADDRESS.equals(data_type)) {
                //地址类型：地址存储的是数组格式：需要区分是数组格式存储的地址，还是地址文本格式
                if (columnValue instanceof List) {
                    columnValue = ((List<?>) columnValue)
                            .stream()
                            .map(mm -> (JSON.parseObject(JSONObject.toJSONString(mm))).getString("name"))
                            .collect(Collectors.toList());
                    List<String> addressList = ((List<String>) columnValue);
                    String address = "";
                    for (int i = 0; i < addressList.size(); i++) {
                        address += addressList.get(i);
                    }
                    columnValue = address;
                }
            } else if (Constants.DM_DATA_TYPE_DATE_TIME.equals(data_type)) {
                //时间格式
                Long timeLong = Long.parseLong(String.valueOf(columnValue));
                java.text.SimpleDateFormat sdf = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                java.util.Date date = sdf.parse(sdf.format(timeLong));
                columnValue = sdf.format(date);
            }
        } catch (Exception e) {
        }
        return columnValue;
    }

    /**
     * 根据规则生成随机字符串
     *
     * @param totalLength 长度
     * @param rule        ：生成规则(多选)：1-数字 2-小写字母 3-大写字母
     * @param isDistinct  ：是否去重
     * @return
     * @throws BusinessException
     */
    public static String randomByRule(Integer totalLength, int[] rule, boolean isDistinct) throws BusinessException {
        StringBuilder sb = new StringBuilder();
        Random rd = new Random();

        //判断必须包含的字符
        List<Integer> ruleList = Ints.asList(rule);
        //判断是否必须包含数字
        if (ruleList.contains(1)) {
            if (sb.toString().length() < totalLength) {
                sb.append((char) ('0' + uniform(0, 10)));
            }
        }
        //判断是否必须包含小写字母
        if (ruleList.contains(2)) {
            if (sb.toString().length() < totalLength) {
                sb.append((char) ('a' + uniform(0, 26)));
            }
        }
        //判断是否必须包含大写字母
        if (ruleList.contains(3)) {
            if (sb.toString().length() < totalLength) {
                sb.append((char) ('A' + uniform(0, 26)));
            }
        }

        int randomLength = sb.toString().length();

        while (randomLength < totalLength) {
            //产生0-2的3位随机数
            int index = rd.nextInt(rule.length);
            int type = rule[index];
            Object str = null;
            switch (type) {
                case 1:
                    //0-9的随机数
                    str = String.valueOf(rd.nextInt(10));
                    break;
                case 2:
                    //ASCII在97-122之间为小写，获取小写随机
                    str = (char) (rd.nextInt(25) + 97);
                    break;
                case 3:
                    //ASCII在65-90之间为大写,获取大写随机
                    str = (char) (rd.nextInt(25) + 65);
                    break;
                default:
                    break;
            }
            if (null != str && (!isDistinct || (isDistinct && !sb.toString().contains(String.valueOf(str))))) {
                sb.append(str);
                randomLength++;
            }

        }
        //将数组chArr随机排序
        char[] chArr = sb.toString().toCharArray();
        for(int i = 0; i < totalLength; i++){
            int r = i + uniform(totalLength - i);
            char temp = chArr[i];
            chArr[i] = chArr[r];
            chArr[r] = temp;
        }
        return new String(chArr);
    }

    public static int uniform(int N) {
        Random random = new Random();
        return random.nextInt(N);
    }
    public static int uniform(int a,int b){
        return a + uniform(b - a);
     }



    public static void addFile(Object downFileValue, List<Object> fileList) {
        //判断下载的内容
        //判断上传文件地址的类型：单个字符串格式或者数组-[{"url":"https:...","title":"file name"}]
        if (downFileValue instanceof List) {
            ((List<?>) downFileValue).forEach(item -> {
            addFile(item, fileList);
            });
        } else if (downFileValue instanceof Map) {
            fileList.add(downFileValue);
        } else if (downFileValue instanceof String) {
            //默认为字符串类型：判断是否为json字符串
            //校验是否为json格式
            //如果不是数组格式，则判断是否为数组格式字符串或者单纯的url字符串
            boolean isJson = false;
            try {
                Object file = JSON.parse(String.valueOf(downFileValue));
                if (file instanceof List) {
                    fileList.addAll((List<?>) file);
                    isJson = true;
                } else if (file instanceof Map) {
                    fileList.add((Map) file);
                    isJson = true;
                }
            } catch (Exception e) {

            }
            if (!isJson) {
                //如果不是数组格式，则认为是文件地址
                Map<String, String> fileMap = new HashMap<>();
                fileMap.put("title", "未命名");
                fileMap.put("url", String.valueOf(downFileValue));
                fileList.add(fileMap);
            }
        } else {
            //未知类型统一当文件地址处理
            Map<String, String> fileMap = new HashMap<>();
            fileMap.put("title", "未命名");
            fileMap.put("url", String.valueOf(downFileValue));
            fileList.add(fileMap);
        }
    }


    public static JSONObject findChild(JSONObject pNode, List<JSONObject> list, String id, String children, String parentId) {
        for (JSONObject node : list) {
            if (pNode.getString(id).equals(node.getString(parentId))) {
                pNode.putIfAbsent(children, new JSONArray(new ArrayList<>()));
                pNode.getJSONArray(children).add(findChild(node, list, id, children, parentId));
            }
        }
        return pNode;
    }


    /**
     * 创建本地临时文件
     *
     * @return 临时文件目录
     */
    public static String createLocalTemporaryFile(JSONObject params) {
        File file = new File("");
        String canonicalPath = null;
        try {
            canonicalPath = file.getCanonicalPath();
        } catch (IOException e) {
            e.printStackTrace();
        }
        String temporaryDir = UUID.randomUUID().toString();
        //临时文件目录
        String filePath = canonicalPath + "/" + temporaryDir + "/";
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            String key = entry.getKey();
            JSONObject val = (JSONObject) entry.getValue();
            File dir = new File(filePath + key);
            if (!dir.exists() && !dir.isDirectory()) {
                //创建文件夹
                dir.mkdirs();
            }
            for (Map.Entry<String, Object> fileEntry : val.entrySet()) {
                String k = fileEntry.getKey();
                String v = (String) fileEntry.getValue();
                File temporaryFile = new File(filePath + key, k);
                try {
                    temporaryFile.createNewFile();
                } catch (IOException e) {
                    e.printStackTrace();
                }
                //获取文件 bytes[]
                Base64.Decoder decoder = Base64.getDecoder();
                byte[] bytes = decoder.decode(v);

                ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
                try (OutputStream out = new FileOutputStream(temporaryFile)) {
                    byte[] buffer = new byte[1024];
                    int r;
                    while ((r = inputStream.read(buffer)) != -1) {
                        out.write(buffer, 0, r);
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
                log.info("目录：{} --文件名：{} ", key, k);
            }
        }
        return filePath;
    }

    public static void deleteLocalTemporaryFile(String path) {
        Path uri = Paths.get(path);
        try {
            Files.walkFileTree(uri,
                    new SimpleFileVisitor<Path>() {
                        // 先去遍历删除文件
                        @Override
                        public FileVisitResult visitFile(Path file,
                                                         BasicFileAttributes attrs) throws IOException {
                            Files.delete(file);
                            log.info("文件被删除 : {}", file);
                            return FileVisitResult.CONTINUE;
                        }

                        // 再去遍历删除目录
                        @Override
                        public FileVisitResult postVisitDirectory(Path dir,
                                                                  IOException exc) throws IOException {
                            Files.delete(dir);
                            log.info("文件目录被删除 : {}", dir);
                            return FileVisitResult.CONTINUE;
                        }
                    });
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 提供给用户使用的基本压缩类
     */
    public static void compressFile(String srcPath, String outPath) throws IOException {
        //读取源文件
        File srcFile = new File(srcPath);
        //判断输出路径是否正确
        File outFile = new File(outPath);
        //如果只是路劲加入对应的压缩名称
        if (outFile.isDirectory()) {
            //用"/"作文判断标准
            if (outPath.endsWith(File.separator)) {
                outPath += srcFile.getName().split("\\.")[0] + ".zip";
            } else {
                outPath += File.separator + srcFile.getName().split("\\.")[0] + ".zip";
            }
        }
        //读取文件流
        FileOutputStream fileOutputStream = new FileOutputStream(outPath);
        ZipOutputStream zipOutputStream = new ZipOutputStream(fileOutputStream);
        //压缩文件
        compressFile(srcFile, srcFile.getName(), zipOutputStream);
        //关闭流
        zipOutputStream.close();
        fileOutputStream.close();
    }

    /**
     * 迭代方式进行文件压缩
     */
    public static void compressFile(File file, String fileName, final ZipOutputStream outputStream) throws IOException {
        //如果是目录
        if (file.isDirectory()) {
            //创建文件夹
            outputStream.putNextEntry(new ZipEntry(fileName + "/"));
            //迭代判断，并且加入对应文件路径
            File[] files = file.listFiles();
            assert files != null;
            for (File f : files) {
                compressFile(f, fileName + "/" + f.getName(), outputStream);
            }
        } else {
            //创建文件
            outputStream.putNextEntry(new ZipEntry(fileName));
            //读取文件并写出
            try (FileInputStream fileInputStream = new FileInputStream(file);
                 BufferedInputStream bufferedInputStream = new BufferedInputStream(fileInputStream)) {
                byte[] bytes = new byte[1024];
                int n;
                while ((n = bufferedInputStream.read(bytes)) != -1) {
                    outputStream.write(bytes, 0, n);
                }
            }
        }
    }

    /**
     * 根据取值路径获取值
     * @param paramsJsonContext
     * @param returnDataKeyList
     * @return
     * @throws BusinessException
     */
    public static JSONObject getReturnData(JSONObject paramsJsonContext, String[] returnDataKeyList) throws BusinessException {
        log.info("========获取流程返回值=======");
        //返回数据处理
        JSONObject jsonObject = new JSONObject();
        //标识是是否是文件下载
        jsonObject.put(Constants.DOWN_FILE_FLAG_KEY, Constants.DOWN_FILE_FLAG_VALUE_0);
        AtomicBoolean checkFile = new AtomicBoolean(false);
        Arrays.stream(returnDataKeyList).forEach(item -> {
            if (StringUtils.isBlank(item)) {
                return;
            }
            String keyPath = "$.data." + item;
            Object keyValue = JsonPath.parse(paramsJsonContext.toJSONString(JSONWriter.Feature.WriteMapNullValue), configuration).read(keyPath);
            boolean isFileValue = ProcessUtil.checkReturnFile(keyValue);
            //判断获取的值是否为文件下载的数据
            if (isFileValue) {
                checkFile.set(true);
                jsonObject.put(Constants.DOWN_FILE_KEY_PATH, keyPath);
                jsonObject.put("system_flow_down_file_key_value_name", item);
            }
            jsonObject.put(item, keyValue);
        });
        if (checkFile.get()) {
            jsonObject.put(Constants.DOWN_FILE_FLAG_KEY, Constants.DOWN_FILE_FLAG_VALUE_1);
        }
        return jsonObject;
    }

    /**
    * 处理选项集，获取选项集的code值
    * @param value
    * @return
    */
    public static Object getOptionValue(Object value){
        try {
            if (null != value && !"".equals(value)) {
                if (value instanceof Map) {
                    JSONObject vjson = JSON.parseObject(JSON.toJSONString(value));
                    JSONArray optionUuids = vjson.getJSONArray("option_uuids");
                    value = StringUtils.join(optionUuids,",");
                } if (value instanceof List) {
                    //如果为list类型，则默认为新流程选项集
                    return value;
                }
            }
        } catch (Exception e) {
            log.info("variableValue兼容获取选项集类型值", e);
        }
        return value;
    }

    
    /**
     * 组装url请求参数
     * @param url
     * @return
     */
    public static String packageUrlParams(String url,  Map<String, Object> params) {
        if (CollectionUtils.isEmpty(params)) {
            return url;
        }
        if (!url.contains("?")) {
            url = url.concat("?");
        }         for (String key : params.keySet()) {
            if (url.indexOf("?") != url.length() - 1) {
                url = url.concat("&");
            }
            Object value = params.get(key);
            if(ObjectUtils.isEmpty(value)){
                value = "";
            }
            url = url.concat(key)
                    .concat("=")+value;
        }
        return url;
    }

    /**
     * 替换{code}占位
     * @author
     * @date 2024-05-07 18:31
     * @param url
     * @param params
     * @return
     */
    public static String replaceUrlPathParams(String url,  Map<String, Object> params) {
        if (CollectionUtils.isEmpty(params)) {
            return url;
        }

        // 创建正则表达式，匹配大括号内的内容
        Pattern pattern = Pattern.compile("\\{(.+?)\\}");
        Matcher matcher = pattern.matcher(url);

        // 替换占位符
        StringBuilder result = new StringBuilder();
        while (matcher.find()) {
            String placeholder = matcher.group(1);
            Object replacement = params.get(placeholder);
            if (Objects.nonNull(replacement)) {
                matcher.appendReplacement(result, String.valueOf(replacement));
            }
        }
        matcher.appendTail(result);
        return result.toString();
    }
                                                        

    
    /**
    * 获取参数的list格式
    * @param variable
    * @return
    */
    public static void getVariableList(List variableList, Object variable){
        //数组转集合
        if (ObjectUtils.isNotEmpty(variable)) {
            if (variable instanceof Object[] variableArray) {
                variableList.addAll(List.of(variableArray));
            } else if (variable instanceof List variables) {
                variableList.addAll(variables);
            } else {
                variableList.add(variable);
            }
        }
    }

    /**
     * object转 list
     *
     * @param object
     * @return
     */
    public static List object2List(Object object) {
        if (ObjectUtils.isEmpty(object)) {
            return new ArrayList();
        }
        List list = new ArrayList();
        if (object instanceof List  objList) {
            list = objList;
        } else {
            list.add(object);
        }
        return list;
    }

    /**
    * 获取对象长度
    *
    * @param object
    * @return
    */
    public static int getObjLength(Object object) {
        if (ObjectUtils.isEmpty(object)) {
            return 0;
        }
        if (object instanceof List<?> list) {
            return list.size();
        } else if (object instanceof Object[] arrays) {
            return arrays.length;
        } else {
            return String.valueOf(object).length();
        }
    }

    /**
    * 构建属性结构
    */
	public static JSONArray buildTree(List<JSONObject> arr, String id, String pid, String childrenKey) {
        //新建一个JSONArray来接收组装成树形结构的返回值
        JSONArray jsonArray = new JSONArray();
        //新建一个JSONObject对象
        JSONObject hash = new JSONObject();
        //将数组转换为object格式
        for (int i = 0; i < arr.size(); i++) {
            //获取当前的JSON对象
            JSONObject json = (JSONObject) arr.get(i);
            //把当前id作为键，当前JSON对象作为值 put回hash这个Object对象中
            //这里的put方法类似于map的put方法
            hash.put(json.getString(id), json);
        }
        //遍历结果集
        for (int j = 0; j < arr.size(); j++) {
            //单条记录
            JSONObject aVal = (JSONObject) arr.get(j);
            //在hash中取出key为单条记录中pid的值
            String pidStr = "";
            //如果父级id不等于null
            if (aVal.get(pid) != null) {
                pidStr = aVal.get(pid).toString();
            }
            //从hash这个对象中获取父级对象  parent
            JSONObject hashParent = (JSONObject) hash.get(pidStr);
            //如果记录的pid存在，则说明它有父节点，将她添加到孩子节点的集合中
            if (hashParent != null) {
                //检查是否有child属性
                if (hashParent.get(childrenKey) != null) {
                    //有子节点 则先将子节点取出
                    JSONArray children = (JSONArray) hashParent.get(childrenKey);
                    //然后把当前这个对象放进子节点之中
                    children.add(aVal);
                    //最后把子节点在放回父节点之中
                    hashParent.put(childrenKey, children);
                } else {
                    //无子节点 则新建一个子节点
                    JSONArray children = new JSONArray();
                    //然后再把当前对象放进去
                    children.add(aVal);
                    //最后在放回父节点之中
                    hashParent.put(childrenKey, children);
                }
            } else {
                jsonArray.add(aVal);
            }
        }
        return jsonArray;
    }

	/**
     * 递归排序
     * @author xiaoyi
     * @date 2024-05-23 16:43
     * @param jsonObject 需要排序的对象
     * @param ascending 是否按升序排序，true为升序，false为降序
     * @return
     */
    public static JSONObject sortJsonObject(JSONObject jsonObject, boolean ascending) {
        TreeMap<String, Object> sortedMap = new TreeMap<>((o1, o2) -> ascending ? o1.compareTo(o2) : o2.compareTo(o1));

        for (String key : jsonObject.keySet()) {
            Object value = jsonObject.get(key);
            if (value instanceof JSONObject) {
                // 如果值是JSONObject，递归排序
                value = sortJsonObject((JSONObject) value, ascending);
            }
            sortedMap.put(key, value);
        }

        return new JSONObject(sortedMap);
    }

}

