package com.wlos.app.utils;

import cn.hutool.core.codec.Base64Decoder;
import cn.hutool.core.codec.Base64Encoder;
import com.wlos.app.exception.BusinessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.*;
import javax.crypto.spec.DESKeySpec;
import javax.crypto.spec.DESedeKeySpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * 工具类
 * @author
 * @date 2024-08-30 17:02:53
 */
public class CipherUtil {

    protected static Logger log = LoggerFactory.getLogger(CipherUtil.class);
    private static char[] HEXCHARS_L = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'};
    private static char[] HEXCHARS_C = new char[]{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

    public static String encryptMD5(String str, boolean lower) {
        try {
            byte[] bytes = str.getBytes(StandardCharsets.UTF_8);
            MessageDigest md5 = MessageDigest.getInstance("MD5");
            md5.update(bytes);
            bytes = md5.digest();
            int j = bytes.length;
            char[] chars = new char[j * 2];
            int k = 0;

            for (int i = 0; i < bytes.length; ++i) {
                byte b = bytes[i];
                if (lower) {
                    chars[k++] = HEXCHARS_L[b >>> 4 & 15];
                    chars[k++] = HEXCHARS_L[b & 15];
                } else {
                    chars[k++] = HEXCHARS_C[b >>> 4 & 15];
                    chars[k++] = HEXCHARS_C[b & 15];
                }
            }

            return new String(chars);
        } catch (NoSuchAlgorithmException var9) {
            log.error("CipherUtil error message:{}", var9);
            return null;
        }
    }

    public static String encryptSHA(String str, boolean lower) {
        try {
            byte[] bytes = str.getBytes();
            MessageDigest sha = MessageDigest.getInstance("SHA");
            sha.update(bytes);
            bytes = sha.digest();
            int j = bytes.length;
            char[] chars = new char[j * 2];
            int k = 0;

            for (int i = 0; i < bytes.length; ++i) {
                byte b = bytes[i];
                if (lower) {
                    chars[k++] = HEXCHARS_L[b >>> 4 & 15];
                    chars[k++] = HEXCHARS_L[b & 15];
                } else {
                    chars[k++] = HEXCHARS_C[b >>> 4 & 15];
                    chars[k++] = HEXCHARS_C[b & 15];
                }
            }

            return new String(chars);
        } catch (NoSuchAlgorithmException var9) {
            log.error("CipherUtil error message:{}", var9);
            return null;
        }
    }

    public static String encryptSHA1(String str, boolean lower) {
        try {
            byte[] bytes = str.getBytes();
            MessageDigest sha = MessageDigest.getInstance("SHA-1");
            sha.update(bytes);
            bytes = sha.digest();
            int j = bytes.length;
            char[] chars = new char[j * 2];
            int k = 0;

            for (int i = 0; i < bytes.length; ++i) {
                byte b = bytes[i];
                if (lower) {
                    chars[k++] = HEXCHARS_L[b >>> 4 & 15];
                    chars[k++] = HEXCHARS_L[b & 15];
                } else {
                    chars[k++] = HEXCHARS_C[b >>> 4 & 15];
                    chars[k++] = HEXCHARS_C[b & 15];
                }
            }

            return new String(chars);
        } catch (NoSuchAlgorithmException var9) {
            log.error("CipherUtil error message:{}", var9);

            return null;
        }
    }

    public static String encryptHMACSHA256(byte[] data, char[] key) {
        PBEKeySpec pbeKeySpec = new PBEKeySpec(key);

        try {
            SecretKeyFactory keyFactory = SecretKeyFactory.getInstance("PBEWithSHA1AndRC2_40");
            SecretKey deskey = keyFactory.generateSecret(pbeKeySpec);
            Mac mac = Mac.getInstance("HmacSHA256");
            mac.init(deskey);
            byte[] digest = mac.doFinal(data);
            return new String(Base64Encoder.encode(digest).getBytes());
        } catch (NoSuchAlgorithmException var7) {
            log.error("CipherUtil error message:{}", var7);
            return null;
        } catch (InvalidKeySpecException var8) {
            log.error("CipherUtil error message:{}", var8);

            return null;
        } catch (InvalidKeyException var9) {
            log.error("CipherUtil error message:{}", var9);
            return null;
        }
    }

    /**
     * sha256_HMAC加密
     *
     * @param message 消息
     * @param secret  秘钥
     * @return 加密后字符串
     */
    public static String encryptHmacSHA(String message, String secret, String encryptType) {
        String hash = "";
        try {
            Mac mac = Mac.getInstance(encryptType);
            SecretKeySpec secret_key = new SecretKeySpec(secret.getBytes(), encryptType);
            mac.init(secret_key);
            byte[] bytes = mac.doFinal(message.getBytes(StandardCharsets.UTF_8));
            hash = byteArrayToHexString(bytes);
        } catch (Exception e) {
            log.error("Error HmacSHA1 ===========" + e.getMessage(), e);
        }
        return hash;
    }

    /**
     * 将加密后的字节数组转换成字符串
     *
     * @param b 字节数组
     * @return 字符串
     */
    private static String byteArrayToHexString(byte[] b) {
        StringBuilder hs = new StringBuilder();
        String stmp;
        for (int n = 0; b != null && n < b.length; n++) {
            stmp = Integer.toHexString(b[n] & 0XFF);
            if (stmp.length() == 1)
                hs.append('0');
            hs.append(stmp);
        }
        return hs.toString().toLowerCase();
    }

    /**
     * md5加密
     *
     * @param encryptType
     * @param encryptContent
     * @param errorSuffix
     * @return
     * @throws BusinessException
     */
    public static String handleMd5Encode(String encryptType, Object encryptContent, String errorSuffix) throws BusinessException {
        try {
            //encryptType 加密格式：1-16位 2-16位小写  3-32位大写  4- 32位小写
            //md5 32位小写
            String md5Value = CipherUtil.encryptMD5(String.valueOf(encryptContent), true);
            //判断数据解析方式
            if ("1".equals(encryptType)) {
                md5Value = md5Value.substring(8, 24).toUpperCase();
            } else if ("2".equals(encryptType)) {
                md5Value = md5Value.substring(8, 24);
            } else if ("3".equals(encryptType)) {
                md5Value = md5Value.toUpperCase();
            }
            return md5Value;
        } catch (Exception e) {
            log.error("加密失败", e);
            throw new BusinessException("加密失败" + errorSuffix, e);
        }
    }

    public static String encryptDES(String message, String secret, String encryptType) throws BusinessException {
        try {
            SecretKey secretKey = getSecretKey(secret, encryptType);
            Cipher cipher = Cipher.getInstance(encryptType + "/ECB/PKCS5Padding");
            // 根据密钥，对Cipher对象进行初始化，ENCRYPT_MODE表示加密模式
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            byte[] bytes = cipher.doFinal(message.getBytes(StandardCharsets.UTF_8));
            return Base64Encoder.encode(bytes);
        } catch (Exception e) {
            log.error("加密失败", e);
            throw new BusinessException("加密失败");
        }
    }

    protected static SecretKey getSecretKey(String key, String encryptType) {
        try {
            KeySpec keySpec = null;
            if (Constants.KEY_DES.equals(encryptType)) {
                keySpec = new DESKeySpec(key.getBytes(StandardCharsets.UTF_8));
            } else if (Constants.KEY_AES.equals(encryptType)) {
                return new SecretKeySpec(key.getBytes(StandardCharsets.UTF_8), encryptType);
            } else if (Constants.KEY_3DES.equals(encryptType)) {
                keySpec = new DESedeKeySpec(key.getBytes(StandardCharsets.UTF_8));
            }

            SecretKeyFactory factory = SecretKeyFactory.getInstance(encryptType);
            SecretKey secretKey = factory.generateSecret(keySpec);
            return secretKey;
        } catch (Exception e) {
            log.error("失败", e);
            throw new BusinessException("ErrorCodeEnum.ERROR_10034");
        }
    }

    public static String decryptDES(String message, String secret, String encryptType) throws BusinessException {
        try {
            SecretKey secretKey = getSecretKey(secret, encryptType);
            Cipher cipher = Cipher.getInstance(encryptType + "/ECB/PKCS5Padding");
            // 根据密钥，对Cipher对象进行初始化，ENCRYPT_MODE表示加密模式
            cipher.init(Cipher.DECRYPT_MODE, secretKey);
            byte[] bytes = cipher.doFinal(Base64Decoder.decode(message));
            return new String(bytes, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("解密失败", e);
            throw new BusinessException("解密失败");
        }
    }

    public static String encryptRAS(String message, String secret, String encryptType) throws BusinessException {
        try {
            //获取公钥
            byte[] decoded = Base64Decoder.decode(secret);
            KeyFactory keyFactory = KeyFactory.getInstance(encryptType);
            Key secretKey = keyFactory.generatePublic(new X509EncodedKeySpec(decoded));
            Cipher cipher = new NullCipher();
            if (encryptType.equals(Constants.KEY_RSA)) {
                cipher = Cipher.getInstance(encryptType);
            }
            // 根据密钥，对Cipher对象进行初始化，ENCRYPT_MODE表示加密模式
            cipher.init(Cipher.ENCRYPT_MODE, secretKey);
            byte[] bytes = cipher.doFinal(message.getBytes(StandardCharsets.UTF_8));
            return Base64Encoder.encode(bytes);
        } catch (Exception e) {
            log.error("加密失败", e);
            throw new BusinessException("加密失败");
        }
    }

    public static String decryptRAS(String message, String secret, String encryptType) throws BusinessException {
        try {
            byte[] decoded = Base64Decoder.decode(secret);
            KeyFactory keyFactory = KeyFactory.getInstance(encryptType);
            Key secretKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(decoded));
            Cipher cipher = new NullCipher();
            if (encryptType.equals(Constants.KEY_RSA)) {
                cipher = Cipher.getInstance(encryptType);
            }
            // 根据密钥，对Cipher对象进行初始化，ENCRYPT_MODE表示加密模式
            cipher.init(Cipher.DECRYPT_MODE, secretKey);
            byte[] bytes = cipher.doFinal(Base64Decoder.decode(message));
            return new String(bytes, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("解密失败", e);
            throw new BusinessException("解密失败");
        }
    }
}
