package com.wlos.app.exception;

import cn.hutool.core.date.DateUtil;
import cn.hutool.extra.servlet.ServletUtil;
import cn.hutool.json.JSONUtil;
import com.fasterxml.jackson.core.JsonParseException;

import com.mysql.cj.jdbc.exceptions.CommunicationsException;
import com.mysql.cj.jdbc.exceptions.MysqlDataTruncation;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.DuplicateKeyException;
import org.apache.ibatis.exceptions.TooManyResultsException;
import org.mybatis.spring.MyBatisSystemException;
import org.springframework.jdbc.BadSqlGrammarException;
import org.springframework.jdbc.CannotGetJdbcConnectionException;
import org.springframework.jdbc.UncategorizedSQLException;
import java.sql.SQLSyntaxErrorException;

import com.wlos.app.utils.Result;
import org.apache.catalina.connector.ClientAbortException;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.env.Environment;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindException;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.multipart.MaxUploadSizeExceededException;
import org.springframework.web.util.ContentCachingRequestWrapper;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.Path;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 * @author : jyh
 * @content : 统一异常处理
 * @since : 2021/3/22  13:16
 */
@RestControllerAdvice
public class GlobalExceptionHandler {

    protected final Logger log = LoggerFactory.getLogger(getClass());
    @Autowired
    private Environment environment;
    @Value("{$spring.application.name}")
    private String applicationName;

    @ExceptionHandler(value = Exception.class)
    public Result handleException(Exception e, HttpServletRequest request) {
        printErrorLog(e, request, CommonCode.FAIL.message());
        return Result.FAIL(CommonCode.FAIL.code(), CommonCode.FAIL.message());
    }

    /***
     * 空指针异常处理
     * @param e 异常信息
     * @param request 请求上下文
     * @return
     */
    @ExceptionHandler(NullPointerException.class)
    public Result<?> nullPointerException(NullPointerException e, HttpServletRequest request) {
        printErrorLog(e, request, CommonCode.NPE_FAIL.message());
        return Result.FAIL(CommonCode.NPE_FAIL.code(), CommonCode.NPE_FAIL.message());
    }

    @ExceptionHandler(value = BusinessException.class)
    public Result handleBusinessException(BusinessException businessException, HttpServletRequest request) {
        Result<?> result;
        Map<String, String> paramMap = ServletUtil.getParamMap(request);
        String jsonBody = "";
        String method = request.getMethod();
        if (request instanceof ContentCachingRequestWrapper) {
            ContentCachingRequestWrapper wrapper = (ContentCachingRequestWrapper) request;
            jsonBody = StringUtils.toEncodedString(wrapper.getContentAsByteArray(), Charset.forName(wrapper.getCharacterEncoding()));
        }
        log.info("==========业务异常=======>请求地址：{},请求方式：[{}]参数：[{}],错误信息：{}", request.getRequestURL(), method, jsonBody == "" ? paramMap : jsonBody, businessException.getMessage());
        if (businessException.resultCode != null) {
            result = Result.FAIL(businessException.resultCode.code(), businessException.resultCode.message());
        } else if (businessException.getErrorCode() != null) {
            result = Result.FAIL(businessException.getErrorCode(), businessException.getErrorMsg());
        } else {
            result = Result.FAIL(businessException.getErrorMsg());
        }
        return result;
    }

    /***
     * 参数校验
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(IllegalArgumentException.class)
    public Result<?> illegalArgumentException(IllegalArgumentException e) {
        log.error("非法参数异常:{}", e.getMessage(), e);
        return Result.FAIL(CommonCode.INVALID_PARAM.code(), "非法参数:"+e.getMessage());
    }

    /**
     * 统一处理请求参数校验(实体对象传参)
     *
     * @param e BindException
     * @return CommonResult
     */
    @ExceptionHandler(BindException.class)
    public Result handleBindException(BindException e) {
        log.error("绑定异常:{}", e.getMessage(), e);
        StringBuilder message = new StringBuilder();
        List<FieldError> fieldErrors = e.getBindingResult().getFieldErrors();
        for (FieldError error : fieldErrors) {
            message.append(error.getField()).append(error.getDefaultMessage()).append(",");
        }
        message = new StringBuilder(message.substring(0, message.length() - 1));
        return Result.FAIL(CommonCode.INVALID_PARAM.code(), message.toString());
    }

    /**
     * 统一处理请求参数校验(普通传参)
     *
     * @param e ConstraintViolationException
     * @return CommonResult
     */
    @ExceptionHandler(value = ConstraintViolationException.class)
    public Result handleConstraintViolationException(ConstraintViolationException e) {
        log.error("约束冲突异常:{}", e.getMessage(), e);
        StringBuilder message = new StringBuilder();
        Set<ConstraintViolation<?>> violations = e.getConstraintViolations();
        for (ConstraintViolation<?> violation : violations) {
            Path path = violation.getPropertyPath();
            String[] pathArr = StringUtils.splitByWholeSeparatorPreserveAllTokens(path.toString(), ".");
            message.append(pathArr[1]).append(violation.getMessage()).append(",");
        }
        message = new StringBuilder(message.substring(0, message.length() - 1));
        return Result.FAIL(message.toString());
    }

    /**
     * 统一处理请求参数校验(json)
     *
     * @param e ConstraintViolationException
     * @return CommonResult
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Result handlerMethodArgumentNotValidException(MethodArgumentNotValidException e) {
        log.error("程序方法参数无效异常:{}", e.getMessage(), e);
        StringBuilder message = new StringBuilder();
        for (FieldError error : e.getBindingResult().getFieldErrors()) {
            message.append(error.getField()).append(error.getDefaultMessage()).append("，");
        }
        message = new StringBuilder(message.substring(0, message.length() - 1));
        return Result.FAIL(CommonCode.INVALID_PARAM.code(), message.toString());
    }

    @ExceptionHandler(value = HttpMediaTypeNotSupportedException.class)
    public Result handleHttpMediaTypeNotSupportedException(HttpMediaTypeNotSupportedException e) {
        String message = "该方法不支持" + StringUtils.substringBetween(e.getMessage(), "'", "'") + "媒体类型";
        log.error(message);
        return Result.FAIL(CommonCode.REQUEST_PARAM_TYPE_NOT_SUPPORTED.code(), message);
    }


    /***
     * 请求格式不正确异常处理
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = HttpRequestMethodNotSupportedException.class)
    public Result httpRequestMethodNotSupportedException(HttpRequestMethodNotSupportedException e) {
        String message = "该方法不支持" + StringUtils.substringBetween(e.getMessage(), "'", "'") + "请求方法";
        log.error(message);
        return Result.FAIL(CommonCode.REQUEST_METHOD_NOT_SUPPORTED.code(), CommonCode.REQUEST_METHOD_NOT_SUPPORTED.message());
    }

    /***
     * 请求格式不正确异常处理
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = {HttpMessageNotReadableException.class})
    public Result httpMessageNotReadableException(HttpMessageNotReadableException e) {
        log.error("请求格式不正确异常:{}", e.getMessage(), e);
        Throwable cause = e.getCause();
        StringBuilder messageError = new StringBuilder();
        messageError.append("请求参数错误");
        if (ObjectUtils.isNotEmpty(cause)) {
            if (cause instanceof JsonParseException) {
                messageError.append(",json参数格式错误，请检查入参！");
            }
        }
        return Result.FAIL(CommonCode.REQUEST_PARAM_TYPE_NOT_SUPPORTED.code(), messageError.toString());
    }

    /***
     * 请求格式不正确异常处理
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = {MissingServletRequestParameterException.class})
    public Result missingServletRequestParameterException(MissingServletRequestParameterException e) {
        String parameterName = e.getParameterName();
        String message = "请求参数[" + parameterName + "]不能为空 !";

        return Result.FAIL(CommonCode.INVALID_PARAM.code(), message);
    }

    /***
     * 数据为正常返回，客户端问题，不归属异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = ClientAbortException.class)
    public Result clientAbortException(ClientAbortException e) {
        log.error("客户端中止异常:{}", e.getMessage(), e);
        return Result.FAIL(CommonCode.BROKEN_PIPE.code(), CommonCode.BROKEN_PIPE.message());
    }

    /***
     * 最大上传大小超出异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = MaxUploadSizeExceededException.class)
    public Result maxUploadSizeExceededException(MaxUploadSizeExceededException e) {
        log.error("最大上传大小超出异常:{}", e.getMessage(), e);
        return Result.FAIL(CommonCode.OVER_MAX_FILE_UPLOAD_SIZE.code(), CommonCode.OVER_MAX_FILE_UPLOAD_SIZE.message());
    }

    /***
     * 获取当前环境
     * @return
     */
    public String getEnv() {
        String[] activeProfiles = environment.getActiveProfiles();
        String activeEnv = "";
        if (ArrayUtils.isEmpty(activeProfiles)) {
            activeEnv = "dev";
        } else {
            activeEnv = activeProfiles[0];
        }
        return activeEnv;
    }


    /***
     * 打印错误信息
     * @param e 异常
     * @param request 请求
     * @param errorTitle 错误主题
     */
    private void printErrorLog(Exception e, HttpServletRequest request, String errorTitle) {
        String method = request.getMethod();
        Map<String, String> paramMap = ServletUtil.getParamMap(request);
        String jsonBody = null;
        if (request instanceof ContentCachingRequestWrapper) {
            ContentCachingRequestWrapper wrapper = (ContentCachingRequestWrapper) request;
            jsonBody = StringUtils.toEncodedString(wrapper.getContentAsByteArray(), Charset.forName(wrapper.getCharacterEncoding()));
        }
        StringBuilder errorBuilder = new StringBuilder("【" + applicationName + "】");
        errorBuilder.append("\n ").append(errorTitle).append(" 【").append(getEnv()).append("】");
        errorBuilder.append("\n 请求地址：").append(request.getRequestURL());
        errorBuilder.append("\n 请求方式：[").append(method).append("]");
        errorBuilder.append("\n 请求时间：").append(DateUtil.now());
        errorBuilder.append("\n 请求参数：");
        if (StringUtils.isBlank(jsonBody)) errorBuilder.append("\n ").append(JSONUtil.toJsonStr(paramMap));
        else errorBuilder.append("\n ").append(jsonBody);
        String fullStackTrace = ExceptionUtils.getFullStackTrace(e);
        if (fullStackTrace.length() > 3000) {
            fullStackTrace = fullStackTrace.substring(0, 3000);
        }
        errorBuilder.append("\n 异常信息: ").append(fullStackTrace);
        log.error(errorBuilder.toString(), e);
        Map<String, Object> map = new HashMap<>(2);
        map.put("text", errorBuilder.toString());
        Map<String, Object> resultMap = new HashMap<>(3);
        resultMap.put("msg_type", "text");
        resultMap.put("content", map);

    }



    /***
     * 未分类的 SQL 异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = UncategorizedSQLException.class)
    public Result uncategorizedSQLException(UncategorizedSQLException e) {
        log.error("数据库操作失败：{}", e.getMessage(), e);
        Throwable cause = e.getCause();
        return Result.FAIL(CommonCode.DATA_SOURCE_EXCEPTION.code(), "数据库字段类型不匹配:[" + cause.getMessage() + "]");
    }

    /***
     * 重复键异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = DuplicateKeyException.class)
    public Result duplicateKeyException(DuplicateKeyException e) {
        log.error("重复键异常：{}", e.getMessage(), e);
        Throwable cause = e.getCause();
        return Result.FAIL(CommonCode.DATA_SOURCE_EXCEPTION.code(), "主键冲突:[" + cause.getMessage() + "]");
    }

    /***
     * MyBatis系统异常
     *
     *
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = MyBatisSystemException.class)
    public Result tooManyResultsException(MyBatisSystemException e) {
        log.error("MyBatis系统异常：{}", e.getMessage(), e);
        StringBuilder messageError = new StringBuilder();
        Throwable cause = e.getCause();
        if (null != cause) {
            if (cause instanceof TooManyResultsException) {
                messageError.append("数据错误:返回数据为对象无法接受多条数据,请重新设置");
            } else {
                messageError.append("数据错误:[").append(cause.getMessage()).append("]");
            }
        }
        return Result.FAIL(CommonCode.DATA_INCORRECT.code(), messageError.toString());
    }

    /***
     * sql语法异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = BadSqlGrammarException.class)
    public Result badSqlGrammarException(BadSqlGrammarException e) {
        log.error("Sql语法异常：{}", e.getMessage(), e);
        Throwable cause = e.getCause();
        StringBuilder messageError = new StringBuilder();
        if (cause instanceof SQLSyntaxErrorException) {
            messageError.append("缺少字段，请同步表结构:[").append(cause.getMessage()).append("]");
        } else {
            messageError.append("数据库字段不匹配，请同步:[").append(cause.getMessage()).append("]");
        }
        return Result.FAIL(CommonCode.DATA_SOURCE_EXCEPTION.code(), messageError.toString());
    }
    /***
     * sql语法异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = SQLSyntaxErrorException.class)
    public Result sQLSyntaxErrorException(SQLSyntaxErrorException e) {
        log.error("Sql语法异常：{}", e.getMessage(), e);
        return Result.FAIL(CommonCode.DATA_SOURCE_EXCEPTION.code(), "数据查询失败，请检查流程配置参数！");
    }

    /***
     * 无法获取 Jdbc 连接异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = CannotGetJdbcConnectionException.class)
    public Result cannotGetJdbcConnectionException(CannotGetJdbcConnectionException e) {
        log.error("无法获取 Jdbc 连接异常：{}", e.getMessage(), e);
        StringBuilder messageError = new StringBuilder();
        Throwable cause = e.getCause();
        if (null != cause) {
            if (cause instanceof CommunicationsException) {
                messageError.append("数据库连接失败:[请检查配置]");
            } else {
                messageError.append("数据库连接失败:[请检查账号密码是否正确]");
            }
        }
        return Result.FAIL(CommonCode.DATA_INCORRECT.code(), messageError.toString());
    }

    /***
     * 数据完整性违规异常
     * @param e 异常信息
     * @return
     */
    @ExceptionHandler(value = {DataIntegrityViolationException.class})
    public Result dataIntegrityViolationException(DataIntegrityViolationException e) {
        log.error("数据完整性违规异常：{}", e.getMessage(), e);
        Throwable cause = e.getCause();
        StringBuilder messageError = new StringBuilder();
        if (cause instanceof MysqlDataTruncation) {
            messageError.append("字段超出长度:[").append(cause.getMessage()).append("]");
        } else {
            messageError.append("数据库字段不匹配:[").append(cause.getMessage()).append("]");
        }
        return Result.FAIL(CommonCode.DATA_SOURCE_EXCEPTION.code(), messageError.toString());
    }


}