package com.wlos.app.enums;

import cn.hutool.core.date.DatePattern;
import com.wlos.app.exception.BusinessException;

import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;

public class AnalysisEnums {

    public enum Mode {
        DIMENSION("dimension", "维度"),
        MEASURE("measure", "度量");

        Mode(String type, String label) {
            this.type = type;
            this.label = label;
        }

        private String type;
        private String label;

        public String getType() {
            return type;
        }

        public String getLabel() {
            return label;
        }
    }

    public enum Data {
        CATEGORY("category", "离散"),
        VALUE("value", "连续");

        Data(String type, String label) {
            this.type = type;
            this.label = label;
        }

        private String type;
        private String label;

        public String getType() {
            return type;
        }

        public String getLabel() {
            return label;
        }

        public static Data getData(String dataType) {
            return Arrays.stream(Data.values())
                    .filter(o -> Objects.equals(o.getType(), dataType))
                    .findFirst()
                    .orElseThrow(() -> new BusinessException("未知的数据特点"));
        }
    }

    public enum Order {
        ASC(1, "正序"),
        DESC(-1, "倒序");

        Order(Integer type, String label) {
            this.type = type;
            this.label = label;
        }

        private Integer type;
        private String label;

        public Integer getType() {
            return type;
        }

        public String getLabel() {
            return label;
        }

        public static Order getOrder(Integer order) {
            return Arrays.stream(Order.values())
                    .filter(o -> Objects.equals(o.getType(), order))
                    .findFirst()
                    .orElseThrow(() -> new BusinessException("未知的排序"));
        }
    }

    public enum Aggregate {
        SUM("sum", "求和"),
        AVG("avg", "平均"),
        MAX("max", "最大"),
        MIN("min", "最小"),
        COUNT("count", "计数"),
        MEDIAN("median", "中位数");

        Aggregate(String type, String label) {
            this.type = type;
            this.label = label;
        }

        private String type;
        private String label;

        public String getType() {
            return type;
        }

        public String getLabel() {
            return label;
        }
    }

    public enum DataType {
        String,
        Integer,
    }

    public enum DateType {
        DATE("Date"),
        DATETIME("Datetime"),
        TIME("Time");

        DateType(String type) {
            this.type = type;
        }

        private String type;

        public String getType() {
            return type;
        }

        public static boolean valid(String dateType) {
            Optional<DateType> optional = Arrays.stream(DateType.values())
                    .filter(type -> type.getType().equals(dateType))
                    .findFirst();
            return optional.isPresent();
        }
    }

    public enum Formatter {
        YEAR(DateTimeFormatter.ofPattern(DatePattern.NORM_YEAR_PATTERN),"%Y", "年"),
        QUARTER(DateTimeFormatter.ofPattern("yyyy年第Q季度"), "%Q", "季度"),
        MONTH(DatePattern.NORM_MONTH_FORMATTER,"%m", "月"),
        WEEK(DateTimeFormatter.ofPattern("yyyy年第w周"), "%Y-%u", "周"),
        DAY(DatePattern.NORM_DATE_FORMATTER,"%Y-%m-%d", "日");

        Formatter(DateTimeFormatter format, String type, String label) {
            this.format = format;
            this.type = type;
            this.label = label;
        }

        public static DateTimeFormatter getFormatter(String type) {
            Formatter format = Arrays.stream(Formatter.values())
                    .filter(formatter -> Objects.equals(formatter.getType(), type))
                    .findFirst()
                    .orElseThrow(() -> new BusinessException("未知的时间格式化类型"));
            return format.getFormat();
        }

        private DateTimeFormatter format;
        private String type;
        private String label;

        public DateTimeFormatter getFormat() {
            return format;
        }
        public String getLabel() {
            return label;
        }
        public String getType() {
            return type;
        }
    }

    public enum Axis {
        X,Y
    }

    public enum SeriesItemTag {
        COLOR("color", "颜色"),
        SYMBOL("symbol", "标记"),
        SYMBOL_SIZE("symbolSize", "标记大小"),
        LABEL("label", "标签"),
        TOOLTIP("tooltip", "提示框"),
        BAR_WIDTH("barWidth", "宽度"),
        START_ANGLE("startAngle", "角度"),
        RADIUS("radius", "半径"),
        LINE_STYLE_COLOR("lineStyleColor", "线条颜色"),
        AREA_STYLE_COLOR("areaStyleColor", "区域颜色"),
        LINE_STYLE_WIDTH("lineStyleWidth", "线条大小"),
        ;

        SeriesItemTag(String tag, String label) {
            this.tag = tag;
            this.label = label;
        }

        private String tag;
        private String label;

        public String getTag() {
            return tag;
        }

        public String getLabel() {
            return label;
        }
    }

}
