package com.wlos.app.da.service;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.RandomUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.wlos.app.enums.ModelAttributeEnum;
import com.wlos.app.exception.BusinessException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.text.NumberFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 自动编号服务
 */
@Service
public class AutoincFieldService {

    @Autowired
    private JdbcTemplate jdbcTemplate;

    /**
     * 自动编号映射关系
     */
    private static Map<String, DateTimeFormatter> DateTimeFormatterMAP = new HashMap(){{
        put("YYYYMMDD", DateTimeFormatter.ofPattern("yyyyMMdd"));
        put("YYYYMM", DateTimeFormatter.ofPattern("yyyyMM"));
        put("MMDD", DateTimeFormatter.ofPattern("MMdd"));
        put("YYYY", DateTimeFormatter.ofPattern("yyyy"));
    }};


    /**
     * 获取下一个自增编号
     * @param attribute
     * @param paramMap
     * @return
     */
    @Transactional(rollbackFor = BusinessException.class)
    public String next(ModelAttributeEnum.Attribute attribute, Map<String, Object> paramMap) {
        Assert.notNull(attribute.extData(),"计数规则不存在");
        ModelAttributeEnum.AutoincFieldSetting autoincFieldSetting = (ModelAttributeEnum.AutoincFieldSetting) attribute.extData();
        JSONArray ruleList = JSONArray.parseArray(autoincFieldSetting.rule());
        StringBuilder joinList = new StringBuilder();

        ruleList.forEach(rule -> {
            JSONObject ruleInfo = (JSONObject) rule;
            String genType = ruleInfo.getString("gen_type");
            Integer reset = autoincFieldSetting.reset();
            if ("auto".equalsIgnoreCase(genType)) {
                //自动计数
                long resetTime = getResetTime(reset);
                String number = autoIncrememt(attribute.uuid(), autoincFieldSetting.incStart(), autoincFieldSetting.incLength(), resetTime);
                joinList.append(number);
            } else if("string".equalsIgnoreCase(genType)) {
                //固定字符
                String str = ruleInfo.getString("string");
                joinList.append(str);
            } else if("date".equalsIgnoreCase(genType)) {
                //创建日期
                String pattern = ruleInfo.getString("patten");
                String formatDate = LocalDateTimeUtil.format(LocalDate.now(), DateTimeFormatterMAP.get(pattern));
                joinList.append(formatDate);
            } else if ("property".equalsIgnoreCase(genType)) {
                //引用属性
                String relationEnName = ruleInfo.getString("relationEnName");
                String relationKey = ruleInfo.getString("enName");

                if (StringUtils.isNotBlank(relationEnName)) {
                    throw new BusinessException("暂不支持引用查询字段");
                } else {
                    Object relationValue = paramMap.get(relationKey);
                    joinList.append(relationValue);
                }
            }
        });
        return joinList.toString();
    }

    /**
     * 基于数据编号自增
     * @param uuid
     * @param incStart
     * @param incLength
     * @param resetTime
     * @return
     */
    private String autoIncrememt(String uuid, Integer incStart, Integer incLength, long resetTime) {
        AutoincField autoincField = null;
        List<AutoincField> autoincFields = jdbcTemplate.query("SELECT uuid, identifier, attributeUuid, data, createdAt, updatedAt FROM dmm_dm_autoincrement_data WHERE attributeUuid =  ?",
                new Object[]{uuid}, (rs, rowNum) -> new AutoincField(
                        rs.getString("uuid"),
                        rs.getString("identifier"),
                        rs.getString("attributeUuid"),
                        rs.getInt("data"),
                        rs.getDate("createdAt"),
                        rs.getDate("updatedAt")
                ));
        if (CollectionUtils.isEmpty(autoincFields)) {
            autoincField = new AutoincField(IdUtil.fastUUID(), RandomUtil.randomString(10), uuid, incStart, new Date(), new Date());
            jdbcTemplate.update("insert into dmm_dm_autoincrement_data (uuid, identifier, attributeUuid, data, createdAt, updatedAt) values (?,?,?,?,?,?)",
                    autoincField.uuid(), autoincField.identifier(), autoincField.attributeUuid(), autoincField.data(), autoincField.createdAt(), autoincField.updatedAt());
            NumberFormat formatter = NumberFormat.getNumberInstance();
            formatter.setMinimumIntegerDigits(incLength);
            formatter.setGroupingUsed(false);
            return formatter.format(incStart);
        } else {
            autoincField = CollectionUtil.getFirst(autoincFields);
            int number = 0;
            if (-1 != resetTime){
                Date createdAt = autoincField.createdAt();
                long time = createdAt.getTime();
                long now = System.currentTimeMillis();
                long difference = (now - time)/1000/60/60;
                if (difference < resetTime){
                    int num = autoincField.data;
                    num += 1 ;
                    number = num;
                }else{
                    jdbcTemplate.update("update dmm_dm_autoincrement_data set data = ?, crateDate = ? where attributeUuid = ?", new Object[]{number, new Date(), uuid});
                }
            }else {
                int num = autoincField.data;
                num += 1 ;
                number = num;
                jdbcTemplate.update("update dmm_dm_autoincrement_data set data = ? where attributeUuid = ?", new Object[]{number, uuid});
            }
            NumberFormat formatter = NumberFormat.getNumberInstance();
            formatter.setMinimumIntegerDigits(incLength);
            formatter.setGroupingUsed(false);
            return formatter.format(number);
        }
    }

    /**
     * 重复周期单位转换
     * @param reset
     * @return
     */
    public long getResetTime(Integer reset) {
        if (reset == 1) {
            return -1;
        } else if (reset == 2) {
            return 24l;
        } else if(reset == 3) {
            return 24*7l;
        } else if(reset == 4) {
            return 24*31l;
        } else if (reset == 5) {
            return 24*365l;
        }
        return 0;
    }

    public record AutoincField(String uuid, String identifier, String attributeUuid, int data, Date createdAt, Date updatedAt) implements Serializable {}


}
