package com.wlos.app.bl.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.json.JSONUtil;
import com.wlos.app.bl.AnalysisDataService;
import com.wlos.app.enums.AnalysisEnums;
import com.wlos.app.model.vo.SortDTO;
import com.wlos.app.model.vo.AnalysisDataDTO;
import com.wlos.app.model.vo.SeriesItem;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Service
public class AnalysisDataServiceImpl implements AnalysisDataService {

    private final Logger logger = LoggerFactory.getLogger(getClass());

    @Override
    public Map<String, Object> analysis(List<Map<String,Object>> dataList, List<AnalysisDataDTO> analysisDataModelList, List<SeriesItem> series) {
        Map<String, Object> chartData = Maps.newLinkedHashMap();

        if (CollectionUtils.isEmpty(dataList)) return chartData;

        // 将是否关联表字段放到dm里
        this.isJoinTable2Dm(dataList, analysisDataModelList);

        // 默认排序（解决分组问题）
        this.sortDataList(dataList, this.getSorts(this.getxAxisSortList(analysisDataModelList)));

        logger.info("图表分析源数据: {}", dataList);
        logger.info("图表分析模型数据: {}", JSONUtil.toJsonStr(analysisDataModelList));
        logger.info("图表分析系列数据: {}", JSONUtil.toJsonStr(series));

        if (CollectionUtils.isEmpty(analysisDataModelList)) {
            return chartData;
        }

        // y轴连续数据（判断个数）
        List<AnalysisDataDTO> yAxisValueList = analysisDataModelList.stream()
                .filter(model -> model.getAxis() == AnalysisEnums.Axis.Y)
                .filter(model -> Objects.equals(AnalysisEnums.Data.VALUE.getType(), model.getType()))
                .collect(Collectors.toList());

        // 多系列数据分析数据转换
        if (ObjectUtils.isNotEmpty(series) && ObjectUtils.isNotEmpty(yAxisValueList)) {
            List<AnalysisDataDTO> xAxisList = analysisDataModelList.stream().filter(model -> model.getAxis() == AnalysisEnums.Axis.X).collect(Collectors.toList());

            // 获取组合排序key
            List<String> seriesAnalysisKeys = this.getAnalysisKeys(analysisDataModelList, series);

            // y轴离散数据
            List<AnalysisDataDTO> yAxisCategoryList = analysisDataModelList.stream()
                    .filter(model -> model.getAxis() == AnalysisEnums.Axis.Y)
                    .filter(model -> Objects.equals(AnalysisEnums.Data.CATEGORY.getType(), model.getType()))
                    .collect(Collectors.toList());

            List<Map<String, Object>> seriesAnalysis = Lists.newArrayList();
            yAxisValueList.forEach(yAxisValue -> {
                int i = yAxisValueList.indexOf(yAxisValue);
                SeriesItem seriesItem = series.get(i);
                List<AnalysisDataDTO> transformAnalysisDataDTO = seriesItem.transformAnalysisDataModelList();
                // x轴=x轴 + 系列  y轴=y轴连续+y轴离散  即所求=x轴+y轴
                List<AnalysisDataDTO> seriesList = ListUtils.union(xAxisList, transformAnalysisDataDTO);
                seriesList.add(yAxisValue);
                seriesList.addAll(yAxisCategoryList);

                List<String> analysisKeys = this.getAnalysisKeys(seriesList);

                List<Map<String, Object>> singleAnalysis = this.singleAnalysis(dataList, seriesList, analysisKeys);
                seriesAnalysis.addAll(singleAnalysis);
            });

            // 将数据按照排序组合
            seriesAnalysis.forEach(s -> {
                List<Object> list = Lists.newArrayList();
                seriesAnalysisKeys.forEach(k -> {
                    list.add(s.getOrDefault(k, null));
                });
                chartData.put("ct_" + IdUtil.simpleUUID(), list);
            });

        } else {
            List<String> analysisKeys = this.getAnalysisKeys(analysisDataModelList);

            List<Map<String, Object>> chartList = singleAnalysis(dataList, analysisDataModelList, analysisKeys);

            chartList.forEach(c -> chartData.put("ct_" + IdUtil.simpleUUID(), c.values()));
        }


        logger.info("图表分析结果数据: {}", chartData);
        return chartData;
    }

    /**
     * 单系列数据分析
     * @param dataList
     * @param analysisDataModelList
     * @param analysisKeys
     * @return
     */
    private List<Map<String, Object>> singleAnalysis(List<Map<String, Object>> dataList, List<AnalysisDataDTO> analysisDataModelList, List<String> analysisKeys) {
        // 日期类型转换
        List<Map<String, Object>> parseDataList = dataList.stream().map(data -> {
            AnalysisDataDTO model = analysisDataModelList.stream()
                    .filter(dataModel -> StringUtils.isNotBlank(dataModel.getDataType()))
                    .filter(dataModel -> StringUtils.isNotBlank(dataModel.getFormatter()))
                    .filter(dataModel -> data.containsKey(this.parseColumnUuid(dataModel)))
                    .findFirst()
                    .orElse(null);
            return Objects.isNull(model) ? data : this.formatDate(model, data);
        }).collect(Collectors.toList());

        List<List<Object>> itemsList = Lists.newArrayList();
        analysisDataModelList.forEach(analysisDataModel -> {
            //进行分组
            Map<String, List<Map<String, Object>>> groupList = parseDataList.stream()
                    .collect(Collectors.groupingBy(d -> this.fetchGroupKey(d, analysisDataModelList), LinkedHashMap::new, Collectors.toList()));

            //运算
            List<Object> operationList = this.operation(groupList, analysisDataModel);

            itemsList.add(operationList);
        });

        // 数据格式转换
        List<Map<String, Object>> chartList = Lists.newArrayList();
        Stream.iterate(0, i -> i + 1).limit(itemsList.get(0).size()).forEach(i -> {
            Map<String, Object> chart = Maps.newLinkedHashMap();
            Stream.iterate(0, j -> j + 1).limit(itemsList.size()).forEach(j -> {
                chart.put(analysisKeys.get(j), itemsList.get(j).get(i));
            });
            chartList.add(chart);
        });

        // 排序
        List<AnalysisDataDTO> rankList = analysisDataModelList.stream()
                .filter(d -> Objects.nonNull(d.getRankType()))
                .collect(Collectors.toList());
        if (CollectionUtil.isNotEmpty(rankList)) {
            this.sortDataList(chartList, this.getSorts(rankList));
        }
        return chartList;
    }

    /**
     * 维度group by字段
     * @param data
     * @param analysisDataModelList
     * @return
     */
    private String fetchGroupKey(Map<String, Object> data, List<AnalysisDataDTO> analysisDataModelList){
        List<AnalysisDataDTO> dimensionList = analysisDataModelList.stream()
                .filter(m -> Objects.equals(AnalysisEnums.Mode.DIMENSION.getType(), m.getMode()))
                .collect(Collectors.toList());
        List<String> keyList = dimensionList.stream()
                .filter(d -> data.containsKey(this.parseColumnUuid(d)))
                .map(d -> Optional.ofNullable(data.get(this.parseColumnUuid(d)))
                        .map(Object::toString)
                        .orElse("")
                )
                .collect(Collectors.toList());
        return StringUtils.join(keyList, ",");
    }

    /**
     * 日期格式化
     * @param model
     * @param data
     * @return
     */
    private Map<String,Object> formatDate(AnalysisDataDTO model, Map<String,Object> data) {
        Assert.isTrue(AnalysisEnums.DateType.valid(model.getDataType()), "非日期类型不能格式化");
        String parseColumnUuid = this.parseColumnUuid(model);
        if (!data.containsKey(parseColumnUuid)) {
            return data;
        }
        DateTime formatField = DateUtil.parse(data.get(parseColumnUuid).toString());;

        Map<String, Object> parseDataMap = Maps.newHashMap(data);
        String formatDate = DateUtil.format(formatField, AnalysisEnums.Formatter.getFormatter(model.getFormatter()));
        parseDataMap.put(parseColumnUuid, formatDate);
        return parseDataMap;
    }

    /**
     * 运算
     * @param groupList
     * @param analysisDataModel
     * @return
     */
    private List<Object> operation(Map<String, List<Map<String, Object>>> groupList, AnalysisDataDTO analysisDataModel) {
        String columnUuid = this.parseColumnUuid(analysisDataModel);

        // 度量
        if (Objects.equals(AnalysisEnums.Mode.MEASURE.getType(), analysisDataModel.getMode())) {
            // 度量默认求和
            List<Object> resultList = Lists.newArrayList();

            if (Objects.equals(AnalysisEnums.Aggregate.SUM.getType(), analysisDataModel.getAggregateType())) {
                groupList.forEach((group, groupInfo) -> {
                    long total = groupInfo.stream().mapToLong(d -> Optional.ofNullable(MapUtil.getLong(d, columnUuid)).orElse(0L)).sum();
                    resultList.add(total);
                });
            }
            if (Objects.equals(AnalysisEnums.Aggregate.AVG.getType(), analysisDataModel.getAggregateType())) {
                groupList.forEach((group, groupInfo) -> {
                    double average = groupInfo.stream().mapToLong(d -> Optional.ofNullable(MapUtil.getLong(d, columnUuid)).orElse(0L)).average().getAsDouble();
                    resultList.add(average);
                });
            }
            if (Objects.equals(AnalysisEnums.Aggregate.MAX.getType(), analysisDataModel.getAggregateType())) {
                groupList.forEach((group, groupInfo) -> {
                    long max = groupInfo.stream().mapToLong(d -> Optional.ofNullable(MapUtil.getLong(d, columnUuid)).orElse(0L)).max().getAsLong();
                    resultList.add(max);
                });
            }
            if (Objects.equals(AnalysisEnums.Aggregate.MIN.getType(), analysisDataModel.getAggregateType())) {
                groupList.forEach((group, groupInfo) -> {
                    long min = groupInfo.stream().mapToLong(d -> Optional.ofNullable(MapUtil.getLong(d, columnUuid)).orElse(0L)).min().getAsLong();
                    resultList.add(min);
                });
            }
            if (Objects.equals(AnalysisEnums.Aggregate.COUNT.getType(), analysisDataModel.getAggregateType())) {
                groupList.forEach((group, groupInfo) -> {
                    long count = groupInfo.stream().map(d -> MapUtil.getStr(d, columnUuid)).count();
                    resultList.add(count);
                });
            }
            if (Objects.equals(AnalysisEnums.Aggregate.MEDIAN.getType(), analysisDataModel.getAggregateType())) {
                groupList.forEach((group, groupInfo) -> {
                    List<Integer> list = groupInfo.stream().map(d -> Optional.ofNullable(MapUtil.getInt(d, columnUuid)).orElse(0)).collect(Collectors.toList());
                    double median = this.median(list);
                    resultList.add(median);
                });
            }
            return resultList;
        } else {
            List<Object> list = Lists.newArrayList();
            groupList.entrySet().forEach(entry -> {
                String key = entry.getKey();
                String value = entry.getValue().stream()
                        .filter(d -> d.containsKey(columnUuid))
                        .map(d -> MapUtil.getStr(d, columnUuid))
                        .filter(Objects::nonNull)
                        .findFirst().orElse("");
                List<String> keyList = Arrays.asList(key.split(","));
                int i = keyList.indexOf(value);
                list.add(i == -1?null:keyList.get(i));
            });
            return list;
        }
    }

    /**
     * 中位数
     * @param list
     * @return
     */
    private double median(List<Integer> list) {
        double s = 0;
        Collections.sort(list);
        int size = list.size();
        if(size % 2 != 1){
            s = (list.get(size/2-1) + list.get(size/2)+0.0)/2;//加0.0是为了计算是将int类型转换为浮点型
        }else {
            s = list.get((size-1)/2);
        }
        return s;
    }

    /**
     * 是否关联表
     * @param dataList
     * @param analysisDataModelList
     */
    private void isJoinTable2Dm(List<Map<String,Object>> dataList, List<AnalysisDataDTO> analysisDataModelList) {
        Map<String, Object> data = dataList.get(0);
        Optional<AnalysisDataDTO> optional = analysisDataModelList.stream()
                .filter(dm -> data.containsKey(dm.getTableUuid() + "__" + dm.getColumnUuid()))
                .findAny();

        analysisDataModelList.forEach(dm -> dm.setJoinTable(optional.isPresent()));
    }

    /**
     * 表关联列uuid
     * @param analysisDataDTO
     * @return
     */
    private String parseColumnUuid(AnalysisDataDTO analysisDataDTO) {
        return analysisDataDTO.isJoinTable()? analysisDataDTO.getTableUuid() + "__" + analysisDataDTO.getColumnUuid(): analysisDataDTO.getColumnUuid();
    }

    /**
     * 分析模型字段标识
     * @param analysisDataModelList
     * @return
     */
    private List<String> getAnalysisKeys(List<AnalysisDataDTO> analysisDataModelList) {
        return analysisDataModelList.stream().map(analysisDataDTO -> this.parseColumnUuid(analysisDataDTO)).collect(Collectors.toList());
    }

    /**
     * 多系列模型字段排序key
     * @param analysisDataModelList
     * @param series
     * @return
     */
    private List<String> getAnalysisKeys(List<AnalysisDataDTO> analysisDataModelList, List<SeriesItem> series) {
        List<AnalysisDataDTO> seriesDataModelList = Lists.newArrayList();
        series.forEach(seriesItem -> {
            List<AnalysisDataDTO> list = seriesItem.transformAnalysisDataModelList();
            seriesDataModelList.addAll(list);
        });
        analysisDataModelList.addAll(seriesDataModelList);
        return analysisDataModelList.stream()
                .map(analysisDataDTO -> this.parseColumnUuid(analysisDataDTO))
                .collect(Collectors.toList());
    }

    /**
     * X轴默认正序排列字段
     * @param analysisDataModelList
     * @return
     */
    private List<AnalysisDataDTO> getxAxisSortList(List<AnalysisDataDTO> analysisDataModelList) {
        return analysisDataModelList.stream()
                .filter(model -> model.getAxis() == AnalysisEnums.Axis.X)
                .peek(model -> model.setRankType(AnalysisEnums.Order.ASC.getType()))
                .collect(Collectors.toList());
    }

    /**
     * 排序
     * @param dataList 数据源
     * @param sorts 排序字段
     */
    private void sortDataList(List<Map<String,Object>> dataList, List<SortDTO> sorts) {
        if (CollectionUtils.isEmpty(sorts)) return;
        Comparator<Map<String,Object>> comparator = null;
        int count = 0;
        for (SortDTO sort : sorts) {
            if (count == 0) {
                comparator = firstSort(sort);
            } else {
                thenOtherSort(comparator, sort);
            }
            count++;
        }

        //排序
        if (Objects.nonNull(comparator)) {
            dataList.sort(comparator);
        }
    }

    /**
     * 排序字段
     * @param analysisDataModelList
     * @return
     */
    private List<SortDTO> getSorts(List<AnalysisDataDTO> analysisDataModelList) {
        return analysisDataModelList.stream().map(analysisDataDTO -> {
            String parseColumnUuid = this.parseColumnUuid(analysisDataDTO);
            return new SortDTO(parseColumnUuid, AnalysisEnums.Data.getData(analysisDataDTO.getType()), AnalysisEnums.Order.getOrder(analysisDataDTO.getRankType()));
        }).collect(Collectors.toList());
    }

    /**
     * 第一排序
     * @param sort
     */
    private Comparator<Map<String, Object>> firstSort(SortDTO sort) {
        Comparator<Map<String, Object>> comparator = null;
        switch (sort.getDataType()) {
            case VALUE -> comparator =  Comparator.comparing(d -> MapUtil.getLong(d, sort.getFieldName()), Comparator.nullsLast(Long::compareTo));
            case CATEGORY -> comparator =  Comparator.comparing(d -> MapUtil.getStr(d, sort.getFieldName()), Comparator.nullsLast(String::compareTo));
        }
        if (AnalysisEnums.Order.DESC == sort.getSort()) {
            comparator = Objects.requireNonNull(comparator).reversed();
        }
        return comparator;
    }

    /**
     * 其他排序
     * @param comparator
     * @param sort
     */
    private void thenOtherSort(Comparator<Map<String, Object>> comparator, SortDTO sort) {
        switch (sort.getDataType()) {
            case VALUE -> comparator =  comparator.thenComparing(d -> MapUtil.getLong(d, sort.getFieldName()), Comparator.nullsLast(Long::compareTo));
            case CATEGORY -> comparator =  comparator.thenComparing(d -> MapUtil.getStr(d, sort.getFieldName()), Comparator.nullsLast(String::compareTo));
        }
        if (AnalysisEnums.Order.DESC == sort.getSort()) {
            comparator = comparator.reversed();
        }
    }

}
