package com.wlos.app.api;

import com.wlos.app.model.SurveyUser;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyUserService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 用户表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:27 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyUserController {

    @Autowired
    private SurveyUserService surveyUserService;


    /**
     * 根据id查询用户表
     *
     * @param id 用户表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyUser/{id}")
    public Result<SurveyUser> getById(@PathVariable String id) {
        SurveyUser surveyUser = surveyUserService.queryById(id);
        return Result.SUCCESS(surveyUser);
    }

    /**
     * 根据id列表查询用户表
     *
     * @param ids 用户表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyUser/batch")
    public Result<List<SurveyUser>> getByIds(@RequestBody List<String> ids) {
        List<SurveyUser> surveyUserList = surveyUserService.listByIds(ids);
        return Result.SUCCESS(surveyUserList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyUser/merge")
    public Result<Set<SurveyUser>> merge(@RequestBody List<List<SurveyUser>> list) {
        Set<SurveyUser> newMergedList = new HashSet<>();

        for (List<SurveyUser> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyUser/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyUser> list) {
        List<String> idList = list.stream().map(SurveyUser::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询用户表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyUser/one")
    public Result<SurveyUser> one(SurveyUser condition) {
        SurveyUser surveyUser = surveyUserService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyUser);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyUser/count")
    public Result<Long> count(SurveyUser condition) {
        long count = surveyUserService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyUser/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyUser condition) {
        QueryWrapper<SurveyUser> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyUserService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyUser/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyUser> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyUserService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 用户表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyUser/page")
    public Result<Page<SurveyUser>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyUser condition) {
        Page<SurveyUser> pageResult = surveyUserService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 用户表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyUser/list")
    public Result<List<SurveyUser>> list(SurveyUser condition) {
        List<SurveyUser> listResult = surveyUserService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyUser/condition/list")
    public Result<List<SurveyUser>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyUser> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyUser> list = surveyUserService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建用户表
     *
     * @param surveyUser 用户表载体
     * @return 用户表 主键
     */
    @PostMapping("/v1/surveyUser")
    public Result<String> create(@RequestBody SurveyUser surveyUser){
        return Result.SUCCESS(surveyUserService.insert(surveyUser));
    }


    /**
     * 根据主键更新用户表
     *
     * @param id 用户表ID
     * @param surveyUser 用户表载体
     * @return Id
     */
    @PutMapping("/v1/surveyUser/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyUser surveyUser){
        return Result.SUCCESS(surveyUserService.update(id, surveyUser));
    }


    /**
     * 根据主键删除用户表
     *
     * @param id 用户表ID
     * @return
     */
    @DeleteMapping("/v1/surveyUser/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyUserService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除用户表
     *
     * @param ids 用户表id集合
     * @return
     */
    @DeleteMapping("/v1/surveyUser/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyUserService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}