package com.wlos.app.api;

import com.wlos.app.model.SurveyTag;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyTagService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 标签表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:25 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyTagController {

    @Autowired
    private SurveyTagService surveyTagService;


    /**
     * 根据id查询标签表
     *
     * @param id 标签表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyTag/{id}")
    public Result<SurveyTag> getById(@PathVariable String id) {
        SurveyTag surveyTag = surveyTagService.queryById(id);
        return Result.SUCCESS(surveyTag);
    }

    /**
     * 根据id列表查询标签表
     *
     * @param ids 标签表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyTag/batch")
    public Result<List<SurveyTag>> getByIds(@RequestBody List<String> ids) {
        List<SurveyTag> surveyTagList = surveyTagService.listByIds(ids);
        return Result.SUCCESS(surveyTagList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyTag/merge")
    public Result<Set<SurveyTag>> merge(@RequestBody List<List<SurveyTag>> list) {
        Set<SurveyTag> newMergedList = new HashSet<>();

        for (List<SurveyTag> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyTag/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyTag> list) {
        List<String> idList = list.stream().map(SurveyTag::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询标签表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyTag/one")
    public Result<SurveyTag> one(SurveyTag condition) {
        SurveyTag surveyTag = surveyTagService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyTag);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyTag/count")
    public Result<Long> count(SurveyTag condition) {
        long count = surveyTagService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyTag/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyTag condition) {
        QueryWrapper<SurveyTag> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyTagService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyTag/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyTag> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyTagService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 标签表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyTag/page")
    public Result<Page<SurveyTag>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyTag condition) {
        Page<SurveyTag> pageResult = surveyTagService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 标签表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyTag/list")
    public Result<List<SurveyTag>> list(SurveyTag condition) {
        List<SurveyTag> listResult = surveyTagService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyTag/condition/list")
    public Result<List<SurveyTag>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyTag> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyTag> list = surveyTagService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建标签表
     *
     * @param surveyTag 标签表载体
     * @return 标签表 主键
     */
    @PostMapping("/v1/surveyTag")
    public Result<String> create(@RequestBody SurveyTag surveyTag){
        return Result.SUCCESS(surveyTagService.insert(surveyTag));
    }


    /**
     * 根据主键更新标签表
     *
     * @param id 标签表ID
     * @param surveyTag 标签表载体
     * @return Id
     */
    @PutMapping("/v1/surveyTag/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyTag surveyTag){
        return Result.SUCCESS(surveyTagService.update(id, surveyTag));
    }


    /**
     * 根据主键删除标签表
     *
     * @param id 标签表ID
     * @return
     */
    @DeleteMapping("/v1/surveyTag/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyTagService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除标签表
     *
     * @param ids 标签表id集合
     * @return
     */
    @DeleteMapping("/v1/surveyTag/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyTagService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}