package com.wlos.app.api;

import com.wlos.app.model.SurveyPlan;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyPlanService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 勘查计划服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:28 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyPlanController {

    @Autowired
    private SurveyPlanService surveyPlanService;


    /**
     * 根据id查询勘查计划
     *
     * @param id 勘查计划Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyPlan/{id}")
    public Result<SurveyPlan> getById(@PathVariable String id) {
        SurveyPlan surveyPlan = surveyPlanService.queryById(id);
        return Result.SUCCESS(surveyPlan);
    }

    /**
     * 根据id列表查询勘查计划
     *
     * @param ids 勘查计划Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyPlan/batch")
    public Result<List<SurveyPlan>> getByIds(@RequestBody List<String> ids) {
        List<SurveyPlan> surveyPlanList = surveyPlanService.listByIds(ids);
        return Result.SUCCESS(surveyPlanList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyPlan/merge")
    public Result<Set<SurveyPlan>> merge(@RequestBody List<List<SurveyPlan>> list) {
        Set<SurveyPlan> newMergedList = new HashSet<>();

        for (List<SurveyPlan> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyPlan/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyPlan> list) {
        List<String> idList = list.stream().map(SurveyPlan::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询勘查计划
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyPlan/one")
    public Result<SurveyPlan> one(SurveyPlan condition) {
        SurveyPlan surveyPlan = surveyPlanService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyPlan);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyPlan/count")
    public Result<Long> count(SurveyPlan condition) {
        long count = surveyPlanService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyPlan/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyPlan condition) {
        QueryWrapper<SurveyPlan> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyPlanService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyPlan/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyPlan> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyPlanService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 勘查计划
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyPlan/page")
    public Result<Page<SurveyPlan>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyPlan condition) {
        Page<SurveyPlan> pageResult = surveyPlanService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 勘查计划
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyPlan/list")
    public Result<List<SurveyPlan>> list(SurveyPlan condition) {
        List<SurveyPlan> listResult = surveyPlanService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyPlan/condition/list")
    public Result<List<SurveyPlan>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyPlan> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyPlan> list = surveyPlanService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建勘查计划
     *
     * @param surveyPlan 勘查计划载体
     * @return 勘查计划 主键
     */
    @PostMapping("/v1/surveyPlan")
    public Result<String> create(@RequestBody SurveyPlan surveyPlan){
        return Result.SUCCESS(surveyPlanService.insert(surveyPlan));
    }


    /**
     * 根据主键更新勘查计划
     *
     * @param id 勘查计划ID
     * @param surveyPlan 勘查计划载体
     * @return Id
     */
    @PutMapping("/v1/surveyPlan/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyPlan surveyPlan){
        return Result.SUCCESS(surveyPlanService.update(id, surveyPlan));
    }


    /**
     * 根据主键删除勘查计划
     *
     * @param id 勘查计划ID
     * @return
     */
    @DeleteMapping("/v1/surveyPlan/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyPlanService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除勘查计划
     *
     * @param ids 勘查计划id集合
     * @return
     */
    @DeleteMapping("/v1/surveyPlan/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyPlanService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}