package com.wlos.app.api;

import com.wlos.app.model.SurveyInfo;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyInfoService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 查勘现场具体问题及反馈详情服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyInfoController {

    @Autowired
    private SurveyInfoService surveyInfoService;


    /**
     * 根据id查询查勘现场具体问题及反馈详情
     *
     * @param id 查勘现场具体问题及反馈详情Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyInfo/{id}")
    public Result<SurveyInfo> getById(@PathVariable String id) {
        SurveyInfo surveyInfo = surveyInfoService.queryById(id);
        return Result.SUCCESS(surveyInfo);
    }

    /**
     * 根据id列表查询查勘现场具体问题及反馈详情
     *
     * @param ids 查勘现场具体问题及反馈详情Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyInfo/batch")
    public Result<List<SurveyInfo>> getByIds(@RequestBody List<String> ids) {
        List<SurveyInfo> surveyInfoList = surveyInfoService.listByIds(ids);
        return Result.SUCCESS(surveyInfoList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyInfo/merge")
    public Result<Set<SurveyInfo>> merge(@RequestBody List<List<SurveyInfo>> list) {
        Set<SurveyInfo> newMergedList = new HashSet<>();

        for (List<SurveyInfo> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyInfo/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyInfo> list) {
        List<String> idList = list.stream().map(SurveyInfo::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询查勘现场具体问题及反馈详情
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyInfo/one")
    public Result<SurveyInfo> one(SurveyInfo condition) {
        SurveyInfo surveyInfo = surveyInfoService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyInfo);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyInfo/count")
    public Result<Long> count(SurveyInfo condition) {
        long count = surveyInfoService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyInfo/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyInfo condition) {
        QueryWrapper<SurveyInfo> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyInfoService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyInfo/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyInfo> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyInfoService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 查勘现场具体问题及反馈详情
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyInfo/page")
    public Result<Page<SurveyInfo>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyInfo condition) {
        Page<SurveyInfo> pageResult = surveyInfoService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 查勘现场具体问题及反馈详情
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyInfo/list")
    public Result<List<SurveyInfo>> list(SurveyInfo condition) {
        List<SurveyInfo> listResult = surveyInfoService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyInfo/condition/list")
    public Result<List<SurveyInfo>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyInfo> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyInfo> list = surveyInfoService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建查勘现场具体问题及反馈详情
     *
     * @param surveyInfo 查勘现场具体问题及反馈详情载体
     * @return 查勘现场具体问题及反馈详情 主键
     */
    @PostMapping("/v1/surveyInfo")
    public Result<String> create(@RequestBody SurveyInfo surveyInfo){
        return Result.SUCCESS(surveyInfoService.insert(surveyInfo));
    }


    /**
     * 根据主键更新查勘现场具体问题及反馈详情
     *
     * @param id 查勘现场具体问题及反馈详情ID
     * @param surveyInfo 查勘现场具体问题及反馈详情载体
     * @return Id
     */
    @PutMapping("/v1/surveyInfo/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyInfo surveyInfo){
        return Result.SUCCESS(surveyInfoService.update(id, surveyInfo));
    }


    /**
     * 根据主键删除查勘现场具体问题及反馈详情
     *
     * @param id 查勘现场具体问题及反馈详情ID
     * @return
     */
    @DeleteMapping("/v1/surveyInfo/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyInfoService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除查勘现场具体问题及反馈详情
     *
     * @param ids 查勘现场具体问题及反馈详情id集合
     * @return
     */
    @DeleteMapping("/v1/surveyInfo/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyInfoService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}