package com.wlos.app.api;

import com.wlos.app.model.SurveyFeedback;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyFeedbackService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 查勘反馈服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyFeedbackController {

    @Autowired
    private SurveyFeedbackService surveyFeedbackService;


    /**
     * 根据id查询查勘反馈
     *
     * @param id 查勘反馈Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyFeedback/{id}")
    public Result<SurveyFeedback> getById(@PathVariable String id) {
        SurveyFeedback surveyFeedback = surveyFeedbackService.queryById(id);
        return Result.SUCCESS(surveyFeedback);
    }

    /**
     * 根据id列表查询查勘反馈
     *
     * @param ids 查勘反馈Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyFeedback/batch")
    public Result<List<SurveyFeedback>> getByIds(@RequestBody List<String> ids) {
        List<SurveyFeedback> surveyFeedbackList = surveyFeedbackService.listByIds(ids);
        return Result.SUCCESS(surveyFeedbackList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyFeedback/merge")
    public Result<Set<SurveyFeedback>> merge(@RequestBody List<List<SurveyFeedback>> list) {
        Set<SurveyFeedback> newMergedList = new HashSet<>();

        for (List<SurveyFeedback> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyFeedback/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyFeedback> list) {
        List<String> idList = list.stream().map(SurveyFeedback::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询查勘反馈
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyFeedback/one")
    public Result<SurveyFeedback> one(SurveyFeedback condition) {
        SurveyFeedback surveyFeedback = surveyFeedbackService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyFeedback);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyFeedback/count")
    public Result<Long> count(SurveyFeedback condition) {
        long count = surveyFeedbackService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyFeedback/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyFeedback condition) {
        QueryWrapper<SurveyFeedback> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyFeedbackService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyFeedback/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyFeedback> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyFeedbackService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 查勘反馈
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyFeedback/page")
    public Result<Page<SurveyFeedback>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyFeedback condition) {
        Page<SurveyFeedback> pageResult = surveyFeedbackService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 查勘反馈
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyFeedback/list")
    public Result<List<SurveyFeedback>> list(SurveyFeedback condition) {
        List<SurveyFeedback> listResult = surveyFeedbackService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyFeedback/condition/list")
    public Result<List<SurveyFeedback>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyFeedback> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyFeedback> list = surveyFeedbackService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建查勘反馈
     *
     * @param surveyFeedback 查勘反馈载体
     * @return 查勘反馈 主键
     */
    @PostMapping("/v1/surveyFeedback")
    public Result<String> create(@RequestBody SurveyFeedback surveyFeedback){
        return Result.SUCCESS(surveyFeedbackService.insert(surveyFeedback));
    }


    /**
     * 根据主键更新查勘反馈
     *
     * @param id 查勘反馈ID
     * @param surveyFeedback 查勘反馈载体
     * @return Id
     */
    @PutMapping("/v1/surveyFeedback/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyFeedback surveyFeedback){
        return Result.SUCCESS(surveyFeedbackService.update(id, surveyFeedback));
    }


    /**
     * 根据主键删除查勘反馈
     *
     * @param id 查勘反馈ID
     * @return
     */
    @DeleteMapping("/v1/surveyFeedback/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyFeedbackService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除查勘反馈
     *
     * @param ids 查勘反馈id集合
     * @return
     */
    @DeleteMapping("/v1/surveyFeedback/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyFeedbackService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}