package com.wlos.app.api;

import com.wlos.app.model.SurveyAssist;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.SurveyAssistService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 勘查协助服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:27 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class SurveyAssistController {

    @Autowired
    private SurveyAssistService surveyAssistService;


    /**
     * 根据id查询勘查协助
     *
     * @param id 勘查协助Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyAssist/{id}")
    public Result<SurveyAssist> getById(@PathVariable String id) {
        SurveyAssist surveyAssist = surveyAssistService.queryById(id);
        return Result.SUCCESS(surveyAssist);
    }

    /**
     * 根据id列表查询勘查协助
     *
     * @param ids 勘查协助Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/surveyAssist/batch")
    public Result<List<SurveyAssist>> getByIds(@RequestBody List<String> ids) {
        List<SurveyAssist> surveyAssistList = surveyAssistService.listByIds(ids);
        return Result.SUCCESS(surveyAssistList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/surveyAssist/merge")
    public Result<Set<SurveyAssist>> merge(@RequestBody List<List<SurveyAssist>> list) {
        Set<SurveyAssist> newMergedList = new HashSet<>();

        for (List<SurveyAssist> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/surveyAssist/extract")
    public Result<List<String>> extract(@RequestBody List<SurveyAssist> list) {
        List<String> idList = list.stream().map(SurveyAssist::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询勘查协助
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/surveyAssist/one")
    public Result<SurveyAssist> one(SurveyAssist condition) {
        SurveyAssist surveyAssist = surveyAssistService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(surveyAssist);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/surveyAssist/count")
    public Result<Long> count(SurveyAssist condition) {
        long count = surveyAssistService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/surveyAssist/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, SurveyAssist condition) {
        QueryWrapper<SurveyAssist> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = surveyAssistService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/surveyAssist/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<SurveyAssist> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = surveyAssistService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 勘查协助
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyAssist/page")
    public Result<Page<SurveyAssist>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            SurveyAssist condition) {
        Page<SurveyAssist> pageResult = surveyAssistService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 勘查协助
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/surveyAssist/list")
    public Result<List<SurveyAssist>> list(SurveyAssist condition) {
        List<SurveyAssist> listResult = surveyAssistService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/surveyAssist/condition/list")
    public Result<List<SurveyAssist>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<SurveyAssist> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<SurveyAssist> list = surveyAssistService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建勘查协助
     *
     * @param surveyAssist 勘查协助载体
     * @return 勘查协助 主键
     */
    @PostMapping("/v1/surveyAssist")
    public Result<String> create(@RequestBody SurveyAssist surveyAssist){
        return Result.SUCCESS(surveyAssistService.insert(surveyAssist));
    }


    /**
     * 根据主键更新勘查协助
     *
     * @param id 勘查协助ID
     * @param surveyAssist 勘查协助载体
     * @return Id
     */
    @PutMapping("/v1/surveyAssist/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody SurveyAssist surveyAssist){
        return Result.SUCCESS(surveyAssistService.update(id, surveyAssist));
    }


    /**
     * 根据主键删除勘查协助
     *
     * @param id 勘查协助ID
     * @return
     */
    @DeleteMapping("/v1/surveyAssist/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        surveyAssistService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除勘查协助
     *
     * @param ids 勘查协助id集合
     * @return
     */
    @DeleteMapping("/v1/surveyAssist/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        surveyAssistService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}