package com.wlos.app.api;

import com.wlos.app.model.BaseRegion;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.BaseRegionService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 地区表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:27 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class BaseRegionController {

    @Autowired
    private BaseRegionService baseRegionService;


    /**
     * 根据id查询地区表
     *
     * @param id 地区表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseRegion/{id}")
    public Result<BaseRegion> getById(@PathVariable String id) {
        BaseRegion baseRegion = baseRegionService.queryById(id);
        return Result.SUCCESS(baseRegion);
    }

    /**
     * 根据id列表查询地区表
     *
     * @param ids 地区表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/baseRegion/batch")
    public Result<List<BaseRegion>> getByIds(@RequestBody List<String> ids) {
        List<BaseRegion> baseRegionList = baseRegionService.listByIds(ids);
        return Result.SUCCESS(baseRegionList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/baseRegion/merge")
    public Result<Set<BaseRegion>> merge(@RequestBody List<List<BaseRegion>> list) {
        Set<BaseRegion> newMergedList = new HashSet<>();

        for (List<BaseRegion> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/baseRegion/extract")
    public Result<List<String>> extract(@RequestBody List<BaseRegion> list) {
        List<String> idList = list.stream().map(BaseRegion::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询地区表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseRegion/one")
    public Result<BaseRegion> one(BaseRegion condition) {
        BaseRegion baseRegion = baseRegionService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(baseRegion);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/baseRegion/count")
    public Result<Long> count(BaseRegion condition) {
        long count = baseRegionService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/baseRegion/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, BaseRegion condition) {
        QueryWrapper<BaseRegion> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = baseRegionService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/baseRegion/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<BaseRegion> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = baseRegionService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 地区表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseRegion/page")
    public Result<Page<BaseRegion>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            BaseRegion condition) {
        Page<BaseRegion> pageResult = baseRegionService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 地区表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseRegion/list")
    public Result<List<BaseRegion>> list(BaseRegion condition) {
        List<BaseRegion> listResult = baseRegionService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/baseRegion/condition/list")
    public Result<List<BaseRegion>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<BaseRegion> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<BaseRegion> list = baseRegionService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建地区表
     *
     * @param baseRegion 地区表载体
     * @return 地区表 主键
     */
    @PostMapping("/v1/baseRegion")
    public Result<String> create(@RequestBody BaseRegion baseRegion){
        return Result.SUCCESS(baseRegionService.insert(baseRegion));
    }


    /**
     * 根据主键更新地区表
     *
     * @param id 地区表ID
     * @param baseRegion 地区表载体
     * @return Id
     */
    @PutMapping("/v1/baseRegion/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody BaseRegion baseRegion){
        return Result.SUCCESS(baseRegionService.update(id, baseRegion));
    }


    /**
     * 根据主键删除地区表
     *
     * @param id 地区表ID
     * @return
     */
    @DeleteMapping("/v1/baseRegion/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        baseRegionService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除地区表
     *
     * @param ids 地区表id集合
     * @return
     */
    @DeleteMapping("/v1/baseRegion/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        baseRegionService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}