package com.wlos.app.api;

import com.wlos.app.model.BaseOpenIssue;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.BaseOpenIssueService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 开放问题表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class BaseOpenIssueController {

    @Autowired
    private BaseOpenIssueService baseOpenIssueService;


    /**
     * 根据id查询开放问题表
     *
     * @param id 开放问题表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseOpenIssue/{id}")
    public Result<BaseOpenIssue> getById(@PathVariable String id) {
        BaseOpenIssue baseOpenIssue = baseOpenIssueService.queryById(id);
        return Result.SUCCESS(baseOpenIssue);
    }

    /**
     * 根据id列表查询开放问题表
     *
     * @param ids 开放问题表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/baseOpenIssue/batch")
    public Result<List<BaseOpenIssue>> getByIds(@RequestBody List<String> ids) {
        List<BaseOpenIssue> baseOpenIssueList = baseOpenIssueService.listByIds(ids);
        return Result.SUCCESS(baseOpenIssueList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/baseOpenIssue/merge")
    public Result<Set<BaseOpenIssue>> merge(@RequestBody List<List<BaseOpenIssue>> list) {
        Set<BaseOpenIssue> newMergedList = new HashSet<>();

        for (List<BaseOpenIssue> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/baseOpenIssue/extract")
    public Result<List<String>> extract(@RequestBody List<BaseOpenIssue> list) {
        List<String> idList = list.stream().map(BaseOpenIssue::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询开放问题表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseOpenIssue/one")
    public Result<BaseOpenIssue> one(BaseOpenIssue condition) {
        BaseOpenIssue baseOpenIssue = baseOpenIssueService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(baseOpenIssue);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/baseOpenIssue/count")
    public Result<Long> count(BaseOpenIssue condition) {
        long count = baseOpenIssueService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/baseOpenIssue/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, BaseOpenIssue condition) {
        QueryWrapper<BaseOpenIssue> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = baseOpenIssueService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/baseOpenIssue/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<BaseOpenIssue> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = baseOpenIssueService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 开放问题表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseOpenIssue/page")
    public Result<Page<BaseOpenIssue>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            BaseOpenIssue condition) {
        Page<BaseOpenIssue> pageResult = baseOpenIssueService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 开放问题表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseOpenIssue/list")
    public Result<List<BaseOpenIssue>> list(BaseOpenIssue condition) {
        List<BaseOpenIssue> listResult = baseOpenIssueService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/baseOpenIssue/condition/list")
    public Result<List<BaseOpenIssue>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<BaseOpenIssue> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<BaseOpenIssue> list = baseOpenIssueService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建开放问题表
     *
     * @param baseOpenIssue 开放问题表载体
     * @return 开放问题表 主键
     */
    @PostMapping("/v1/baseOpenIssue")
    public Result<String> create(@RequestBody BaseOpenIssue baseOpenIssue){
        return Result.SUCCESS(baseOpenIssueService.insert(baseOpenIssue));
    }


    /**
     * 根据主键更新开放问题表
     *
     * @param id 开放问题表ID
     * @param baseOpenIssue 开放问题表载体
     * @return Id
     */
    @PutMapping("/v1/baseOpenIssue/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody BaseOpenIssue baseOpenIssue){
        return Result.SUCCESS(baseOpenIssueService.update(id, baseOpenIssue));
    }


    /**
     * 根据主键删除开放问题表
     *
     * @param id 开放问题表ID
     * @return
     */
    @DeleteMapping("/v1/baseOpenIssue/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        baseOpenIssueService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除开放问题表
     *
     * @param ids 开放问题表id集合
     * @return
     */
    @DeleteMapping("/v1/baseOpenIssue/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        baseOpenIssueService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}