package com.wlos.app.api;

import com.wlos.app.model.BaseFixedIssue;
import com.wlos.app.utils.*;
import com.wlos.app.da.service.BaseFixedIssueService;
import com.wlos.app.model.vo.OperateCondition;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.apache.commons.lang3.ObjectUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;

import java.util.*;


/**
 * 固定问题表服务
 *
 * @author fos
 * @date Aug 30, 2024, 5:03:26 PM
 * @since v0.1
 */
@RestController
@RequestMapping(Constants.REQUEST_MAPPING_PREFIX)
public class BaseFixedIssueController {

    @Autowired
    private BaseFixedIssueService baseFixedIssueService;


    /**
     * 根据id查询固定问题表
     *
     * @param id 固定问题表Id
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseFixedIssue/{id}")
    public Result<BaseFixedIssue> getById(@PathVariable String id) {
        BaseFixedIssue baseFixedIssue = baseFixedIssueService.queryById(id);
        return Result.SUCCESS(baseFixedIssue);
    }

    /**
     * 根据id列表查询固定问题表
     *
     * @param ids 固定问题表Ids
     * @return 列表数据集 
     */
    @PostMapping("/v1/baseFixedIssue/batch")
    public Result<List<BaseFixedIssue>> getByIds(@RequestBody List<String> ids) {
        List<BaseFixedIssue> baseFixedIssueList = baseFixedIssueService.listByIds(ids);
        return Result.SUCCESS(baseFixedIssueList);
    }


    /**
     * 多组集合数据去重合并
     *
     * @param list 需要合并list集合数据 
     * @return 合并后列表数据集 
     */
    @PostMapping("/v1/baseFixedIssue/merge")
    public Result<Set<BaseFixedIssue>> merge(@RequestBody List<List<BaseFixedIssue>> list) {
        Set<BaseFixedIssue> newMergedList = new HashSet<>();

        for (List<BaseFixedIssue> item : list) {
            newMergedList.addAll(item);
        }

        return Result.SUCCESS(newMergedList);
    }


    /**
     * 获取集合列表的主键集合
     *
     * @param list 数据
     * @return 主键集合 
     */
    @PostMapping("/v1/baseFixedIssue/extract")
    public Result<List<String>> extract(@RequestBody List<BaseFixedIssue> list) {
        List<String> idList = list.stream().map(BaseFixedIssue::getId).toList();

        return Result.SUCCESS(idList);
    }


    /**
     * 根据条件查询固定问题表
     *
     * @param condition 条件
     * @return 单条数据结果
     */
    @GetMapping("/v1/baseFixedIssue/one")
    public Result<BaseFixedIssue> one(BaseFixedIssue condition) {
        BaseFixedIssue baseFixedIssue = baseFixedIssueService.getOne(new QueryWrapper<>(condition));
        return Result.SUCCESS(baseFixedIssue);
    }


    /**
     * 条件查询总行数count
     *
     * @param condition 条件
     * @return 总行数
     */
    @GetMapping("/v1/baseFixedIssue/count")
    public Result<Long> count(BaseFixedIssue condition) {
        long count = baseFixedIssueService.count(new QueryWrapper<>(condition));
        return Result.SUCCESS(count);
    }


    /**
     * 简单聚合函数查询
     *
     * @param func       函数类型
     * @param funcColumn 函数处理字段
     * @param condition  查询条件
     * @return 统计后数据
     */
    @GetMapping("/v1/baseFixedIssue/func")
    public Result<Map<String, Object>> funcWithSimple(@RequestParam String func, @RequestParam(required = false) String funcColumn, BaseFixedIssue condition) {
        QueryWrapper<BaseFixedIssue> codeTemplateQueryWrapper = FuncUtils.funSimple(func, funcColumn, condition);
        List<Map<String, Object>> maps = baseFixedIssueService.listMaps(codeTemplateQueryWrapper);
        Map<String, Object> result = null;
        if (ObjectUtils.isNotEmpty(maps)) {
            result = maps.get(0);
        }
        return Result.SUCCESS(result);
    }


    /**
     * 带分组条件聚合函数查询
     *
     * @param operateCondition 函数参数
     * @return 分组统计后数据
     */
    @PostMapping("/v1/baseFixedIssue/funcByGroup")
    public Result<List<Map<String, Object>>> countByGroup(@RequestBody OperateCondition operateCondition) {
        QueryWrapper<BaseFixedIssue> codeTemplateQueryWrapper = FuncUtils.funcByGroup(operateCondition);
        List<Map<String, Object>> maps = baseFixedIssueService.listMaps(codeTemplateQueryWrapper);
        return Result.SUCCESS(maps);
    }




    /**
     * 列表分页条件查询 固定问题表
     * @param pageNum  页码
     * @param pageSize 页数
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseFixedIssue/page")
    public Result<Page<BaseFixedIssue>> page(@RequestParam(required = false) Long pageNum,
                                            @RequestParam(required = false) Long pageSize,
                                            BaseFixedIssue condition) {
        Page<BaseFixedIssue> pageResult = baseFixedIssueService.page(new Page<>(pageNum, pageSize), condition);                                           
        return Result.SUCCESS(pageResult);
    }


    /**
     * 列表不分页条件查询 固定问题表
     * @param condition 条件
     * @return 列表数据
     */
    @GetMapping("/v1/baseFixedIssue/list")
    public Result<List<BaseFixedIssue>> list(BaseFixedIssue condition) {
        List<BaseFixedIssue> listResult = baseFixedIssueService.list(new QueryWrapper<>(condition));                                           
        return Result.SUCCESS(listResult);
    }


    /**
     * 列表不分页复杂条件查询 {cnName}
     * @param condition 条件
     * @return 列表数据
     */
    @PostMapping("/v1/baseFixedIssue/condition/list")
    public Result<List<BaseFixedIssue>> list(@RequestBody OperateCondition condition) {
        QueryWrapper<BaseFixedIssue> queryWrapper = new QueryWrapper<>();
        FuncUtils.buildWhere(condition.getWhereList(),queryWrapper);
        List<BaseFixedIssue> list = baseFixedIssueService.list(queryWrapper);
        return Result.SUCCESS(list);
    }




    /**
     * 创建固定问题表
     *
     * @param baseFixedIssue 固定问题表载体
     * @return 固定问题表 主键
     */
    @PostMapping("/v1/baseFixedIssue")
    public Result<String> create(@RequestBody BaseFixedIssue baseFixedIssue){
        return Result.SUCCESS(baseFixedIssueService.insert(baseFixedIssue));
    }


    /**
     * 根据主键更新固定问题表
     *
     * @param id 固定问题表ID
     * @param baseFixedIssue 固定问题表载体
     * @return Id
     */
    @PutMapping("/v1/baseFixedIssue/{id}")
    public Result<Integer> update(@PathVariable String id, @RequestBody BaseFixedIssue baseFixedIssue){
        return Result.SUCCESS(baseFixedIssueService.update(id, baseFixedIssue));
    }


    /**
     * 根据主键删除固定问题表
     *
     * @param id 固定问题表ID
     * @return
     */
    @DeleteMapping("/v1/baseFixedIssue/{id}")
    public Result<Void> deleteById(@PathVariable String id){
        baseFixedIssueService.deleteById(id);
        return Result.SUCCESS();
    }


    /**
     * 根据主键集合批量删除固定问题表
     *
     * @param ids 固定问题表id集合
     * @return
     */
    @DeleteMapping("/v1/baseFixedIssue/batch")
    public Result<Void> deleteByIds(@RequestBody List<String> ids){
        baseFixedIssueService.deleteByIds(ids);
        return Result.SUCCESS();
    }


}