Object.defineProperty(exports, "__esModule", { value: true });
exports.createSourceFile = void 0;
const reactivity_1 = require("@vue/reactivity");
const sourceMaps_1 = require("./utils/sourceMaps");
const code_gen_1 = require("@volar/code-gen");
;
function createSourceFile(fileName, scriptSnapshot, ts, plugins) {
    // refs
    const snapshot = (0, reactivity_1.shallowRef)(scriptSnapshot);
    const fileContent = (0, reactivity_1.computed)(() => snapshot.value.getText(0, snapshot.value.getLength()));
    const sfc = (0, reactivity_1.reactive)({
        template: null,
        script: null,
        scriptSetup: null,
        styles: [],
        customBlocks: [],
        templateAst: (0, reactivity_1.computed)(() => {
            var _a;
            return (_a = compiledSFCTemplate.value) === null || _a === void 0 ? void 0 : _a.ast;
        }),
        scriptAst: (0, reactivity_1.computed)(() => {
            if (sfc.script) {
                return ts.createSourceFile(fileName + '.' + sfc.script.lang, sfc.script.content, ts.ScriptTarget.Latest);
            }
        }),
        scriptSetupAst: (0, reactivity_1.computed)(() => {
            if (sfc.scriptSetup) {
                return ts.createSourceFile(fileName + '.' + sfc.scriptSetup.lang, sfc.scriptSetup.content, ts.ScriptTarget.Latest);
            }
        }),
    }) /* avoid Sfc unwrap in .d.ts by reactive */;
    // cache
    let parsedSfcCache;
    let compiledSFCTemplateCache;
    // computeds
    const parsedSfc = (0, reactivity_1.computed)(() => {
        var _a;
        // incremental update
        if (parsedSfcCache === null || parsedSfcCache === void 0 ? void 0 : parsedSfcCache.plugin.updateSFC) {
            const change = snapshot.value.getChangeRange(parsedSfcCache.snapshot);
            if (change) {
                const newSfc = parsedSfcCache.plugin.updateSFC(parsedSfcCache.sfc, {
                    start: change.span.start,
                    end: change.span.start + change.span.length,
                    newText: snapshot.value.getText(change.span.start, change.span.start + change.newLength),
                });
                if (newSfc) {
                    parsedSfcCache.snapshot = snapshot.value;
                    parsedSfcCache.sfc = newSfc;
                    return newSfc;
                }
            }
        }
        for (const plugin of plugins) {
            const sfc = (_a = plugin.parseSFC) === null || _a === void 0 ? void 0 : _a.call(plugin, fileName, fileContent.value);
            if (sfc) {
                if (!sfc.errors.length) {
                    parsedSfcCache = {
                        snapshot: snapshot.value,
                        sfc,
                        plugin,
                    };
                }
                return sfc;
            }
        }
    });
    const compiledSFCTemplate = (0, reactivity_1.computed)(() => {
        var _a;
        if (sfc.template) {
            (0, reactivity_1.pauseTracking)();
            // don't tracking
            const newSnapshot = snapshot.value;
            const templateOffset = sfc.template.startTagEnd;
            (0, reactivity_1.resetTracking)();
            // tracking
            sfc.template.content;
            // incremental update
            if (compiledSFCTemplateCache === null || compiledSFCTemplateCache === void 0 ? void 0 : compiledSFCTemplateCache.plugin.updateSFCTemplate) {
                const change = newSnapshot.getChangeRange(compiledSFCTemplateCache.snapshot);
                if (change) {
                    const newText = newSnapshot.getText(change.span.start, change.span.start + change.newLength);
                    const newResult = compiledSFCTemplateCache.plugin.updateSFCTemplate(compiledSFCTemplateCache.result, {
                        start: change.span.start - templateOffset,
                        end: change.span.start + change.span.length - templateOffset,
                        newText,
                    });
                    if (newResult) {
                        compiledSFCTemplateCache.snapshot = newSnapshot;
                        compiledSFCTemplateCache.result = newResult;
                        return {
                            errors: [],
                            warnings: [],
                            ast: newResult.ast,
                        };
                    }
                }
            }
            for (const plugin of plugins) {
                const errors = [];
                const warnings = [];
                let result;
                try {
                    result = (_a = plugin.compileSFCTemplate) === null || _a === void 0 ? void 0 : _a.call(plugin, sfc.template.lang, sfc.template.content, {
                        onError: (err) => errors.push(err),
                        onWarn: (err) => warnings.push(err),
                        expressionPlugins: ['typescript'],
                    });
                }
                catch (e) {
                    const err = e;
                    errors.push(err);
                }
                if (result || errors.length) {
                    if (result && !errors.length && !warnings.length) {
                        compiledSFCTemplateCache = {
                            snapshot: newSnapshot,
                            result: result,
                            plugin,
                        };
                    }
                    return {
                        errors,
                        warnings,
                        ast: result === null || result === void 0 ? void 0 : result.ast,
                    };
                }
            }
        }
    });
    const pluginEmbeddedFiles = plugins.map(plugin => {
        const embeddedFiles = {};
        const files = (0, reactivity_1.computed)(() => {
            if (plugin.getEmbeddedFileNames) {
                const embeddedFileNames = plugin.getEmbeddedFileNames(fileName, sfc);
                for (const oldFileName of Object.keys(embeddedFiles)) {
                    if (!embeddedFileNames.includes(oldFileName)) {
                        delete embeddedFiles[oldFileName];
                    }
                }
                for (const embeddedFileName of embeddedFileNames) {
                    if (!embeddedFiles[embeddedFileName]) {
                        embeddedFiles[embeddedFileName] = (0, reactivity_1.computed)(() => {
                            const file = {
                                fileName: embeddedFileName,
                                capabilities: {
                                    diagnostics: false,
                                    foldingRanges: false,
                                    formatting: false,
                                    documentSymbol: false,
                                    codeActions: false,
                                    inlayHints: false,
                                },
                                isTsHostFile: false,
                                codeGen: new code_gen_1.CodeGen(),
                                teleportMappings: [],
                            };
                            for (const plugin of plugins) {
                                if (plugin.resolveEmbeddedFile) {
                                    plugin.resolveEmbeddedFile(fileName, sfc, file);
                                }
                            }
                            return file;
                        });
                    }
                }
            }
            return Object.values(embeddedFiles);
        });
        return (0, reactivity_1.computed)(() => {
            const baseOffsetMap = new Map();
            return files.value.map(_file => {
                const file = _file.value;
                const sourceMap = new sourceMaps_1.EmbeddedFileSourceMap();
                for (const mapping of file.codeGen.mappings) {
                    const vueRange = embeddedRangeToVueRange(mapping.data, mapping.sourceRange);
                    let additional;
                    if (mapping.additional) {
                        additional = [];
                        for (const add of mapping.additional) {
                            const addVueRange = embeddedRangeToVueRange(mapping.data, add.sourceRange);
                            additional.push(Object.assign(Object.assign({}, add), { sourceRange: addVueRange }));
                        }
                    }
                    sourceMap.mappings.push(Object.assign(Object.assign({}, mapping), { sourceRange: vueRange, additional }));
                }
                const embedded = {
                    file,
                    sourceMap,
                    teleport: new sourceMaps_1.Teleport(file.teleportMappings),
                };
                return embedded;
            });
            function embeddedRangeToVueRange(data, range) {
                var _a;
                if (data.vueTag) {
                    const key = data.vueTag + '-' + data.vueTagIndex;
                    let baseOffset = baseOffsetMap.get(key);
                    if (baseOffset === undefined) {
                        if (data.vueTag === 'script' && sfc.script) {
                            baseOffset = sfc.script.startTagEnd;
                        }
                        else if (data.vueTag === 'scriptSetup' && sfc.scriptSetup) {
                            baseOffset = sfc.scriptSetup.startTagEnd;
                        }
                        else if (data.vueTag === 'template' && sfc.template) {
                            baseOffset = sfc.template.startTagEnd;
                        }
                        else if (data.vueTag === 'style') {
                            baseOffset = sfc.styles[data.vueTagIndex].startTagEnd;
                        }
                        else if (data.vueTag === 'customBlock') {
                            baseOffset = sfc.customBlocks[data.vueTagIndex].startTagEnd;
                        }
                        if (baseOffset !== undefined) {
                            baseOffsetMap.set(key, baseOffset);
                        }
                    }
                    if (baseOffset !== undefined) {
                        return {
                            start: baseOffset + range.start,
                            end: baseOffset + range.end,
                        };
                    }
                }
                if (data.vueTag === 'scriptSrc' && ((_a = sfc.script) === null || _a === void 0 ? void 0 : _a.src)) {
                    const vueStart = fileContent.value.substring(0, sfc.script.startTagEnd).lastIndexOf(sfc.script.src);
                    const vueEnd = vueStart + sfc.script.src.length;
                    return {
                        start: vueStart - 1,
                        end: vueEnd + 1,
                    };
                }
                return range;
            }
        });
    });
    const allEmbeddeds = (0, reactivity_1.computed)(() => {
        const all = [];
        for (const embeddedFiles of pluginEmbeddedFiles) {
            for (const embedded of embeddedFiles.value) {
                all.push(embedded);
            }
        }
        return all;
    });
    const teleports = (0, reactivity_1.computed)(() => {
        const _all = [];
        for (const embedded of allEmbeddeds.value) {
            if (embedded.teleport) {
                _all.push({
                    file: embedded.file,
                    teleport: embedded.teleport,
                });
            }
        }
        return _all;
    });
    const embeddeds = (0, reactivity_1.computed)(() => {
        const embeddeds = [];
        let remain = [...allEmbeddeds.value];
        while (remain.length) {
            const beforeLength = remain.length;
            consumeRemain();
            if (beforeLength === remain.length) {
                break;
            }
        }
        for (const e of remain) {
            embeddeds.push({
                self: e,
                embeddeds: [],
            });
            if (e.file.parentFileName) {
                console.error('Unable to resolve embedded: ' + e.file.parentFileName + ' -> ' + e.file.fileName);
            }
        }
        return embeddeds;
        function consumeRemain() {
            for (let i = remain.length - 1; i >= 0; i--) {
                const embedded = remain[i];
                if (!embedded.file.parentFileName) {
                    embeddeds.push({
                        self: embedded,
                        embeddeds: [],
                    });
                    remain.splice(i, 1);
                }
                else {
                    const parent = findParentStructure(embedded.file.parentFileName, embeddeds);
                    if (parent) {
                        parent.embeddeds.push({
                            self: embedded,
                            embeddeds: [],
                        });
                        remain.splice(i, 1);
                    }
                }
            }
        }
        function findParentStructure(fileName, strus) {
            var _a;
            for (const stru of strus) {
                if (((_a = stru.self) === null || _a === void 0 ? void 0 : _a.file.fileName) === fileName) {
                    return stru;
                }
                let _stru = findParentStructure(fileName, stru.embeddeds);
                if (_stru) {
                    return _stru;
                }
            }
        }
    });
    update(scriptSnapshot, true);
    return {
        fileName,
        get text() {
            return fileContent.value;
        },
        update,
        get compiledSFCTemplate() {
            return compiledSFCTemplate.value;
        },
        get tsFileName() {
            var _a, _b;
            return (_b = (_a = allEmbeddeds.value.find(e => e.file.fileName.replace(fileName, '').match(/^\.(js|ts)x?$/))) === null || _a === void 0 ? void 0 : _a.file.fileName) !== null && _b !== void 0 ? _b : '';
        },
        get sfc() {
            return sfc;
        },
        get embeddeds() {
            return embeddeds.value;
        },
        get allEmbeddeds() {
            return allEmbeddeds.value;
        },
        get teleports() {
            return teleports.value;
        },
    };
    function update(newScriptSnapshot, init = false) {
        if (newScriptSnapshot === snapshot.value && !init) {
            return;
        }
        snapshot.value = newScriptSnapshot;
        // TODO: wait for https://github.com/vuejs/core/pull/5912
        if (parsedSfc.value) {
            updateTemplate(parsedSfc.value.descriptor.template);
            updateScript(parsedSfc.value.descriptor.script);
            updateScriptSetup(parsedSfc.value.descriptor.scriptSetup);
            updateStyles(parsedSfc.value.descriptor.styles);
            updateCustomBlocks(parsedSfc.value.descriptor.customBlocks);
        }
        else {
            updateTemplate(null);
            updateScript(null);
            updateScriptSetup(null);
            updateStyles([]);
            updateCustomBlocks([]);
        }
        function updateTemplate(block) {
            var _a;
            const newData = block ? {
                tag: 'template',
                start: fileContent.value.substring(0, block.loc.start.offset).lastIndexOf('<'),
                end: block.loc.end.offset + fileContent.value.substring(block.loc.end.offset).indexOf('>') + 1,
                startTagEnd: block.loc.start.offset,
                endTagStart: block.loc.end.offset,
                content: block.content,
                lang: (_a = block.lang) !== null && _a !== void 0 ? _a : 'html',
            } : null;
            if (sfc.template && newData) {
                updateBlock(sfc.template, newData);
            }
            else {
                sfc.template = newData;
            }
        }
        function updateScript(block) {
            var _a;
            const newData = block ? {
                tag: 'script',
                start: fileContent.value.substring(0, block.loc.start.offset).lastIndexOf('<'),
                end: block.loc.end.offset + fileContent.value.substring(block.loc.end.offset).indexOf('>') + 1,
                startTagEnd: block.loc.start.offset,
                endTagStart: block.loc.end.offset,
                content: block.content,
                lang: (_a = block.lang) !== null && _a !== void 0 ? _a : 'js',
                src: block.src,
            } : null;
            if (sfc.script && newData) {
                updateBlock(sfc.script, newData);
            }
            else {
                sfc.script = newData;
            }
        }
        function updateScriptSetup(block) {
            var _a;
            const newData = block ? {
                tag: 'scriptSetup',
                start: fileContent.value.substring(0, block.loc.start.offset).lastIndexOf('<'),
                end: block.loc.end.offset + fileContent.value.substring(block.loc.end.offset).indexOf('>') + 1,
                startTagEnd: block.loc.start.offset,
                endTagStart: block.loc.end.offset,
                content: block.content,
                lang: (_a = block.lang) !== null && _a !== void 0 ? _a : 'js',
            } : null;
            if (sfc.scriptSetup && newData) {
                updateBlock(sfc.scriptSetup, newData);
            }
            else {
                sfc.scriptSetup = newData;
            }
        }
        function updateStyles(blocks) {
            var _a;
            for (let i = 0; i < blocks.length; i++) {
                const block = blocks[i];
                const newData = {
                    tag: 'style',
                    start: fileContent.value.substring(0, block.loc.start.offset).lastIndexOf('<'),
                    end: block.loc.end.offset + fileContent.value.substring(block.loc.end.offset).indexOf('>') + 1,
                    startTagEnd: block.loc.start.offset,
                    endTagStart: block.loc.end.offset,
                    content: block.content,
                    lang: (_a = block.lang) !== null && _a !== void 0 ? _a : 'css',
                    module: typeof block.module === 'string' ? block.module : block.module ? '$style' : undefined,
                    scoped: !!block.scoped,
                };
                if (sfc.styles.length > i) {
                    updateBlock(sfc.styles[i], newData);
                }
                else {
                    sfc.styles.push(newData);
                }
            }
            while (sfc.styles.length > blocks.length) {
                sfc.styles.pop();
            }
        }
        function updateCustomBlocks(blocks) {
            var _a;
            for (let i = 0; i < blocks.length; i++) {
                const block = blocks[i];
                const newData = {
                    tag: 'customBlock',
                    start: fileContent.value.substring(0, block.loc.start.offset).lastIndexOf('<'),
                    end: block.loc.end.offset + fileContent.value.substring(block.loc.end.offset).indexOf('>') + 1,
                    startTagEnd: block.loc.start.offset,
                    endTagStart: block.loc.end.offset,
                    content: block.content,
                    lang: (_a = block.lang) !== null && _a !== void 0 ? _a : 'txt',
                    type: block.type,
                };
                if (sfc.customBlocks.length > i) {
                    updateBlock(sfc.customBlocks[i], newData);
                }
                else {
                    sfc.customBlocks.push(newData);
                }
            }
            while (sfc.customBlocks.length > blocks.length) {
                sfc.customBlocks.pop();
            }
        }
        function updateBlock(oldBlock, newBlock) {
            for (let key in newBlock) {
                oldBlock[key] = newBlock[key];
            }
        }
    }
}
exports.createSourceFile = createSourceFile;
//# sourceMappingURL=sourceFile.js.map