package com.bcxin.ars.util.redis;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.Map;
import java.util.concurrent.TimeUnit;

@Service
public class RedisUtil {

    /**
     *日志
     */
    private Logger log = LoggerFactory.getLogger(RedisUtil.class);

	@Autowired
    protected RedisTemplate<String, String> redisTemplate;

    @Value("${redisPrefixKey}")
    public String REDIS_PREFIX_KEY="ARS:";

    /**
     * 设置redisTemplate
     */
    public void setRedisTemplate(RedisTemplate<String, String> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    /**
     * 获取 RedisSerializer
     */
    protected RedisSerializer<String> getRedisSerializer() {
        return redisTemplate.getStringSerializer();
    }

    /**
     * 保存
     * @param key
     * @param value
     * @return
     */
    public void set(String key, String value) {
        redisTemplate.opsForValue().set(REDIS_PREFIX_KEY + key, value);
    }

    /**
     * 保存
     * @param map
     * @return
             */
    public void multiSet(Map<String, String> map) {
        redisTemplate.opsForValue().multiSet(map);
    }

    /**
     * 保存带过期时间
     * @param key
     * @param value
     * @param timeout
     * @return
     */
    public void set(String key, String value, int timeout) {
        redisTemplate.opsForValue().set(REDIS_PREFIX_KEY + key, value,timeout, TimeUnit.SECONDS);
    }
    public void set(String key, String value ,int timeout, TimeUnit timeUnit) {
        redisTemplate.opsForValue().set(REDIS_PREFIX_KEY + key, value, timeout, timeUnit);
    }

    /**
     * 根据指定key获取值
     * @param key
     * @return
     */
    public String get(String key) {
        //log.info(REDIS_PREFIX_KEY + key);
        return redisTemplate.opsForValue().get(REDIS_PREFIX_KEY + key);
    }


    /**
     * 判断指定key获取是否存在
     * @param key
     * @return
     */
    public boolean hasKey(String key) {
        String value = redisTemplate.opsForValue().get(REDIS_PREFIX_KEY + key);
        if(value!=null && "".equals(value)){
            return true;
        }else{
            return false;
        }
    }

    /**
     * 删除单个Key
     * @param key
     */
    public void delete(String key) {
        redisTemplate.delete(REDIS_PREFIX_KEY + key);
    }

    /**
     *
     * @param key
     * @param length
     * @param timeout(天)
     * @return
     */
    public String createAutoID(String key,int length,long timeout) {
        key = MessageFormat.format("{0}{1}",  REDIS_PREFIX_KEY,key);
        //查询 key 是否存在， 不存在返回 1 ，存在的话则自增加1
        Long autoID = redisTemplate.opsForValue().increment(key, 1);
        redisTemplate.expire(key,timeout,TimeUnit.DAYS);
        return getSequence(autoID,length);
    }

    public Long getCount(String key,long timeout) {
        key = MessageFormat.format("{0}{1}",  REDIS_PREFIX_KEY,key);
        //查询 key 是否存在， 不存在返回 1 ，存在的话则自增加1
        Long autoID = redisTemplate.opsForValue().increment(key, 1);
        redisTemplate.expire(key,timeout,TimeUnit.DAYS);
        return autoID;
    }

    /**
     * 将传入的数 seq 格式化成 length 位，不够前边补 0
     * @param seq
     * @param length
     * @return
     */
    private String getSequence(long seq, int length) {
        String str = String.valueOf(seq);
        int len = str.length();
        if (len >= length) {
            return str;
        }
        int rest = length - len;
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < rest; i++) {
            sb.append('0');
        }
        sb.append(str);
        return sb.toString();
    }
}
