package com.bcxin.ars.util.poi;

import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.springframework.util.StringUtils;

import java.io.IOException;

public class ExcelUtil {

    public static String getCellValue(Cell cell) {
        String cellValue = "";
        DataFormatter formatter = new DataFormatter();
        if (cell != null) {
            switch (cell.getCellType()) {
                case Cell.CELL_TYPE_NUMERIC:
                    if (DateUtil.isCellDateFormatted(cell)) {
                        cellValue = formatter.formatCellValue(cell);
                    } else {
                        double value = cell.getNumericCellValue();
                        int intValue = (int) value;
                        cellValue = value - intValue == 0 ? String.valueOf(intValue) : String.valueOf(value);
                    }
                    break;
                case Cell.CELL_TYPE_STRING:
                    cellValue = cell.getStringCellValue();
                    break;
                case Cell.CELL_TYPE_BOOLEAN:
                    cellValue = String.valueOf(cell.getBooleanCellValue());
                    break;
                case Cell.CELL_TYPE_FORMULA:
                    cellValue = String.valueOf(cell.getCellFormula());
                    break;
                case Cell.CELL_TYPE_BLANK:
                    cellValue = "";
                    break;
                case Cell.CELL_TYPE_ERROR:
                    cellValue = "";
                    break;
                default:
                    cellValue = cell.toString().trim();
                    break;
            }
        }
        return cellValue.trim();
    }

    public static HSSFDataValidation setDataValidation(String strFormula, int firstRow, int endRow, int firstCol, int endCol) {
        // 设置数据有效性加载在哪个单元格上。四个参数分别是：起始行、终止行、起始列、终止列
        CellRangeAddressList regions = new CellRangeAddressList(firstRow, endRow, firstCol, endCol);
        DVConstraint constraint = DVConstraint.createFormulaListConstraint(strFormula);
        HSSFDataValidation dataValidation = new HSSFDataValidation(regions, constraint);
        dataValidation.createErrorBox("Error", "Error");
        dataValidation.createPromptBox("", null);
        return dataValidation;
    }

    public static CellStyle getNecessaryTitleStyle(HSSFWorkbook wb) {
        CellStyle style = wb.createCellStyle();
        HSSFFont font = wb.createFont();
        font.setFontName("微软雅黑");
        font.setFontHeightInPoints((short) 12);
        font.setColor((short) 2);
        style.setFont(font);
        style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND);
        style.setFillForegroundColor(HSSFColor.GREY_25_PERCENT.index);
        style.setAlignment(HSSFCellStyle.ALIGN_CENTER);
        HSSFDataFormat format = wb.createDataFormat();
        style.setDataFormat(format.getFormat("@"));
        return style;
    }

    public static CellStyle getCommonTitleStyle(HSSFWorkbook wb) {
        CellStyle style = wb.createCellStyle();
        HSSFFont font = wb.createFont();
        font.setFontName("微软雅黑");
        font.setFontHeightInPoints((short) 12);
        style.setFont(font);
        style.setFillPattern(HSSFCellStyle.SOLID_FOREGROUND);
        style.setFillForegroundColor(HSSFColor.GREY_25_PERCENT.index);
        style.setAlignment(HSSFCellStyle.ALIGN_CENTER);
        HSSFDataFormat format = wb.createDataFormat();
        style.setDataFormat(format.getFormat("@"));
        return style;
    }

    public static CellStyle getExportStyle(Workbook wb, short size) {
        CellStyle style = wb.createCellStyle();
        Font font = wb.createFont();
        font.setFontName("宋体");
        font.setFontHeightInPoints(size);
        style.setFont(font);
        style.setAlignment(HSSFCellStyle.ALIGN_CENTER);//水平居中
        style.setVerticalAlignment(CellStyle.VERTICAL_CENTER);//垂直居中
        DataFormat format = wb.createDataFormat();
        style.setDataFormat(format.getFormat("@"));
        style.setWrapText(true);//自动换行
        style.setBorderBottom(HSSFCellStyle.BORDER_THIN); //下边框
        style.setBorderLeft(HSSFCellStyle.BORDER_THIN);//左边框
        style.setBorderTop(HSSFCellStyle.BORDER_THIN);//上边框
        style.setBorderRight(HSSFCellStyle.BORDER_THIN);//右边框
        return style;
    }

    /**
     * @param wb
     * @param fontSize 字体大小
     * @param weight   是否加粗
     * @param center   是否水平居中
     * @return
     */
    public static CellStyle getExportHeadStyle(Workbook wb, short fontSize, String weight, String center) {
        CellStyle style = wb.createCellStyle();
        Font font = wb.createFont();
        font.setFontName("宋体");
        //是否加粗
        if (weight.equals("1")) {
            font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);
        }
        //是否居中
        if (center.equals("1")) {
            style.setAlignment(HSSFCellStyle.ALIGN_CENTER);//水平居中

        }
        style.setVerticalAlignment(CellStyle.VERTICAL_CENTER);//垂直居中
        //设置字体大小
        style.setWrapText(true);//自动换行
        font.setFontHeightInPoints(fontSize);
        style.setFont(font);
        DataFormat format = wb.createDataFormat();
        style.setDataFormat(format.getFormat("@"));
        return style;
    }

    /**
     * 创建第i个单元格并设置样式
     *
     * @param rowTitle
     * @param i
     * @param value
     * @param style
     * @return
     */
    public static void setStyleCell(Row rowTitle, int i, String value, CellStyle style) {
        Cell cell = rowTitle.createCell(i);//创建标题单元格
        cell.setCellValue(value);//赋值
        cell.setCellStyle(style);//设置样式
    }

    public static CellStyle getImportStyle(Workbook wb, short fontSize, String fontName) {
        CellStyle cellStyle = wb.createCellStyle();
        Font font = wb.createFont();
        //设置字体
        if (!StringUtils.isEmpty(fontName)) {
            font.setFontName(fontName);
        }
        //设置字体大小
        font.setFontHeightInPoints(fontSize);
        cellStyle.setFont(font);
        cellStyle.setWrapText(true);//自动换行
        cellStyle.setBorderBottom(HSSFCellStyle.BORDER_THIN); //下边框  
        cellStyle.setBorderLeft(HSSFCellStyle.BORDER_THIN);//左边框  
        cellStyle.setBorderTop(HSSFCellStyle.BORDER_THIN);//上边框  
        cellStyle.setBorderRight(HSSFCellStyle.BORDER_THIN);//右边框 
        cellStyle.setAlignment(HSSFCellStyle.ALIGN_CENTER); // 水平居中
        cellStyle.setVerticalAlignment(CellStyle.VERTICAL_CENTER);//垂直居中
        return cellStyle;
    }

    public static CellStyle getContext(Workbook wb, short fontSize, String fontName) {
        CellStyle cellStyle = wb.createCellStyle();
        Font font = wb.createFont();
        //设置字体
        if (!StringUtils.isEmpty(fontName)) {
            font.setFontName(fontName);
        }
        //设置字体大小
        font.setFontHeightInPoints(fontSize);
        cellStyle.setFont(font);
        cellStyle.setWrapText(true);//自动换行
        return cellStyle;
    }

    public static CellStyle getStringStyle(HSSFWorkbook wb) {
        CellStyle style = wb.createCellStyle();
        HSSFDataFormat format = wb.createDataFormat();
        style.setDataFormat(format.getFormat("@"));
        return style;
    }

    /**
     * 合并单元格
     *
     * @param sheetIx  指定 Sheet 页，从 0 开始f
     * @param firstRow 开始行
     * @param lastRow  结束行
     * @param firstCol 开始列
     * @param lastCol  结束列
     */
    public static void region(int sheetIx,
                              int firstRow,
                              int lastRow,
                              int firstCol,
                              int lastCol,
                              Workbook workbook) {
        Sheet sheet = workbook.getSheetAt(sheetIx);
        sheet.addMergedRegion(new CellRangeAddress(firstRow, lastRow, firstCol, lastCol));
    }

    /**
     * 指定行是否为空
     *
     * @param sheetIx  指定 Sheet 页，从 0 开始
     * @param rowIndex 指定开始行，从 0 开始
     * @return true 不为空，false 不行为空
     * @throws IOException
     */
    public boolean isRowNull(int sheetIx, int rowIndex, HSSFWorkbook workbook) throws IOException {
        Sheet sheet = workbook.getSheetAt(sheetIx);
        return sheet.getRow(rowIndex) == null;
    }

    /**
     * 设置row 和 column 位置的单元格值
     *
     * @param cellStyle 指定 Sheet 页，从 0 开始
     * @param rowIndex  指定行，从0开始
     * @param colIndex  指定列，从0开始
     * @param value     值
     * @return
     * @throws IOException
     */
    public static boolean setValueAt(CellStyle cellStyle, int rowIndex, int colIndex, String value, HSSFWorkbook workbook, HSSFRow row) throws IOException {
        HSSFCell col = row.createCell(colIndex);
        col.setCellValue(new HSSFRichTextString(value));
        col.setCellStyle(cellStyle);
        return true;
    }

    public static CellStyle getExampleStyle(HSSFWorkbook wb) {
        CellStyle style = wb.createCellStyle();
        HSSFFont font = wb.createFont();
        font.setFontName("微软雅黑");
        font.setFontHeightInPoints((short) 10);
        style.setFont(font);
        style.setAlignment(HSSFCellStyle.ALIGN_CENTER);
        style.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
        style.setWrapText(true);
        return style;
    }

    public static CellStyle getExample1Style(HSSFWorkbook wb) {
        CellStyle style = wb.createCellStyle();
        HSSFFont font = wb.createFont();
        font.setFontName("微软雅黑");
        font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD);//粗体显示
        font.setFontHeightInPoints((short) 14);
        style.setFont(font);
        style.setAlignment(HSSFCellStyle.ALIGN_LEFT);
        style.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);
        style.setWrapText(true);
        return style;
    }
}
