package com.bcxin.sync.apis.controllers;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.datasource.DataSource;
import cn.myapps.common.model.task.Task;
import cn.myapps.designtime.application.service.ApplicationDesignTimeService;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.datasource.service.DataSourceDesignTimeService;
import cn.myapps.designtime.form.service.FormDesignTimeService;
import cn.myapps.designtime.task.service.TaskDesignTimeService;
import cn.myapps.runtime.dynaform.form.ejb.Form;
import com.bcxin.runtime.domain.enums.BooleanStatus;
import com.bcxin.runtime.domain.metas.commands.RefreshAppCommand;
import com.bcxin.runtime.domain.metas.commands.RefreshTaskCommand;
import com.bcxin.runtime.domain.metas.entities.ApplicationMetaEntity;
import com.bcxin.runtime.domain.metas.entities.enums.FormType;
import com.bcxin.runtime.domain.metas.entities.enums.StartupType;
import com.bcxin.runtime.domain.metas.repositories.ApplicationMetaRepository;
import com.bcxin.runtime.domain.metas.services.ApplicationMetaService;
import com.bcxin.runtime.domain.metas.services.TaskMetaService;
import com.bcxin.runtime.domain.snapshoots.JdbcConnectionSnapshot;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasNoSupportException;
import com.bcxin.sync.apis.responses.AppMetaResponse;
import com.bcxin.sync.apis.responses.DataSourceMetaResponse;
import com.bcxin.sync.apis.responses.FormMetaResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Locale;
import java.util.stream.Collectors;

@RestController("com.bcxin.sync.apis.controllers.MetaController")
@RequestMapping(value = "/v2/sync/metas")
public class MetaController {

    private final ApplicationMetaRepository applicationMetaRepository;
    private final ApplicationMetaService applicationMetaService;
    private final JsonProvider jsonProvider;
    private final TaskMetaService taskMetaService;

    public MetaController(ApplicationMetaRepository applicationMetaRepository,
                          ApplicationMetaService applicationMetaService,
                          JsonProvider jsonProvider, TaskMetaService taskMetaService) {
        this.applicationMetaRepository = applicationMetaRepository;
        this.applicationMetaService = applicationMetaService;
        this.jsonProvider = jsonProvider;
        this.taskMetaService = taskMetaService;
    }

    @PostMapping("/refresh")
    public ResponseEntity refreshMetas() throws Exception {
        this.refreshApp("");

        return this.getMetas();
    }

    @PostMapping("/apps/{appName}/refresh")
    public ResponseEntity refreshAppNameMetas(String appName) throws Exception {
        this.refreshApp(appName);

        return this.getMetas();
    }

    @PostMapping("/tasks/refresh")
    public ResponseEntity refreshTasks() throws Exception {
        Collection<RefreshTaskCommand.Task> tasks = this.buildTasks();
        RefreshTaskCommand taskCommand = RefreshTaskCommand.create(tasks);
        this.taskMetaService.refresh(taskCommand);

        return ResponseEntity.ok("完成任务的刷新");
    }

    @GetMapping
    public ResponseEntity getMetas() {
        Collection<ApplicationMetaEntity> metaEntities = this.applicationMetaRepository.findAll();

        Collection<AppMetaResponse> appMetaResponses =
                metaEntities.stream().map(app -> {
                    Collection<FormMetaResponse> formMetaResponses = new ArrayList<>();
                    if (app.getFormMetas() != null) {
                        formMetaResponses =
                                app.getFormMetas().stream().map(fm -> {
                                    return new FormMetaResponse(fm.getId(), fm.getFormId(), fm.getName(), fm.getTableName());
                                }).collect(Collectors.toList());
                    }

                    Collection<DataSourceMetaResponse> dataSourceMetaResponses = new ArrayList<>();
                    if (app.getDataSourceMetas() != null) {
                        dataSourceMetaResponses =
                                app.getDataSourceMetas().stream().map(ds -> {
                                    return new DataSourceMetaResponse(ds.getId(), ds.getDataSourceId(), ds.getName(),
                                            ds.getUseType(), ds.getConfig(), ds.getDbType(),
                                            ds.getIsSelected() == BooleanStatus.TRUE);
                                }).collect(Collectors.toList());
                    }

                    AppMetaResponse metaResponse = new AppMetaResponse(app.getId(), app.getAppId(),
                            app.getName(), app.getIsOnline() == BooleanStatus.TRUE, app.getNote());
                    metaResponse.setFormMetaResponses(formMetaResponses);
                    metaResponse.setDataSourceMetaResponses(dataSourceMetaResponses);

                    return metaResponse;
                }).collect(Collectors.toList());

        return ResponseEntity.ok(appMetaResponses);
    }

    private void refreshApp(String appName) throws Exception {
        Collection<RefreshAppCommand.Application> commandApps = this.buildApps(appName);
        this.applicationMetaService.refresh(RefreshAppCommand.create(commandApps));
    }

    private Collection<RefreshAppCommand.Application> buildApps(String name) throws Exception {
        ApplicationDesignTimeService appService = DesignTimeServiceManager.applicationDesignTimeService();
        //ApplicationService appService = (ApplicationService) RunTimeServiceFactory.resolve(ApplicationService.class);
        DataPackage<Application> applicationDataPackages = appService.query("", "", 1, Integer.MAX_VALUE);

        //DataSourceService dataSourceService = (DataSourceService) RunTimeServiceFactory.resolve(DataSourceService.class);
        DataSourceDesignTimeService dataSourceService= DesignTimeServiceManager.dataSourceDesignTimeService();
        DataPackage<DataSource> dataSourceDataPackages = dataSourceService.query(null, null, 1, Integer.MAX_VALUE);
        FormDesignTimeService formService = DesignTimeServiceManager.formDesignTimeService();
        //FormService formService = (FormService) RunTimeServiceFactory.resolve(FormService.class);
        DataPackage<Form> formDataPackages = formService.query(null, null, 1, Integer.MAX_VALUE);

        Collection<RefreshAppCommand.Application> applications = applicationDataPackages.datas
                .stream().map(ii -> {
                    boolean isMatched = !StringUtils.hasLength(name) || ii.getName().toLowerCase(Locale.ROOT).contains(name);
                    if (!isMatched) {
                        return null;
                    }

                    RefreshAppCommand.Application app = RefreshAppCommand.Application.create(ii.getId(), ii.getName(), true, ii.getDescription());

                    if (dataSourceDataPackages != null) {
                        Collection<RefreshAppCommand.DataSource> selectedDataSources = dataSourceDataPackages
                                .datas.stream()
                                .filter(ds -> (StringUtils.hasLength(ds.getApplicationid()) && ds.getApplicationid().equals(ii.getId())) ||
                                        (StringUtils.hasLength(ds.getParentId()) && ds.getParentId().equals(ii.getId()))
                                )
                                .map(ds -> {
                                    JdbcConnectionSnapshot snapshot = JdbcConnectionSnapshot.create(
                                            ds.getDriverClass(), ds.getUrl(), ds.getUsername(),
                                            ds.getPassword(), ds.getPoolsize(), ds.getPathSuffix(),
                                            ds.getPath(), ds.getUrlPkgPrefixes(), ds.getTimeout());
                                    String config = jsonProvider.getJson(snapshot);
                                    return RefreshAppCommand.DataSource.create(ds.getId(), ds.getName(),
                                            ds.getUseType(), ds.getDbType(), ds.getDriverClass(), config,
                                            ds.getId().equals(ii.getDatasourceId())
                                    );
                                })
                                .collect(Collectors.toList());

                        app.setDataSources(selectedDataSources);
                    }

                    if (formDataPackages != null) {
                        Collection<RefreshAppCommand.Form> selectedForms = formDataPackages
                                .datas.stream()
                                .filter(fm -> (StringUtils.hasLength(fm.getApplicationid()) && fm.getApplicationid().equals(ii.getId())) ||
                                        (StringUtils.hasLength(fm.getParentId()) && fm.getParentId().equals(ii.getId()))
                                )
                                .map(fm -> RefreshAppCommand.Form.create(fm.getId(), fm.getName(),
                                        fm.getTableMapping().getTableName(),
                                        translate2FormType(fm.getType())))
                                .collect(Collectors.toList());
                        app.setForms(selectedForms);
                    }

                    return app;
                }).filter(ii -> ii != null).collect(Collectors.toList());

        return applications;
    }

    private Collection<RefreshTaskCommand.Task> buildTasks() throws Exception {
        TaskDesignTimeService taskService =  DesignTimeServiceManager.taskDesignTimeService();
        //TaskService taskService = (TaskService) RunTimeServiceFactory.resolve(TaskService.class);
        DataPackage<Task> taskDataPackage = taskService.query(null, null, 0, Integer.MAX_VALUE);

        return taskDataPackage.getDatas().stream().map(tk -> {
            RefreshTaskCommand.TaskScriptConfig config =
                    RefreshTaskCommand.TaskScriptConfig.create(
                            StartupType.ValueOf(tk.getStartupType()),
                            tk.getPeriod(), tk.getTerminateScript(), tk.getTaskScript());
            RefreshTaskCommand.Task task =
                    RefreshTaskCommand.Task.create(tk.getApplicationid(), tk.getId(), tk.getName(),
                            tk.getDescription(), tk.getPath(),
                            config);

            return task;
        }).collect(Collectors.toList());
    }

    private FormType translate2FormType(int formType) {
        switch (formType) {
            case Form.FORM_TYPE_NORMAL:
                return FormType.NORMAL;
            case Form.FORM_TYPE_FRAGMENT:
                return FormType.FRAGMENT;
            case Form.FORM_TYPE_HOMEPAGE:
                return FormType.HOMEPAGE;
            case Form.FORM_TYPE_NORMAL_MAPPING:
                return FormType.MAPPING;
            case Form.FORM_TYPE_SUBFORM:
                return FormType.SUBFORM;
            case Form.FORM_TYPE_SEARCHFORM:
                return FormType.SEARCHFORM;
            case Form.FORM_TYPE_TEMPLATEFORM:
                return FormType.TEMPLATEFORM;
            default:
                throw new SaasNoSupportException(String.format("不支持的FormType(%s)", String.valueOf(formType)));
        }
    }
}
