package com.bcxin.tenant.domain.services.impls;

import cn.hutool.core.collection.CollectionUtil;
import com.bcxin.Infrastructures.UnitWork;
import com.bcxin.api.interfaces.tenants.responses.TenantUserExtensionResponse;
import com.bcxin.tenant.domain.entities.TenantUserExtensionEntity;
import com.bcxin.tenant.domain.repositories.TenantUserExtensionRepository;
import com.bcxin.tenant.domain.services.commands.tenantUsers.TenantUserExtensionCommand;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;

@Slf4j
@Service
public class TenantUserExtensionService {
    private final Logger logger = LoggerFactory.getLogger(TenantUserExtensionService.class);

    private final TenantUserExtensionRepository tenantUserExtensionRepository;
    private final UnitWork unitWork;
    private final String envRegionCode;

    public TenantUserExtensionService(TenantUserExtensionRepository tenantUserExtensionRepository,
                                      UnitWork unitWork,
                                      @Value("${env.region-code:110000}") String envRegionCode) {
        this.tenantUserExtensionRepository = tenantUserExtensionRepository;
        this.unitWork = unitWork;
        this.envRegionCode = envRegionCode;
    }


    /**
     * description：通过用户id，获取拓展信息列表
     * author：linchunpeng
     * date：2025/5/21
     */
    public List<TenantUserExtensionResponse> getExtensionListByTenantUserId(String tenantUserId) {
        List<TenantUserExtensionResponse> responseList = new ArrayList<>();
        if ("140500".equals(envRegionCode)) {
            List<TenantUserExtensionEntity> existList = tenantUserExtensionRepository.findByTenantUserId(tenantUserId);
            if (CollectionUtil.isNotEmpty(existList)) {
                for (TenantUserExtensionEntity entity : existList) {
                    responseList.add(new TenantUserExtensionResponse(entity.getExtensionKey(), entity.getExtensionValue()));
                }
            }
        }
        return responseList;
    }

    /**
     * description：保存用户拓展信息
     * author：linchunpeng
     * date：2025/5/21
     */
    public void saveTenantUserExtension(String tenantUserId, List<TenantUserExtensionCommand> extensionList) {
        try {
            log.info("保存用户拓展信息，参数：tenantUserId：{}， extensionList：{}， ", tenantUserId, extensionList);
            if ("140500".equals(envRegionCode) && CollectionUtil.isNotEmpty(extensionList)) {
                //是晋城的 && 要保存的拓展信息不为空
                this.unitWork.executeTran(() -> {
                    List<TenantUserExtensionEntity> existList = tenantUserExtensionRepository.findByTenantUserId(tenantUserId);
                    for (TenantUserExtensionCommand command : extensionList) {
                        //查看数据库中有没有存在的key
                        Optional<TenantUserExtensionEntity> sameExtension = existList.stream().filter(exist -> exist.getExtensionKey().equals(command.getExtensionKey())).findFirst();
                        TenantUserExtensionEntity entity;
                        if (sameExtension.isPresent()) {
                            //有相同的key，则修改
                            entity = sameExtension.get();
                            entity.setExtensionValue(command.getExtensionValue());
                        } else {
                            //没有相同的key，则新增
                            entity = new TenantUserExtensionEntity();
                            entity.setTenantUserId(tenantUserId);
                            entity.setExtensionKey(command.getExtensionKey());
                            entity.setExtensionValue(command.getExtensionValue());
                            entity.setCreateTime(new Date());
                        }
                        tenantUserExtensionRepository.save(entity);
                    }
                });
            }
        } catch (Exception e) {
            log.error("保存用户拓展信息异常：{}", e.getMessage(), e);
        }
    }

}
