package com.bcxin.tenant.apis.impls;

import com.alibaba.excel.EasyExcel;
import com.alibaba.fastjson.JSONObject;
import com.bcxin.Infrastructures.IdWorker;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.enums.CompanyCertificateType;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.ConfigDictRpcProvider;
import com.bcxin.api.interfaces.commons.responses.CompanyInfoCompleteResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyNameExistResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyPromoterResponse;
import com.bcxin.api.interfaces.commons.responses.CompanyResponse;
import com.bcxin.api.interfaces.tenants.OrganizationRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.OrganizationCriteria;
import com.bcxin.api.interfaces.tenants.requests.organizations.*;
import com.bcxin.api.interfaces.tenants.responses.*;
import com.bcxin.tenant.apis.translates.OrganizationTranslate;
import com.bcxin.tenant.domain.configs.PlatformUploadConfig;
import com.bcxin.tenant.domain.entities.EmployeeEntity;
import com.bcxin.tenant.domain.entities.OrganizationEntity;
import com.bcxin.tenant.domain.entities.TenantOrganizationRequestsEntity;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.readers.dtos.CompanyDto;
import com.bcxin.tenant.domain.readers.dtos.MyRegistrationOrganizationDto;
import com.bcxin.tenant.domain.readers.dtos.OrganizationDto;
import com.bcxin.tenant.domain.repositories.EmployeeRepository;
import com.bcxin.tenant.domain.repositories.OrganizationRepository;
import com.bcxin.tenant.domain.repositories.TenantOrganizationRepository;
import com.bcxin.tenant.domain.repositories.dtos.OrganizationExportDto;
import com.bcxin.tenant.domain.services.OrganizationService;
import com.bcxin.tenant.domain.services.commands.organizations.ApproveCompanyCommand;
import com.bcxin.tenant.domain.services.commands.organizations.CreateMyOrganizationCommand;
import com.bcxin.tenant.domain.services.commands.organizations.RegisterCompanyCommand;
import com.bcxin.tenant.domain.services.commands.organizations.UpdateCompanyCommand;
import com.bcxin.tenant.domain.services.commands.tenantUsers.NotifyNewUserCommand;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.CollectionUtils;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",retries = 0,timeout = 120_000)
public class OrganizationRpcProviderImpl implements OrganizationRpcProvider {
    private final OrganizationService organizationService;
    private final OrganizationRepository organizationRepository;
    private final EmployeeRepository employeeRepository;
    private final OrganizationTranslate organizationTranslate;
    private final TenantDbReader dbReader;
    private final IdWorker idWorker;
    private final TenantOrganizationRepository tenantOrganizationRepository;
    private final ConfigDictRpcProvider configDictRpcProvider;

    public OrganizationRpcProviderImpl(OrganizationService organizationService,
                                       OrganizationRepository organizationRepository,
                                       OrganizationTranslate organizationTranslate,
                                       EmployeeRepository employeeRepository,
                                       TenantDbReader dbReader,
                                       IdWorker idWorker, TenantOrganizationRepository tenantOrganizationRepository,
                                       ConfigDictRpcProvider configDictRpcProvider) {
        this.organizationService = organizationService;
        this.organizationRepository = organizationRepository;
        this.employeeRepository = employeeRepository;
        this.organizationTranslate = organizationTranslate;
        this.dbReader = dbReader;
        this.idWorker = idWorker;
        this.tenantOrganizationRepository = tenantOrganizationRepository;
        this.configDictRpcProvider = configDictRpcProvider;
    }

    @Override
    public void notifyMessage(String id, String eventId) {
        this.organizationService.dispatch(NotifyNewUserCommand.create(id, eventId));
    }

    @Override
    public TenantOrganizationResponse getOrganizationReqewsts(String number) {
        TenantOrganizationRequestsEntity tenantOrganizationRequestsEntity = tenantOrganizationRepository.getByNumber(number);
        if (tenantOrganizationRequestsEntity == null) {
            throw new NotFoundTenantException();
        }
        TenantCompanyResponse tenantCompanyResponse = JSONObject.parseObject(tenantOrganizationRequestsEntity.getContent(), TenantCompanyResponse.class);
        if (tenantCompanyResponse == null) {
            throw new NotFoundTenantException();
        }
        TenantOrganizationResponse tenantOrganizationResponse = new TenantOrganizationResponse();
        tenantOrganizationResponse.setContent(tenantCompanyResponse);
        tenantOrganizationResponse.setIndustryCode(tenantOrganizationRequestsEntity.getIndustryCode());
        tenantOrganizationResponse.setInstitutionalCode(tenantOrganizationRequestsEntity.getInstitutionalCode());
        return tenantOrganizationResponse;
    }

    @Override
    public String register(CompanyRegistrationRequest request) {
        //先查询政务外网审批通过的数据
        TenantOrganizationRequestsEntity tenantOrganizationRequestsEntity = null;
        TenantOrganizationResponse reqewsts = null;
        if (StringUtils.isNotBlank(request.getNumber())) {
            reqewsts = this.getOrganizationReqewsts(request.getNumber());
            if (reqewsts != null && reqewsts.getContent() != null) {
                TenantCompanyResponse content = reqewsts.getContent();
                content.setCompanyRegistrationRequestValue(request);

                tenantOrganizationRequestsEntity = tenantOrganizationRepository.getByNumber(request.getNumber());
            }
        }
        //内保行业，需要判断有没有分享人
        if (request.getIndustryCode().equals("04")) {
            if (StringUtils.isNotBlank(request.getShareOrganizationId()) && StringUtils.isNotBlank(request.getShareEmployeeId())) {
                //需要分享人的组织id 和 分享人的职员id 都不为空，才算
                EmployeeEntity employee = employeeRepository.getByOrganIdAndId(request.getShareOrganizationId(), request.getShareEmployeeId());
                if (employee != null) {
                    //职员存在，才有效
                    request.setShareOrganizationName(employee.getOrganization().getName());
                    request.setShareEmployeeName(employee.getTenantUser().getName());
                } else {
                    //职员不存在，设置无效
                    request.setShareOrganizationId(null);
                    request.setShareEmployeeId(null);
                }
            }
        }
        //大活行业、内保行业-党政机关、群防群治行业、社区主管部门，需要设置一些默认值
        if (request.getIndustryCode().equals("02") || request.getInstitutionalCode().equals("0406") || request.getIndustryCode().equals("06") || request.getIndustryCode().equals("07")) {
            if (request.getCertificateType() == null) {
                request.setCertificateType(CompanyCertificateType.UniformCreditCode);
            }
            if (request.getUnifySocialCreditCode() == null) {
                request.setUnifySocialCreditCode("");
            }
            if (request.getUnifySocialCreditCodeFile() == null) {
                request.setUnifySocialCreditCodeFile("");
            }
        }
        //生成注册数据
        RegisterCompanyCommand command = this.organizationTranslate.translate(request);
        //注册操作
        String comId = this.organizationService.register(command);

        if (tenantOrganizationRequestsEntity != null) {
            tenantOrganizationRequestsEntity.setReferenceId(comId);
            tenantOrganizationRepository.save(tenantOrganizationRequestsEntity);
        }

        return comId;
    }

    @Override
    public void approvedUpdate(CompanyApprovedUpdateRequest request) {
        CompanyUpdateRequest companyUpdateRequest = new CompanyUpdateRequest();
        companyUpdateRequest.setId(request.getId());
        companyUpdateRequest.setName(request.getName());
        companyUpdateRequest.setPlaceOfRegister(request.getPlaceOfRegister());
        companyUpdateRequest.setPlaceOfBusiness(request.getPlaceOfBusiness());
        companyUpdateRequest.setLegalPersonName(request.getLegalPersonName());
        companyUpdateRequest.setLegalPersonCredentialType(request.getLegalPersonCredentialType());
        companyUpdateRequest.setLegalPersonCredentialNumber(request.getLegalPersonCredentialNumber());
        companyUpdateRequest.setRegisteredCapital(request.getRegisteredCapital());
        companyUpdateRequest.setUnifySocialCreditCode(request.getUnifySocialCreditCode());
        companyUpdateRequest.setUnifySocialCreditCodeFile(request.getUnifySocialCreditCodeFile());
        companyUpdateRequest.setServicePermitNumber(request.getServicePermitNumber());
        companyUpdateRequest.setNameOfIssuingAuthority(request.getNameOfIssuingAuthority());
        companyUpdateRequest.setFirstIssueServicePermit(request.getFirstIssueServicePermit());
        companyUpdateRequest.setServicePermitAttachment(request.getServicePermitAttachment());
        companyUpdateRequest.setSecurityApprovalNumber(request.getSecurityApprovalNumber());
        companyUpdateRequest.setServiceScopeList(request.getServiceScopeList());
        companyUpdateRequest.setTrainContent(request.getTrainContent());
        companyUpdateRequest.setPlaceOfSupervise(request.getPlaceOfSupervise() != null ? request.getPlaceOfSupervise().getDistrict() != null ? request.getPlaceOfSupervise().getDistrict().getCode() : null : null);
        companyUpdateRequest.setSuperviseDepartId(request.getSuperviseDepartId());
        companyUpdateRequest.setSuperviseDepartName(request.getSuperviseDepartName());
        companyUpdateRequest.setAdministrator(request.getAdministrator());

        UpdateCompanyCommand command = this.organizationTranslate.translate(companyUpdateRequest);
        this.organizationService.update(command);
    }

    @Override
    public void update(CompanyUpdateRequest request) {
        UpdateCompanyCommand command = this.organizationTranslate.translate(request);
        this.organizationService.update(command);
    }

    /**
     * description：判断企业名称是否存在
     * author：linchunpeng
     * date：2023/12/7
     */
    @Override
    public CompanyNameExistResponse nameExist(String name) {
        CompanyNameExistResponse response = new CompanyNameExistResponse();
        boolean existsByName = organizationRepository.existsByName(name);
        response.setExist(existsByName);
        response.setName(name);
        return response;
    }

    /**
     * description: 判断企业信息是否完善
     * author: linchunpeng
     * date:  2023-05-22 11:47
     */
    @Override
    public CompanyInfoCompleteResponse infoComplete(String organizationId) {
        CompanyInfoCompleteResponse response = new CompanyInfoCompleteResponse();
        Optional<OrganizationEntity> optional = organizationRepository.findById(organizationId);
        if (optional.isPresent()) {
            OrganizationEntity organization = optional.get();
            response.setId(organizationId);
            if (organization.infoCompleted()) {
                response.setInfoCompleted(true);
            } else {
                response.setInfoCompleted(false);
                response.setInfoCompletedUrl("/static/portal/vue/index.html#/open?appId=__DXZvc8mnEmDqMARhK7G&actionContent=__AOkTvsah9VLicJhQ1FO&opentarget=detail&linkType=00");
            }
            if (organization.superviseCompleted()) {
                response.setSuperviseCompleted(true);
            } else {
                response.setSuperviseCompleted(false);
                response.setSuperviseCompletedUrl("/static/portal/vue/index.html#/open?appId=__DXZvc8mnEmDqMARhK7G&actionContent=__DRWRVHekZL0pUfzfTwb&opentarget=detail&linkType=00");
            }
        }
        return response;
    }

    @Override
    public List<OrganizationExactSearchResponse> exactSearch(OrganizationRequest request) {
        if (StringUtils.isBlank(request.getName()) && StringUtils.isBlank(request.getUnifySocialCreditCode())) {
            return null;
        }

        List<CompanyDto> list = dbReader.exactSearch(request.getName(), request.getUnifySocialCreditCode());

        if (list != null && list.size() > 0) {
            List<OrganizationExactSearchResponse> resultList = new ArrayList<>();
            for (CompanyDto companyDto : list) {
                resultList.add(OrganizationExactSearchResponse.create(companyDto.getId(), companyDto.getName(), companyDto.getUnifySocialCreditCode()));
            }
            return resultList;
        }

        return null;
    }

    @Override
    public CompanyResponse get(String id) {
        Optional<OrganizationEntity> organizationOptional = this.organizationRepository.findById(id);

        if (!organizationOptional.isPresent()) {
            throw new NotFoundTenantException();
        }

        OrganizationEntity organizationEntity = organizationOptional.get();
        return this.organizationTranslate.translate(organizationEntity, null);
    }


    @Override
    public CompanyPromoterResponse isOrganizationPromoter(String id) {
        Optional<OrganizationEntity> organizationOptional = this.organizationRepository.findById(id);

        if (!organizationOptional.isPresent()) {
            throw new NotFoundTenantException();
        }

        OrganizationEntity organizationEntity = organizationOptional.get();
        Boolean isOrganizationPromoter = dbReader.isOrganizationPromoter(organizationEntity.getId());

        return new CompanyPromoterResponse(organizationEntity.getId(), isOrganizationPromoter);
    }

    @Override
    public CompanyResponse getDetail(String id) {
        Optional<OrganizationEntity> organizationOptional = this.organizationRepository.findById(id);

        if (!organizationOptional.isPresent()) {
            throw new NotFoundTenantException();
        }

        Collection<EmployeeEntity> employeeEntities = employeeRepository.getRegisterEmployeeByOrganId(id);

        CompanyResponse.ContactRequest admin = null;
        for (EmployeeEntity employeeEntity : employeeEntities) {
            if ("系统管理员".equals(employeeEntity.getPosition())) {
                admin = new CompanyResponse.ContactRequest(employeeEntity.getTenantUser().getName(),
                        employeeEntity.getTenantUser().getTelephone(),
                        employeeEntity.getTenantUser().getSelectedCredential().getCredentialType().ordinal(),
                        employeeEntity.getTenantUser().getSelectedCredential().getNumber(), null);
            }
        }
        OrganizationEntity organizationEntity = organizationOptional.get();
        CompanyResponse response = this.organizationTranslate.translate(organizationOptional.get(), admin);
        return response;
    }

    /**
     * 获取扩展后的行业代码列表（当搜索"其他行业"时，包含03、08和未来添加的行业）
     */
    private Set<String> getExpandedIndustryCodes(String industryCode) {
        if (!"03".equals(industryCode)) {
            return null;
        }
        
        // 获取所有行业类型
        Map<String, List<Map<String, String>>> registerDict = configDictRpcProvider.findRegisterDict();
        List<Map<String, String>> industryTypeList = registerDict.get("industryType");
        
        // 提取所有行业代码
        Set<String> allIndustryCodes = new HashSet<>();
        if (industryTypeList != null) {
            for (Map<String, String> industryType : industryTypeList) {
                String code = industryType.get("value");
                if (code != null) {
                    allIndustryCodes.add(code);
                }
            }
        }
        
        // 筛选出"未来添加的行业"（排除01-07，保留03、08及其他）
        Set<String> futureIndustryCodes = new HashSet<>();
        futureIndustryCodes.add("03"); // 其他行业
        futureIndustryCodes.add("08"); // 游泳培训行业
        
        // 添加所有不在01-07范围内的行业代码
        Set<String> standardIndustryCodes = new HashSet<>(Arrays.asList("01", "02", "04", "05", "06", "07"));
        for (String code : allIndustryCodes) {
            if (!standardIndustryCodes.contains(code)) {
                futureIndustryCodes.add(code);
            }
        }
        
        return futureIndustryCodes;
    }

    @Override
    public String createOrgExcel(com.bcxin.api.interfaces.tenants.criterias.OrganizationCriteria criteria) throws Exception {
        List<OrganizationExportDto> list;
        
        // 如果搜索条件是"其他行业"（03），需要扩展为包含"03"、"08"和未来添加的行业
        Set<String> expandedIndustryCodes = getExpandedIndustryCodes(criteria.getIndustryCode());
        if (expandedIndustryCodes != null && !expandedIndustryCodes.isEmpty()) {
            // 使用扩展后的行业代码列表进行查询
            list = dbReader.searchOrgList(criteria, expandedIndustryCodes);
        } else {
            list = dbReader.searchOrgList(criteria);
        }
        
        String path = PlatformUploadConfig.getDownloadPath() + idWorker.nextId() + "组织数据.xlsx";
        File file = new File(path);
        if (!file.getParentFile().exists()) {
            file.mkdirs();
        }
        //file.createNewFile();
        EasyExcel.write(path, OrganizationExportDto.class).sheet().doWrite(list);
        return path;
    }

    @Override
    public Collection<OrganizationAppGetResponse> getOrganApps(String id) {
        return null;
    }

    @Override
    public Collection<String> getApps(String id) {
        Collection<String> appIds = this.dbReader.getMyOrganApps(id);

        return appIds;
    }

    @Override
    public void approve(String id, ApproveCompanyRequest request) {
        this.organizationService.dispatch(ApproveCompanyCommand.create(id, request.isApproved(), request.getNote()));
    }

    @Override
    public Pageable<OrganizationResponse> search(OrganizationCriteria criteria) {
        com.bcxin.tenant.domain.readers.criterias.OrganizationCriteria domainCriteria;
        
        // 如果搜索条件是"其他行业"（03），需要扩展为包含"03"、"08"和未来添加的行业
        Set<String> expandedIndustryCodes = getExpandedIndustryCodes(criteria.getIndustryCode());
        if (expandedIndustryCodes != null && !expandedIndustryCodes.isEmpty()) {
            // 使用多行业代码查询
            domainCriteria = com.bcxin.tenant.domain.readers.criterias.OrganizationCriteria.createWithIndustryCodes(
                    criteria.getPlaceOfSupervise() == null ? null : criteria.getPlaceOfSupervise().getDistrict() == null ? null : criteria.getPlaceOfSupervise().getDistrict().getCode(),
                    criteria.getName(),
                    new ArrayList<>(expandedIndustryCodes),
                    criteria.getStatuses(),
                    criteria.getStartDate(),
                    criteria.getEndDate(),
                    criteria.getPageIndex(),
                    criteria.getPageSize());
        } else {
            // 使用原来的单个行业代码查询
            domainCriteria = com.bcxin.tenant.domain.readers.criterias.OrganizationCriteria.create(
                    criteria.getPlaceOfSupervise() == null ? null : criteria.getPlaceOfSupervise().getDistrict() == null ? null : criteria.getPlaceOfSupervise().getDistrict().getCode(),
                    criteria.getName(),
                    criteria.getIndustryCode(),
                    criteria.getStatuses(),
                    criteria.getStartDate(),
                    criteria.getEndDate(),
                    criteria.getPageIndex(),
                    criteria.getPageSize());
        }
        
        Pageable<com.bcxin.tenant.domain.readers.dtos.OrganizationDto> organizationDtoPageable =
                this.dbReader.searchOrganizations(domainCriteria);

        Collection<OrganizationDto> organizationDTOs = organizationDtoPageable.getData();
        Collection<String> industryCodes = organizationDTOs.stream()
                .map(ii->ii.getIndustryCode()).collect(Collectors.toSet());

        List<Map<String, String>> industries = new ArrayList<>();
        if(!CollectionUtils.isEmpty(industryCodes)) {
           industries = this.configDictRpcProvider.getIndustryList();
        }

        Collection<String> institutionalCodes = organizationDTOs.stream()
                .map(ii->ii.getInstitutionalCode()).collect(Collectors.toSet());
        List<Map<String, String>> institutionals = new ArrayList<>();
        if(!CollectionUtils.isEmpty(institutionalCodes)) {
            institutionals = this.configDictRpcProvider.getInstitutionalListByIds(institutionalCodes);
        }

        List<Map<String, String>> finalIndustries = industries;
        List<Map<String, String>> finalInstitutionals = institutionals;
        return Pageable.create(
                organizationDtoPageable.getPageIndex(), organizationDtoPageable.getPageSize(), organizationDtoPageable.getTotalCount(),
                organizationDtoPageable.getData().stream().map(ix -> {
                    String placeOfRegister = null;
                    if (ix.getPlaceOfRegister() != null) {
                        placeOfRegister = ix.getPlaceOfRegister().getShortLocation();
                    }

                    String placeOfBusiness = null;
                    if (ix.getPlaceOfBusiness() != null) {
                        placeOfBusiness = ix.getPlaceOfBusiness().getShortLocation();
                    }
                    OrganizationResponse response =
                            OrganizationResponse.create(
                                    ix.getId(),
                                    ix.getCode(),
                                    ix.getName(),
                                    ix.getStatus(),
                                    ix.getIndustryCode(),
                                    finalIndustries.stream().filter(ii->ii.get("id").equals(ix.getIndustryCode())).map(ii->ii.get("name")).findFirst().orElse(null),
                                    ix.getInstitutionalCode(),
                                    finalInstitutionals.stream().filter(ii->ii.get("id").equals(ix.getInstitutionalCode())).map(ii->ii.get("name")).findFirst().orElse(null),
                                    placeOfRegister,
                                    placeOfBusiness,
                                    ix.getCreatedTime(),
                                    ix.getApproveTime(),
                                    ix.getLonLatJson(),
                                    ix.getSuperviseRegionName(),
                                    ix.getSuperviseDepartName()
                            );

                    return response;
                }).collect(Collectors.toList())
        );
    }

    @Override
    public Pageable<CompanyCredentialResponse> getCompanies(OrganizationCriteria criteria) {
        return dbReader.getCompanys(
                com.bcxin.tenant.domain.readers.criterias.OrganizationCriteria.create(
                        criteria.getPlaceOfSupervise() == null ? null : criteria.getPlaceOfSupervise().getDistrict() == null ? null : criteria.getPlaceOfSupervise().getDistrict().getCode(),
                        criteria.getName(),
                        criteria.getIndustryCode(),
                        criteria.getStatuses(),
                        criteria.getStartDate(),
                        criteria.getEndDate(),
                        criteria.getPageIndex(),
                        criteria.getPageSize()));
    }

    @Override
    public void create(CreateOrganizationRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("未授权用户不允许访问");
        }

        this.organizationService.create(CreateMyOrganizationCommand.create(
                request.getIndustryCode(),
                request.getInstitutionalCode(),
                request.getName(),
                request.getUnifySocialCreditCode(),
                request.getUnifySocialCreditCodeFile(),
                userModel.getId(),true
        ));
    }

    @Override
    public Collection<MyRegistrationOrganizationResponse> getMyRegistrationOrganizations(int pageIndex, int pageSize) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("未授权用户不允许访问");
        }

        Collection<MyRegistrationOrganizationDto> dtos =
                this.dbReader.getMyRegistrationsByUserId(userModel.getId(), pageIndex, pageSize);

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

        return dtos.stream().map(ii -> {
            String approvedTime = "";
            if(ii.getApprovedTime()!=null) {
                approvedTime = dateFormat.format(ii.getApprovedTime());
            }

            return
                    MyRegistrationOrganizationResponse.create(
                            ii.getId(), ii.getName(), ii.getStatus(), approvedTime, ii.getApprovedNote(), ii.getInstitutionalCode(),
                            ii.getIndustryCode(),dateFormat.format(ii.getCreatedTime())
                    );
                }
        ).collect(Collectors.toList());
    }
}
