package com.bcxin.tenant.apis.impls;

import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcel;
import com.bcxin.Infrastructures.enums.DepartImPermissionType;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.commons.CommonImportResponse;
import com.bcxin.api.interfaces.tenants.DepartmentRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.DepartmentRequest;
import com.bcxin.api.interfaces.tenants.requests.ImportDepartmentRequest;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchAssignImPermissionRequest;
import com.bcxin.api.interfaces.tenants.requests.employees.BatchMoveDepartmentEmployeeRequest;
import com.bcxin.api.interfaces.tenants.responses.DepartmentGetResponse;
import com.bcxin.api.interfaces.tenants.responses.DepartmentTreeGetResponse;
import com.bcxin.tenant.apis.translates.CommonTranslate;
import com.bcxin.tenant.apis.translates.DepartmentTranslate;
import com.bcxin.tenant.domain.configs.PlatformUploadConfig;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.repositories.DepartmentRepository;
import com.bcxin.tenant.domain.repositories.dtos.DepartmentDto;
import com.bcxin.tenant.domain.services.DepartmentService;
import com.bcxin.tenant.domain.services.commands.*;
import com.bcxin.tenant.domain.services.commands.results.ImportDepartmentCommandResult;
import com.bcxin.tenant.domain.snapshots.DepartmentImportedItemSnapshot;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.StringUtils;

import javax.validation.Valid;
import java.io.File;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 120 *1000,retries = 0)
public class DepartmentRpcProviderImpl implements DepartmentRpcProvider {
    private final DepartmentRepository departmentRepository;
    private final DepartmentService departmentService;
    private final DepartmentTranslate departmentTranslate;
    private final CommonTranslate commonTranslate;
    private final TenantDbReader dbReader;

    public DepartmentRpcProviderImpl(DepartmentRepository departmentRepository,
                                     DepartmentService departmentService,
                                     DepartmentTranslate departmentTranslate,
                                     CommonTranslate commonTranslate,
                                     TenantDbReader dbReader) {
        this.departmentRepository = departmentRepository;
        this.departmentService = departmentService;
        this.departmentTranslate = departmentTranslate;
        this.commonTranslate = commonTranslate;
        this.dbReader = dbReader;
    }

    @Override
    public Collection<DepartmentGetResponse> getDirectlySubDepartments(String organizationId, String parentId) {
        Collection<DepartmentDto> departments = null;

        if (!StringUtils.hasLength(parentId)) {
            departments = this.departmentRepository.getRootDtoByOrganId(organizationId);
        } else {
            departments = this.departmentRepository.getDtoByOrganIdAndParentId(organizationId, parentId);
        }

        return departments.stream().map(ii -> departmentTranslate.translate(ii))
                .collect(Collectors.toList());
    }

    @Override
    public Collection<DepartmentTreeGetResponse> getDepartments(String organizationId) {
        Collection<DepartmentDto> departments = this.dbReader.getManagedDepartmentDtos(organizationId);

        return this.departmentTranslate.translate2TreeGetResponse(departments);
    }

    @Override
    public DepartmentGetResponse getDepartment(String organizationId, String id) {
        DepartmentDto departmentDto = this.departmentRepository.getDtoByOrganIdAndId(organizationId, id);
        if (departmentDto == null) {
            throw new NotFoundTenantException();
        }

        return this.departmentTranslate.translate(departmentDto);
    }

    @Override
    public String create(String organizationId, @Valid DepartmentRequest request) {
        String departId = this.departmentService.create(
                DepartmentCommand.create(organizationId, request.getName(), request.getParentId(),
                        request.getDisplayOrder(), translate2DepartImPermissionType(request.getPermissionType()),
                        request.getAllowedDepartIds(), request.getLeaderEmployeeIdList())
        );

        return departId;
    }

    @Override
    public void update(String organizationId, String id, @Valid DepartmentRequest request) {
        this.departmentService.update(id,
                DepartmentCommand.create(organizationId, request.getName(), request.getParentId(),
                        request.getDisplayOrder(), translate2DepartImPermissionType(request.getPermissionType()),
                        request.getAllowedDepartIds(), request.getLeaderEmployeeIdList())
        );
    }

    @Override
    public void delete(String organizationId, String id) {
        this.departmentService.delete(DeleteDepartmentCommand.create(id, organizationId));
    }

    public void update(String organizationId, String id, int displayOrder) {
        ChangeDepartmentDisplayOrderCommand command =
                ChangeDepartmentDisplayOrderCommand.create(organizationId, id, displayOrder);

        this.departmentService.dispatch(command);
    }

    @Override
    public String exportExcel(String organizationId) {
        Collection<DepartmentDto> list = this.dbReader.getManagedDepartmentDtos(organizationId);
        Collection<DepartmentDto> departList = this.dbReader.getAllManagedDepartmentDtos(organizationId);;
        Map<String, DepartmentDto> departMap = departList.stream().collect(Collectors.toMap(DepartmentDto::getId,Function.identity()));
        for (DepartmentDto departmentDto : list) {
            if(StrUtil.isNotEmpty(departmentDto.getParentId())) {
                departmentDto.setParent(departMap.get(departmentDto.getParentId()).getName());
            }
        }
        String path = PlatformUploadConfig.getDownloadPath()+"部门数据"+ Math.random()+".xlsx";
        File file = new File(path);
        if(!file.getParentFile().exists()){
            file.mkdirs();
        }
        EasyExcel.write(path, DepartmentDto.class).sheet().doWrite(list);
        return path;
    }

    @Override
    public CommonImportResponse batchImport(String organizationId,
                            String path,
                            Collection<ImportDepartmentRequest> requests) {
        AtomicInteger paramIndex = new AtomicInteger();
        Collection<DepartmentImportedItemSnapshot> departmentCommandItems =
                requests.parallelStream().map(ii -> {
                    paramIndex.getAndIncrement();
                    return DepartmentImportedItemSnapshot.create(
                            paramIndex.get(),
                            ii.getName(), ii.getDisplayOrder(), ii.getParentTreeName(),
                            ii.getPermissionType());
                }).collect(Collectors.toList());

        ImportDepartmentCommand command = ImportDepartmentCommand.create(path, organizationId, departmentCommandItems);

        ImportDepartmentCommandResult result = this.departmentService.batchImport(command);

        CommonImportResponse commonImportResponse = CommonImportResponse.create(
                result.getDownloadResourceId(),
                result.getResult().getTotalSuccess() + result.getResult().getTotalFailed(),
                result.getResult().getTotalSuccess(),
                result.getResult().getTotalFailed(), "");

        return commonImportResponse;
    }

    @Override
    public void batchMove(String organizationId, BatchMoveDepartmentEmployeeRequest request) {
        this.departmentService.dispatch(BatchMoveDepartEmployeeCommand.create(organizationId, request.getFromDepartIds(), request.getDestDepartId()));
    }

    @Override
    public void batchImPermissions(String organizationId, BatchAssignImPermissionRequest request) {
        BatchAssignImPermissionCommand command =
                BatchAssignImPermissionCommand.create(
                        organizationId,
                        request.getDepartIds(),
                        translate2DepartImPermissionType(request.getPermissionType()),
                        request.getAllowedDepartIds());

        this.departmentService.dispatch(command);
    }

    @Override
    public void batchDelete(String organizationId, Collection<String> ids) {
        this.departmentService.dispatch(BatchDeleteDepartmentCommand.create(organizationId, ids));
    }

    private DepartImPermissionType translate2DepartImPermissionType(com.bcxin.api.interfaces.enums.DepartImPermissionType permissionType) {
        switch (permissionType) {
            case JustOnDepart:
                return DepartImPermissionType.JustOnDepart;
            case Whole:
                return DepartImPermissionType.Whole;
            case Special:
                return DepartImPermissionType.Special;
            case JustSelf:
                return DepartImPermissionType.JustSelf;
            case OnDepartAndSub:
                return DepartImPermissionType.OnDepartAndSub;
            default:
                throw new NotSupportTenantException(String.format("非法类型(%s)", permissionType));
        }
    }
}
