package com.bcxin.identity.domains.entities;

import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.api.interfaces.enums.UpdatePasswordType;
import com.bcxin.identity.domains.components.PasswordEncoder;
import com.bcxin.identity.domains.enums.AlgorithmType;
import com.bcxin.identity.domains.enums.EventAction;
import com.bcxin.identity.domains.enums.PrincipalStatus;
import com.bcxin.identity.domains.events.IdentityUserDomainEvent;
import com.bcxin.identity.domains.exceptions.IdentityNotMatchedException;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.apache.commons.lang3.StringUtils;

import javax.persistence.*;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Date;

@Getter
@Setter(AccessLevel.PROTECTED)
@Entity
@Table(name = "identity_userpassword",
        uniqueConstraints = @UniqueConstraint(name = "identity_username_password_username",columnNames = "user_name"))
public class UserNamePasswordPrincipalEntity extends PrincipalAbstract implements IAggregate {
    @Column(name = "user_name", nullable = false,length = 50)
    private String userName;
    @Column(name = "password", nullable = false, length = 200)
    private String password;

    @Enumerated(EnumType.ORDINAL)
    @Column(nullable = false,name = "algorithm_type")
    private AlgorithmType algorithmType;


    @Column(name = "update_password_time")
    private Date updatePasswordTime;
    @Column(name = "login_fail_count")
    private Integer loginFailCount;
    @Column(name = "login_fail_time")
    private Date loginFailTime;

    protected UserNamePasswordPrincipalEntity(){}

    public static UserNamePasswordPrincipalEntity create(
            IdentityUserEntity identityUser,
            String id,
            String userName,
            AlgorithmType algorithmType,
            String password) {
        UserNamePasswordPrincipalEntity account =
                new UserNamePasswordPrincipalEntity();
        account.setIdentityUser(identityUser);
        account.setUserName(userName);
        account.setPassword(password);
        account.setId(id);
        account.setAlgorithmType(algorithmType);
        account.setStatus(PrincipalStatus.ACTIVATE);

        return account;
    }

    public void changePassword(PasswordEncoder passwordEncoder,
                               String oldPassword, String newPassword,
                               String confirmPassword) {
        if (!passwordEncoder.isMatched(this.getPassword(), oldPassword)) {
            throw new IdentityNotMatchedException();
        }

        if (!StringUtils.equals(newPassword, confirmPassword)) {
            throw new BadTenantException("新密码与确认密码不一致!");
        }

        this.setAlgorithmType(AlgorithmType.SM2);
        this.setPassword(passwordEncoder.encode(this.getAlgorithmType(), newPassword));

        //修改密码， 清除登录失败计数、记录修改密码时间，当做第一次登录需要修改密码
        this.loginFailClearCount();
        this.setUpdatePasswordTime(new Date());

        this.recordEvent(
                IdentityUserDomainEvent.create(EventAction.CHANGED_PWD,
                        "更改密码", this.getIdentityUser(), this));
    }

    public void changeUserName(String userName) {
        if (!org.springframework.util.StringUtils.hasLength(userName)) {
            throw new ArgumentTenantException("用户名不能为空");
        }

        this.setUserName(userName);
        this.setLastUpdatedTime(Timestamp.from(Instant.now()));
    }

    public void resetPassword(PasswordEncoder passwordEncoder,
                              String newPassword,
                              String confirmPassword,
                              boolean isAdminReset) {
        if (!StringUtils.equals(newPassword, confirmPassword)) {
            throw new BadTenantException("新密码与确认密码不一致!");
        }

        this.setAlgorithmType(AlgorithmType.SM2);
        this.setPassword(passwordEncoder.encode(this.getAlgorithmType(), newPassword));

        //重置密码， 清除登录失败计数、清除修改密码时间，当做第一次登录需要修改密码
        this.loginFailClearCount();
        if (isAdminReset) {
            this.setUpdatePasswordTime(null);
        } else {
            this.setUpdatePasswordTime(new Date());
        }

        this.recordEvent(
                IdentityUserDomainEvent.create(EventAction.CHANGED_PWD,
                        "更改密码", this.getIdentityUser(), this));
    }

    /**
     * description：登录失败增加计数
     * author：linchunpeng
     * date：2025/3/31
     */
    public int loginFailAddCount() {
        Date now = new Date();
        int count = 0;
        if (this.getLoginFailCount() != null
                && this.getLoginFailTime() != null
                && DateUtil.between(this.getLoginFailTime(), now, DateUnit.MINUTE) < 5) {
            //登录失败计数不为空 && 最近登录失败时间不为空 && 最近登录失败时间在5分钟内
            count = this.getLoginFailCount();
        }
        count++;
        this.setLoginFailCount(count);
        this.setLoginFailTime(now);
        return count;
    }

    /**
     * description：清除登录失败计数
     * author：linchunpeng
     * date：2025/3/31
     */
    public void loginFailClearCount() {
        this.setLoginFailCount(0);
    }

    /**
     * description：是否锁定
     * author：linchunpeng
     * date：2025/3/31
     */
    public boolean isLock() {
        return this.getLoginFailCount() != null && this.getLoginFailCount() >= 5
                && this.getLoginFailTime() != null && DateUtil.between(this.getLoginFailTime(), new Date(), DateUnit.MINUTE) < 30;
    }

    /**
     * description：获取锁定剩余时间，分钟
     * author：linchunpeng
     * date：2025/3/31
     */
    public long getLockRemainingTime() {
        if (this.isLock()) {
            return 30 - DateUtil.between(this.getLoginFailTime(), new Date(), DateUnit.MINUTE);
        }
        return 0;
    }

    /**
     * description：获取修改密码类型
     * author：linchunpeng
     * date：2025/3/31
     */
    public UpdatePasswordType getUpdatePasswordType() {
        //修改密码类型，0：无需提示，1：还没修改过密码，2：上次修改密码的时间超过90天
        if (this.getUpdatePasswordTime() == null) {
            return UpdatePasswordType.NO_UPDATE;
        }
        if (DateUtil.between(this.getUpdatePasswordTime(), new Date(), DateUnit.DAY) >= 90) {
            return UpdatePasswordType.EXCEED_TIME;
        }
        return UpdatePasswordType.NO_TIPS;
    }

}
