package com.bcxin.tenant.backend.providers;

import cn.hutool.core.thread.ThreadUtil;
import com.bcxin.Infrastructures.components.CacheProvider;
import com.bcxin.Infrastructures.components.models.DataCacheItem;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.backends.BackendRpfProvider;
import com.bcxin.api.interfaces.backends.requests.EventSubscriberRequest;
import com.bcxin.api.interfaces.backends.requests.LogRecorderRequest;
import com.bcxin.tenant.backend.components.SubscriberEventActionComponent;
import com.bcxin.tenant.backend.constants.CacheNames;
import com.bcxin.tenant.domain.entities.EventSubscriberEntity;
import com.bcxin.tenant.domain.entities.TenantEventEntity;
import com.bcxin.tenant.domain.repositories.EventSubscriberRepository;
import com.bcxin.tenant.domain.repositories.TenantEventRepository;
import com.bcxin.tenant.domain.repositories.TenantUserRepository;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.util.CollectionUtils;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.Timestamp;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.UUID;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",timeout = 30 *1000,retries = 0)
public class BackendRpfProviderImpl implements BackendRpfProvider {

    private final SubscriberEventActionComponent subscriberEventActionComponent;
    private final CacheProvider cacheProvider;
    private final EventSubscriberRepository subscriberRepository;
    private final TenantUserRepository tenantUserRepository;
    private final TenantEventRepository tenantEventRepository;
    private final ThreadPoolTaskExecutor taskExecutor;
    private final DataSource dataSource;

    public BackendRpfProviderImpl(SubscriberEventActionComponent subscriberEventActionComponent,
                                  CacheProvider cacheProvider,
                                  EventSubscriberRepository subscriberRepository,
                                  TenantUserRepository tenantUserRepository,
                                  TenantEventRepository tenantEventRepository,
                                  ThreadPoolTaskExecutor taskExecutor,
                                  DataSource dataSource) {
        this.subscriberEventActionComponent = subscriberEventActionComponent;
        this.cacheProvider = cacheProvider;
        this.subscriberRepository = subscriberRepository;
        this.tenantUserRepository = tenantUserRepository;
        this.tenantEventRepository = tenantEventRepository;
        this.taskExecutor = taskExecutor;
        this.dataSource = dataSource;
    }

    @Override
    public void dispatch(EventSubscriberRequest request) {
        /**
         * 确保推送的时候, 代码已经提交
         */
        Collection<TenantEventEntity> tenantEventEntities = getAndDoCheck(request.getTenantUserId());
        if(CollectionUtils.isEmpty(tenantEventEntities)) {
            return;
        }

        Collection<EventSubscriberEntity> subscribers = getOnlineSubscribers(request.getSelectors());
        subscribers.forEach(subscriber -> {
            String connectedIds = null;
            try {
                Collection<String> tenantEventIds =
                        tenantEventEntities
                                .stream().filter(ii -> ii.getMapKey().equalsIgnoreCase(subscriber.getSelector()))
                                .map(ii -> ii.getId()).collect(Collectors.toList());

                if (!CollectionUtils.isEmpty(tenantEventIds)) {
                    connectedIds = tenantEventIds.stream().collect(Collectors.joining(";"));
                    this.subscriberEventActionComponent.execute(subscriber, tenantEventIds);
                }
            }
            catch (Exception ex) {
                System.err.println(String.format(
                        "BackendRpfProviderImpl.Execute.EventSubscriberRequest.dispatch:name=%s;selector=%s;handler=%s;fetchVersion=%s;connectedIds=%s",
                        subscriber.getName(),
                        subscriber.getSelector(),
                        subscriber.getHandler(),
                        subscriber.getFetchVersion(),
                        connectedIds)
                );

                ex.printStackTrace();

            }
        });
    }

    @Override
    public void dispatch(LogRecorderRequest request) {
        final String InsertSQL="insert into tenant_sys_logs(id,title,category,content,created_time)values(?,?,?,?,?)";
        this.taskExecutor.execute(() -> {
            try (Connection conn = this.dataSource.getConnection()) {
                PreparedStatement statement = conn.prepareStatement(InsertSQL);
                statement.setString(1, UUID.randomUUID().toString());
                statement.setString(2,request.getTitle());
                statement.setString(3,request.getCategory());
                statement.setString(4,request.getContent());
                statement.setTimestamp(5, Timestamp.from(Instant.now()));

                statement.executeUpdate();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        });
    }

    private Collection<EventSubscriberEntity> getOnlineSubscribers(Collection<String> selectors) {
        Collection<EventSubscriberEntity> all =
                (Collection<EventSubscriberEntity>) this.cacheProvider.get(CacheNames.SUBSCRIBER_CACHE_NAMES, () -> {
                    return DataCacheItem.create(this.subscriberRepository.getOnline());
                }).getData();

        return all.stream().filter(ii -> selectors.contains(ii.getSelector())).collect(Collectors.toList());
    }

    private Collection<TenantEventEntity> getAndDoCheck(String tenantUserId) {
        int retryTime = 0;
        Collection<TenantEventEntity> tenantEventEntities = this.tenantEventRepository.getEventsByUserId(tenantUserId);
        Instant nowInstant = Instant.now();
        while (tenantEventEntities.size() == 0) {
            tenantEventEntities = this.tenantEventRepository.getEventsByUserId(tenantUserId);

            if (tenantEventEntities.size() > 0) {
                return tenantEventEntities;
            }

            ThreadUtil.sleep(500);
            retryTime++;
            //10秒之后过期
            if (Instant.now().plus(-60, ChronoUnit.SECONDS).isAfter(nowInstant)) {
                break;
            }

            if (retryTime > 5) {
                break;
            }
        }

        return tenantEventEntities;
    }
}
