package com.bcxin.tenant.apis.impls;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import com.bcxin.Infrastructures.Pageable;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.enums.*;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.utils.JwtUtil;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.identities.requests.GetCredentialRequest;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.criterias.TenantUserRegionCriteria;
import com.bcxin.api.interfaces.tenants.requests.TocWechatCgiConfirmCheckStatusRequest;
import com.bcxin.api.interfaces.tenants.requests.organizations.LocationRequest;
import com.bcxin.api.interfaces.tenants.requests.tenantUsers.*;
import com.bcxin.api.interfaces.tenants.responses.*;
import com.bcxin.tenant.domain.dto.XlcpReportDto;
import com.bcxin.tenant.domain.entities.TenantUserEntity;
import com.bcxin.tenant.domain.entities.valueTypes.ItemValueType;
import com.bcxin.tenant.domain.entities.valueTypes.LocationValueType;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.readers.criterias.UserAppealsCriteria;
import com.bcxin.tenant.domain.readers.dtos.*;
import com.bcxin.tenant.domain.repositories.OrganizationRepository;
import com.bcxin.tenant.domain.repositories.TenantUserRepository;
import com.bcxin.tenant.domain.repositories.TenantUserThirdCheckStatusRepository;
import com.bcxin.tenant.domain.repositories.dtos.OrganizationDto;
import com.bcxin.tenant.domain.services.TenantUserService;
import com.bcxin.tenant.domain.services.TenantUserThirdCheckStatusService;
import com.bcxin.tenant.domain.services.XlcpService;
import com.bcxin.tenant.domain.services.commands.SyncTenantUserLocationCommand;
import com.bcxin.tenant.domain.services.commands.UpdateTenantUserCommand;
import com.bcxin.tenant.domain.services.commands.tenantUsers.*;
import com.bcxin.tenant.domain.services.impls.TenantUserExtensionService;
import org.apache.dubbo.config.annotation.DubboService;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",retries = 0,timeout = 120_000)
public class UserRpcProviderImpl implements UserRpcProvider {
    private final TenantUserService userService;
    private final TenantUserExtensionService tenantUserExtensionService;
    private final TenantUserThirdCheckStatusService userThirdCheckStatusService;
    private final TenantUserRepository userRepository;
    private final TenantUserThirdCheckStatusRepository userThirdCheckStatusRepository;
    private final TenantDbReader dbReader;
    private final XlcpService xlcpService;
    private final OrganizationRepository organizationRepository;

    public UserRpcProviderImpl(TenantUserService userService,
                               TenantUserExtensionService tenantUserExtensionService,
                               XlcpService xlcpService,
                               TenantUserRepository userRepository,
                               TenantDbReader dbReader, JsonProvider jsonProvider,
                               TenantUserThirdCheckStatusService userThirdCheckStatusService,
                               TenantUserThirdCheckStatusRepository userThirdCheckStatusRepository,
                               OrganizationRepository organizationRepository
                               ) {
        this.userService = userService;
        this.tenantUserExtensionService = tenantUserExtensionService;
        this.userRepository = userRepository;
        this.organizationRepository = organizationRepository;
        this.dbReader = dbReader;
        this.userThirdCheckStatusService = userThirdCheckStatusService;
        this.userThirdCheckStatusRepository = userThirdCheckStatusRepository;
        this.xlcpService = xlcpService;
    }

    @Override
    public UserProfileGetResponse get(String id) {
        Optional<TenantUserEntity> tenantUserOptional =
                this.userRepository.findById(id);
        if (!tenantUserOptional.isPresent()) {
            throw new NotFoundTenantException();
        }

        TenantUserEntity tenantUser = tenantUserOptional.get();
        String oneInchColorWhitePhoto = null;
        String certificateImage = tenantUser.getCertificateImage();
        if(!StringUtils.hasLength(certificateImage)){
            oneInchColorWhitePhoto = tenantUser.getOneInchColorWhitePhoto();
        }else{
            oneInchColorWhitePhoto = certificateImage;
        }
        return new UserProfileGetResponse(
                id,
                tenantUser.getName(),
                tenantUser.getTelephone(),
                tenantUser.getLonLatJson(),
                tenantUser.getBirthdate(),
                tenantUser.getCheckedStatus(),
                tenantUser.getAuthenticateStatus(),
                tenantUser.getAuthenticatedResult(),
                tenantUser.getStature(),
                tenantUser.getHeadPhoto(),
                tenantUser.getWechatNicky(),
                tenantUser.getSex(),
                tenantUser.getNation(),
                tenantUser.getDiseasesHistory(), tenantUser.getWorkYear(), tenantUser.getPoliticsStatus(),
                tenantUser.getMilitaryStatus(),
                tenantUser.getEducation(),
                tenantUser.getHouseholdType(),
                tenantUser.getNativePlace(),
                tenantUser.getMaritalStatus(),
                oneInchColorWhitePhoto,
                tenantUser.getTwoInchColorBluePhoto(),
                tenantUser.getCid(),
                tenantUser.getEmergencyContact(),
                tenantUser.getEmergencyPhone(),
                tenantUser.getLicenseLevel(),
                tenantUser.getThirdPartyLoginNo(),
                tenantUser.getImIdentity(),
                tenantUser.getPlaceOfNow() == null ? null:translateLocationItem(tenantUser.getPlaceOfNow()),
                tenantUser.getLastAppealStatus(), tenantUser.getLastAppealResult(), tenantUser.getLastAppealTime(),
                tenantUserExtensionService.getExtensionListByTenantUserId(tenantUser.getId()));
    }

    private LocationRequest translateLocationItem(LocationValueType placeOfNow) {
        return LocationRequest.create(
                translateItemValueItem(placeOfNow.getProvince() == null ? null:placeOfNow.getProvince()),
                translateItemValueItem(placeOfNow.getCity() == null ? null:placeOfNow.getCity()),
                translateItemValueItem(placeOfNow.getDistrict() == null ? null:placeOfNow.getDistrict()),
                placeOfNow.getAddress()
        );
    }

    private LocationRequest.ItemValueRequest translateItemValueItem(ItemValueType itemValueRequest) {
        if(itemValueRequest==null) {
            return null;
        }

        return LocationRequest.ItemValueRequest.create(itemValueRequest.getCode(), itemValueRequest.getName());
    }


    @Override
    public Collection<MyOrganizationProfileGetResponse> getById(String id) {
        Collection<MyOrganizationProfileDto> myOrganizationProfileDtos
                = this.dbReader.getMyOrganizationProfiles(id);

        Collection<MyOrganizationProfileGetResponse> result = myOrganizationProfileDtos.stream().map(ii ->
        {
            return MyOrganizationProfileGetResponse.create(
                    ii.getId(), ii.getName(), ii.getIndustryCode(),
                    ii.getInstitutionalCode(), ii.getEmployeeId(), ii.getTenantUserId(),
                    ii.getEmployeeOccupationType(), ii.getEmployeeStatus(), ii.getEmployeeHiredTime(),
                    ii.getMasterSlaveType(),
                    JwtUtil.getToken(ii.getEmployeeId()),
                    JwtUtil.getToken(OrganizationLevelDto.create(ii.getEmployeeId(),
                            OrganizationLevel.getSelectedOrganizationLevels(ii.getOrganizationLevel()).stream().map(yy-> yy.getValue())
                                    .collect(Collectors.toList()))),
                    ii.isDomainAdmin(),
                    ii.isDepartAdmin(),
                    ii.getAreaCode(),
                    ii.isHiredDatedLimited(),
                    ii.isLeaveDatedLimited(),
                              true,
                    OrganizationLevel.getSelectedOrganizationLevels(ii.getOrganizationLevel())
            );
        }).collect(Collectors.toList());

        return result;
    }

    @Override
    public Collection<MyTeamGetResponse> getMyTeamsByUserId(String id) {
        Collection<MyTeamGetResponse> result = new HashSet<>();
        Collection<MyTeamDto> myTeams = this.dbReader.getMyTeamsByUserId(id);
        if (!CollectionUtils.isEmpty(myTeams)) {
            result = myTeams.stream().sorted(Comparator.comparing(MyTeamDto::getCreatedTime))
                            .map(ix -> {
                                Collection<OrganizationLevel> orgLevels =
                                        OrganizationLevel.getSelectedOrganizationLevels(ix.getOrganizationLevel());

                                return MyTeamGetResponse.create(
                                        ix.getId(), ix.getName(), ix.getIndustryCode(),
                                        ix.getInstitutionalCode(),
                                        OrganizationResourceType.Member,
                                        ix.getMemberId(),
                                        ix.getAreaCode(),
                                        ix.getMemberType(),
                                        orgLevels
                                );
                            }).collect(Collectors.toList());
        }

        return result;
    }

    @Override
    public UserOrganBasicGetResponse getByIdAndOrganId(String organId, String tenantUserId, PersonResourceType resourceType) {
        UserOrganBasicDto userOrganBasicDto = this.dbReader.getUserOrganBasicDto(organId, tenantUserId, resourceType);
        if (userOrganBasicDto == null) {
            return null;
        }

        return UserOrganBasicGetResponse.create(
                userOrganBasicDto.getOrganizationId(),
                userOrganBasicDto.getOrgName(),
                userOrganBasicDto.getEmployeeId(),
                userOrganBasicDto.getName(), userOrganBasicDto.getMasterSlaveType(),
                new HashMap<>(), userOrganBasicDto.isDomainAdmin(),
                userOrganBasicDto.isDepartAdmin(),
                userOrganBasicDto.getAreaCode());
    }

    @Override
    public void update(String id, UpdateTenantUserRequest request) {
        this.userService.dispatch(
                UpdateTenantUserCommand.create(id,
                        request.getHeadPhoto(), request.getSex(),
                        request.getNation(),request.getDiseasesHistory(),request.getWorkYear(), request.getPoliticsStatus(),
                        request.getStature(), request.getMilitaryStatus(),
                        request.getBirthdate(),
                        request.getEducation(),
                        request.getHouseholdType(),
                        request.getNativePlace(),
                        request.getMaritalStatus(),
                        request.getEmail(),
                        request.getEmergencyContact(),
                        request.getEmergencyPhone(),
                        request.getLicenseLevel(),
                        request.getPlaceOfNow()));
    }

    @Override
    public void update(String id, SyncTenantUserLocationRequest request) {
        this.userService.dispatch(SyncTenantUserLocationCommand.create(id, request.getLatitude(), request.getLongitude()));
    }

    @Override
    public void updateDeviceId(String id, String cid) {
        this.userService.dispatch(SyncDeviceCommand.create(id, cid));
    }

    @Override
    public void updateBasic(String id, UpdateBasicTenantUserRequest request) {
        this.userService.dispatch(UpdateBasicTenantUserCommand.create(id, request.getName(), request.getTelephone()));
    }

    @Override
    public void changeCredential(String id,UpdateCredentialImageRequest request) {
        this.userService.dispatch(UpdateCredentialCommand.create(
                id,
                request.getHeadPhoto(),
                request.getFrontPhoto(),
                request.getReversePhoto(),
                request.getAddress(),
                request.getValidDateFrom(),
                request.getValidDateTo()
                )
        );
    }

    @Override
    public void updateCredential(String id, UpdateCredentialRequest request) {
        this.userService.dispatch(UpdateUserCredentialCommand.create(
                id,
                request.getCredentialType(),
                request.getName(),
                request.getHeadPhoto(),
                request.getNumber(),
                request.getValidDateFrom(),
                request.getValidDateTo(),
                request.getFrontPhoto(),
                request.getReversePhoto(),
                request.getAddress(),
                request.isFromMobile(),
                request.getCertificateImage())
        );
    }

    @Override
    public void updatePhoto(String id, UpdateTenantPhotoRequest request) {
        this.userService.dispatch(UpdateUserPhotoCommand.create(id,
                request.getOneInchColorWhitePhoto(), request.getTwoInchColorBluePhoto()));
    }

    /**
     * description：修改昵称/头像
     * author：linchunpeng
     * date：2025/5/20
     */
    @Override
    public void updateNickHeadPhoto(String id, UpdateNickHeadPhotoRequest request) {
        this.userService.dispatch(UpdateUserNickHeadPhotoCommand.create(id,
                request.getNick(), request.getHeadPhoto()));
    }

    @Override
    public void syncWechatLabel(String id, SyncWechatLabelRequest request) {
        this.userService.dispatch(SyncWechatLabelCommand.create(id, request.getOpenId(), request.getNicky()));
    }

    @Override
    public UserDetailGetResponse getDetail(String id) {
        TenantUserDto tenantUser = this.dbReader.getTenantUserById(id);
        if(tenantUser==null) {
            return null;
        }

        String oneInchColorWhitePhoto = null;
        String certificateImage = tenantUser.getCertificateImage();
        if(!StringUtils.hasLength(certificateImage)){
            oneInchColorWhitePhoto = tenantUser.getOneInchColorWhitePhoto();
        }else{
            oneInchColorWhitePhoto = certificateImage;
        }

        return UserDetailGetResponse.create(
                tenantUser.getId(),
                tenantUser.getName(),
                tenantUser.getEmail(),
                tenantUser.getNicky(),
                tenantUser.getTelephone(),

                tenantUser.getLonLatJson(),
                tenantUser.getBirthdate(),
                tenantUser.getCheckedStatus(),
                tenantUser.getAuthenticateStatus(),
                tenantUser.getAuthenticatedResult(),
                tenantUser.getStature(),
                tenantUser.getHeadPhoto(),
                tenantUser.getSex(),
                tenantUser.getNation(),
                tenantUser.getWorkYear(),
                tenantUser.getDiseasesHistory(),
                tenantUser.getPoliticsStatus(),
                tenantUser.getMilitaryStatus(),
                tenantUser.getEducation(),
                tenantUser.getHouseholdType(),
                tenantUser.getNativePlace(),
                tenantUser.getMaritalStatus(),
                oneInchColorWhitePhoto,
                tenantUser.getTwoInchColorBluePhoto(),
                tenantUser.getCid(),
                tenantUser.getEmergencyContact(),
                tenantUser.getEmergencyPhone(),
                tenantUser.getLicenseLevel(),
                tenantUser.getThirdPartyLoginNo(),
                tenantUser.getImIdentity(),
                tenantUser.getPlaceOfNow() == null ? null:translateLocationItem(tenantUser.getPlaceOfNow()),
                UserDetailGetResponse.CredentialInfo.create(
                        tenantUser.getCredentialName(),
                        tenantUser.getCredentialHeadPhoto(),
                        tenantUser.getCredentialType(),
                        tenantUser.getCredentialNumber(),
                        tenantUser.getCredentialValidDateFrom(),
                        tenantUser.getCredentialValidDateTo(),
                        tenantUser.getCredentialFrontPhoto(),
                        tenantUser.getCredentialReversePhoto(),
                        tenantUser.getCredentialAddress()
                ),
                tenantUserExtensionService.getExtensionListByTenantUserId(tenantUser.getId())
        );
    }

    @Override
    public void reRealNameAuthenticate(String id) {
        this.userService.dispatch(RedoAuthorizationCommand.create(id));
    }

    @Override
    public void notifyMessage(String id,String eventId) {
        this.userService.dispatch(NotifyNewUserCommand.create(id,eventId));
    }
    @Override
    public Pageable<CredentialResponse> getCredentials(String id,String organizationId,GetCredentialRequest request){
        return this.dbReader.getCredentials(id,organizationId,request);

    };
    @Override
    public List<XlcpReportResponse> xlcps(String id,String organizationId){
        Collection<XlcpReportDto> list = this.xlcpService.findByEmployeeIdAndProjectName(id, "安保人员心理健康评估量表");
        if (list.isEmpty()) {
            return new ArrayList<>();
        }

        List<XlcpReportResponse> result = new ArrayList<>();
        int index = 1;
        for (XlcpReportDto dto : list) {
            result.add(XlcpReportResponse.create(
                    dto.getNumber(),
                    dto.getProjectName(),
                    dto.getCpDate(),
                    dto.getFileUrl()
            ));
        }
        return result;

    }
    @Override
    public boolean xlcpValidOrNot(String organizationId,String id,String currentUserId){
        //过滤不是北京的或者注册地不存在的不做心理测评
        OrganizationDto dto = organizationRepository.getDtoById(organizationId);
        try {
            if(!dto.getPlaceOfRegister().getProvince().getCode().startsWith("11")){
                return true;
            }
        }catch (Exception e){
            return true;
        }
        int sign = this.xlcpService.isWithinTheDeadline(id);
        if(sign == 1){
            TenantUserDto tenantUser = this.dbReader.getTenantUserById(currentUserId);
            if(UserCheckedStatus.Matched.equals(tenantUser.getCheckedStatus()) && RealNameAuthenticatedStatus.Authenticating.equals(tenantUser.getAuthenticateStatus())){
                return false;
            }
        }
        return sign == 2 ? false : true;
    }

    @Override
    public List<CurrentCredentialResponse> getCredentials(QueryCredentialRequest queryRequest) {
        return this.dbReader.getCredentials(queryRequest);
    }

    @Override
    public List<QualificationCredentialResponse> getQualificationCredentials(QueryCredentialRequest queryRequest) {
        return this.dbReader.getQualificationCredentials(queryRequest);
    }

    @Override
    public String getCidByPhone(String phone) {
        InviteUserDto userDto = this.dbReader.getByPhone(phone);
        if(userDto != null){
            return userDto.getCid();
        }
        return null;
    }

    @Override
    public List<UserRegionGetResponse> getUsersByRegion(TenantUserRegionCriteria criteria) {
        List<UserRegionDto> usersByRegion = this.dbReader.getUsersByRegion(criteria);
        List<UserRegionGetResponse> collect = usersByRegion.stream().map(item -> {
            return new UserRegionGetResponse(item.getName(), item.getNumber(), item.getOneInchColorWhitePhoto());
        }).collect(Collectors.toList());
        return collect;
    }

    @Override
    public void updateUserRealName(List<UpdateTenantUserRealNameRequest> requests) {
        this.dbReader.updateTenantUserRealNameRequest(requests);
    }

    @Override
    public UserCredentialResponse getUserCredential(String id, Integer credentialType, String credentialNumber) {
        UserCredentialDto dto = this.dbReader.queryUserCredential(id,credentialType,credentialNumber);
        return dto != null ? UserCredentialResponse.create(dto.getName(),
                dto.getNumber(),
                dto.getCredentialType().intValue(),
                dto.getFzjgmc(),
                dto.getFzrq(),
                dto.getTelephone(),
                dto.getCheckedStatus(),
                dto.getAuthenticatedStatus(),
                dto.getIdnum(),
                dto.getHeadPhoto(),
                dto.getReversePhoto(),
                dto.getFrontPhoto(),
                dto.getAddress()) : null;
    }

    @Override
    public void updateCredentialByBkt(String id, UpdateCredentialRequest request) {
        this.userThirdCheckStatusService.dispatch(UpdateCredentialCommand.create(
                id,
                request.getHeadPhoto(),
                request.getFrontPhoto(),
                request.getReversePhoto(),
                request.getAddress(),
                request.getValidDateFrom(),
                request.getValidDateTo()
        ));
    }

    @Override
    public boolean getCheckStatusByBkt(String id) {
        String tenantId = userThirdCheckStatusRepository.getById(id, ThirdSystemValue.BKT);

        return StringUtils.hasLength(tenantId);
    }

    @Override
    public void createAuthAppeal(String id, CreateAuthAppealRequest request) {
        this.userService.createAppeal(CreateAuthAppealCommand.create(
                id,
                request.getAppealPhoto(),
                request.getRegionCode()));
    }

    @Override
    public Collection<UserAppealsResponse> getUserAppeals(GetAuthAppealsRequest request) {
        Collection<UserAppealDto> userAppeals = this.dbReader.getUserAppeals(UserAppealsCriteria.create(
                request.getUserId(),
                request.getPageIndex(),
                request.getPageSize()));
        Collection<UserAppealsResponse> result = userAppeals.stream()
                .map(ix -> UserAppealsResponse.create(DateUtil.formatDateTime(ix.getCreatedTime()),
                        ix.getStatus().getTypeName(),
                        ix.getResult(),
                        ix.getLastApproverDepartName(),
                        ix.getLastApproverName(),
                        DateUtil.formatDateTime(ix.getLastApproverTime())))
                .collect(Collectors.toList());
        return result;
    }

    /**
     * description：平台操作-变更用户姓名/手机信息
     * author：linchunpeng
     * date：2024/11/12
     */
    @Override
    public void updateUserBasicByPlatform(PlatformUpdateUserBasicRequest request) {
        this.userService.dispatch(PlatformUpdateUserBasicCommand.create(request));
    }

    /**
     * description：平台操作重新实名认证
     * author：linchunpeng
     * date：2024/11/12
     */
    @Override
    public void resetAuthenticateStatusByPlatform(PlatformResetAuthenticateStatusRequest request) {
        this.userService.dispatch(PlatformResetAuthenticateStatusCommand.create(request));
    }

    /**
     * description：重置密码
     * author：linchunpeng
     * date：2024/11/12
     */
    @Override
    public void resetPasswordByPlatform(PlatformResetPasswordRequest request) {
        this.userService.dispatch(PlatformResetPasswordCommand.create(request));
    }

    @Override
    public TocWechatCgiConfirmCheckStatusResponse confirmCheckStatusForToc(TocWechatCgiConfirmCheckStatusRequest request) {
        TocWechatCgiConfirmCheckStatusCommand.TocWechatCgiConfirmCheckStatusCommandResult result
                = this.userService.dispatch(TocWechatCgiConfirmCheckStatusCommand.create(
                request.getTenantUserId(),
                request.getName(),
                request.getCredentialType(),
                request.getNumber(),
                request.getHeadPhoto(),
                request.getFrontPhoto(),
                request.getReversePhoto(),
                request.getAddress(),
                request.getValidDateFrom(),
                request.getValidDateTo(),
                request.getCertificateImage(),
                request.getPhone()
        ));

        return TocWechatCgiConfirmCheckStatusResponse.create(result.getTenantUserId());
    }

    /**
     * description：保存用户拓展信息
     * author：linchunpeng
     * date：2025/5/22
     */
    @Override
    public void saveTenantUserExtension(String id, TenantUserExtensionListRequest request) {
        List<TenantUserExtensionCommand> extensionList = new ArrayList<>();
        if (CollectionUtil.isNotEmpty(request.getExtensionList())) {
            for (TenantUserExtensionRequest extensionRequest : request.getExtensionList()) {
                extensionList.add(TenantUserExtensionCommand.create(extensionRequest.getExtensionKey(), extensionRequest.getExtensionValue()));
            }
            this.tenantUserExtensionService.saveTenantUserExtension(id, extensionList);
        }
    }
}
