package com.bcxin.tenant.apis.impls;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.enums.ResourceReferenceType;
import com.bcxin.Infrastructures.exceptions.ConflictTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.api.interfaces.ApiConstant;
import com.bcxin.api.interfaces.tenants.ExternalMemberRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.externalMembers.*;
import com.bcxin.api.interfaces.tenants.responses.ExternalMemberJoinRecordResponse;
import com.bcxin.api.interfaces.tenants.responses.ExternalMemberSearchResponse;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.readers.criterias.MyExternalMemberRecordCriteria;
import com.bcxin.tenant.domain.readers.dtos.MyExternalMemberRecordDTO;
import com.bcxin.tenant.domain.repositories.criterias.ExternalMemberSearchCriteria;
import com.bcxin.tenant.domain.repositories.dtos.ExternalMemberDTO;
import com.bcxin.tenant.domain.services.ExternalMemberService;
import com.bcxin.tenant.domain.services.commands.externalMembers.CreateExternalMemberCommand;
import com.bcxin.tenant.domain.services.commands.externalMembers.DeleteExternalMemberCommand;
import com.bcxin.tenant.domain.services.commands.externalMembers.UpdateExternalMemberCommand;
import com.bcxin.tenant.domain.services.commands.externalMembers.UpdateExternalMemberStatusCommand;
import org.apache.dubbo.config.annotation.DubboService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.stream.Collectors;

@DubboService(version = ApiConstant.VERSION,validation = "true",retries = 0)
public class ExternalMemberRpcProviderImpl implements ExternalMemberRpcProvider {
    private static final Logger logger = LoggerFactory.getLogger(ExternalMemberRpcProviderImpl.class);

    private final ExternalMemberService externalMemberService;
    private final TenantDbReader dbReader;

    public ExternalMemberRpcProviderImpl(ExternalMemberService externalMemberService,
                                         TenantDbReader dbReader) {
        this.externalMemberService = externalMemberService;
        this.dbReader = dbReader;
    }

    @Override
    public void create(CreateExternalMemberRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        CreateExternalMemberCommand command = CreateExternalMemberCommand.create(
                ResourceReferenceType.Organization,
                request.getOrganizationId(),
                request.getGroupId(),
                userModel.getId(),
                request.getInviteType(),
                request.getInviteCode(),
                request.getInvitorId()
        );

        this.externalMemberService.create(command);
    }

    @Override
    public void update(UpdateExternalMemberRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        UpdateExternalMemberCommand command = UpdateExternalMemberCommand.create(
                ResourceReferenceType.Organization,
                userModel.getOrganId(),
                request.getIds(),
                request.getGroupIds(),
                userModel.getEmployeeId(),
                userModel.getName()
        );

        this.externalMemberService.update(command);
    }

    @Override
    public void delete(DeleteExternalMemberRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        DeleteExternalMemberCommand command = DeleteExternalMemberCommand.create(
                ResourceReferenceType.Organization,
                userModel.getOrganId(),
                request.getIds(),
                userModel.getEmployeeId(),
                userModel.getName()
        );

        this.externalMemberService.delete(command);
    }

    @Override
    public void updateStatus(UpdateExternalMemberStatusRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException("未授权");
        }

        UpdateExternalMemberStatusCommand command = UpdateExternalMemberStatusCommand.create(
                ResourceReferenceType.Organization,
                userModel.getOrganId(),
                request.getIds(),
                userModel.getEmployeeId(),
                userModel.getName(),
                request.getStatus(),
                request.getNote()
        );

        this.externalMemberService.updateStatus(command);
    }

    @Override
    public Collection<ExternalMemberSearchResponse> search(ExternalMemberSearchRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("未授权用户");
        }

        Collection<ExternalMemberDTO> externalMembers =
                this.dbReader.search(
                        ExternalMemberSearchCriteria.create(
                                userModel.getOrganId(),
                                request.getGroupIds(),
                                request.getStatuses(),
                                request.getKeyword(),
                                request.getPageIndex(),
                                request.getPageSize()
                        ));

        return externalMembers.stream().map(ii -> {
            return ExternalMemberSearchResponse.create(
                    ii.getId(),
                    ii.getTenantUserName(),
                    ii.getTenantUserId(),
                    ii.getTelephone(),
                    ii.getCredentialType(),
                    ii.getCredentialNumber(),
                    ii.getInviteType(),
                    ii.getInviteCode(),
                    ii.getCreatedTime(),
                    ii.getApprovedStatus(),
                    ii.getApprovedNote(),
                    ii.getInviteGroupName(),
                    ii.getPrincipalGroupIds(),
                    ii.getMemberType()
            );
        }).collect(Collectors.toList());
    }

    @Override
    public Collection<ExternalMemberJoinRecordResponse> getMyJoinRecords(ExternalMemberJoinRecordSearchRequest request) {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("未授权");
        }

        Collection<MyExternalMemberRecordDTO> records =
                this.dbReader.getMyExternalMemberRecords(MyExternalMemberRecordCriteria.create(
                        request.getPageIndex(), request.getPageSize(),
                        userModel.getId(),
                        request.getKeyword()
                ));

        return records.stream().map(ii -> {
            return ExternalMemberJoinRecordResponse.create(
                    ii.getId(),
                    ii.getOrgId(),
                    ii.getOrgName(),
                    ii.getApprovedStatus(),
                    ii.getApprovedNote(),
                    ii.getApprovedTime(),
                    ii.getCreatedTime()
            );
        }).collect(Collectors.toList());
    }
}
