package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.jdks.ProjectMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.ProjectMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.ProjectMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.ProjectMetaResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;

import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

/**
 * ProjectMetaController 集成测试
 * <p>
 * 测试覆盖：
 * 1. 创建项目 (POST /meta/projects/)
 * 2. 更新项目 (PUT /meta/projects/{id})
 * 3. 删除项目 (DELETE /meta/projects/{id})
 * 4. 搜索项目 (POST /meta/projects/search)
 * 5. 获取单个项目 (GET /meta/projects/{id})
 * <p>
 * 注意：使用 @SpringBootTest 进行集成测试，会真实操作数据库
 * 使用 @Transactional 确保测试后数据回滚
 */
@SpringBootTest
@AutoConfigureMockMvc
@ActiveProfiles("test")
@Transactional
@DisplayName("ProjectMetaController 集成测试")
class ProjectMetaControllerTest {

    @Autowired
    private MockMvc mockMvc;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private ProjectMetaRpcProvider rpcProvider;

    private ProjectMetaRequest projectRequest;
    private ProjectMetaSearchRequest searchRequest;

    // 为模块测试用例创建的两个项目
    private String projectId1 = "1001";
    private String projectId2 = "1002";

    @BeforeEach
    void setUp() {
        // 初始化测试数据
        projectRequest = new ProjectMetaRequest();
        projectRequest.setName("测试项目");
        projectRequest.setNote("这是一个测试项目");
        projectRequest.setEnabled(true);

        searchRequest = new ProjectMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(10);

        // 为模块测试用例创建两个项目
        createTestProjects();
    }

    /**
     * 创建两个测试项目供模块测试用例使用
     */
    private void createTestProjects() {
        try {
//            判断两个项目是否创建
            ProjectMetaResponse project1 = rpcProvider.getById(projectId1);
            if (Objects.isNull(project1)) {
                // 创建项目1
                ProjectMetaRequest request1 = new ProjectMetaRequest();
                request1.setId(projectId1);
                request1.setName("模块测试项目1");
                request1.setNote("这是为模块测试用例创建的第一个项目");
                request1.setEnabled(true);
                rpcProvider.create(request1);
            }

            ProjectMetaResponse project2 = rpcProvider.getById(projectId2);
            if (Objects.isNull(project2)) {
                // 创建项目2
                ProjectMetaRequest request2 = new ProjectMetaRequest();
                request2.setId(projectId2);
                request2.setName("模块测试项目2");
                request2.setNote("这是为模块测试用例创建的第二个项目");
                request2.setEnabled(true);
                rpcProvider.create(request2);
            }

        } catch (Exception e) {
            // 如果创建失败，记录错误但不影响测试运行
            System.err.println("创建测试项目失败: " + e.getMessage());
        }
    }

    @Test
    @DisplayName("测试创建项目 - 成功")
    void testCreateProject_Success() throws Exception {
        // When & Then - 真实创建项目到数据库
        mockMvc.perform(post("/meta/projects/")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(projectRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据真正保存到数据库 - 通过搜索验证
        ProjectMetaSearchRequest searchRequest = new ProjectMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(100);
        EntityCollection<ProjectMetaResponse> results = rpcProvider.search(searchRequest);

        // 验证至少创建了一个项目
        assertTrue(results.getTotalCount() > 0, "应该至少有一个项目被创建");
    }

    @Test
    @DisplayName("测试创建项目 - 空参数")
    void testCreateProject_EmptyRequest() throws Exception {
        // Given - 创建一个无效的请求（没有必填字段）
        ProjectMetaRequest invalidRequest = new ProjectMetaRequest();

        // When & Then - 可能创建成功（取决于验证逻辑）或失败
        mockMvc.perform(post("/meta/projects/")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(invalidRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest()); // 根据实际验证逻辑调整

        // 验证数据是否被创建（可能为null，取决于验证逻辑）
        // 这个测试主要用于验证接口不会崩溃
    }

    @Test
    @DisplayName("测试更新项目 - 成功")
    void testUpdateProject_Success() throws Exception {
        // Given - 先创建一个项目
        rpcProvider.create(projectRequest);

        // 搜索获取刚创建的项目ID
        ProjectMetaSearchRequest searchReq = new ProjectMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ProjectMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的项目");

        String projectId = ((java.util.List<ProjectMetaResponse>) searchResults.getData()).get(0).getId();

        // 修改请求
        ProjectMetaRequest updateRequest = new ProjectMetaRequest();
        updateRequest.setName("更新后的项目名称");
        updateRequest.setNote("更新后的备注");
        updateRequest.setEnabled(false);

        // When & Then
        mockMvc.perform(put("/meta/projects/{id}", projectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已更新
        ProjectMetaResponse updated = rpcProvider.getById(projectId);
        assertNotNull(updated);
        assertEquals("更新后的项目名称", updated.getName());
        assertEquals("更新后的备注", updated.getNote());
        assertFalse(updated.isEnabled());
    }

    @Test
    @DisplayName("测试更新项目 - 项目不存在")
    void testUpdateProject_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String projectId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(put("/meta/projects/{id}", projectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(projectRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试删除项目 - 成功")
    void testDeleteProject_Success() throws Exception {
        // Given - 先创建一个项目
        rpcProvider.create(projectRequest);

        // 搜索获取刚创建的项目ID
        ProjectMetaSearchRequest searchReq = new ProjectMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ProjectMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的项目");

        String projectId = ((java.util.List<ProjectMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(delete("/meta/projects/{id}", projectId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已删除
        ProjectMetaResponse deleted = rpcProvider.getById(projectId);
        assertNull(deleted, "项目应该已被删除");
    }

    @Test
    @DisplayName("测试删除项目 - 项目不存在")
    void testDeleteProject_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String projectId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(delete("/meta/projects/{id}", projectId))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试搜索项目 - 成功返回列表")
    void testSearchProjects_Success() throws Exception {
        // Given - 先创建两个项目
        ProjectMetaRequest request1 = new ProjectMetaRequest();
        request1.setName("搜索测试项目1");
        request1.setNote("备注1");
        request1.setEnabled(true);
        rpcProvider.create(request1);

        ProjectMetaRequest request2 = new ProjectMetaRequest();
        request2.setName("搜索测试项目2");
        request2.setNote("备注2");
        request2.setEnabled(true);
        rpcProvider.create(request2);

        // When & Then
        mockMvc.perform(post("/meta/projects/search")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.totalCount").value(org.hamcrest.Matchers.greaterThanOrEqualTo(2)))
                .andExpect(jsonPath("$.data.pageSize", is(10)));

        // 验证 RPC 调用返回的结果
        EntityCollection<ProjectMetaResponse> results = rpcProvider.search(searchRequest);
        assertTrue(results.getTotalCount() >= 2, "应该至少有两个项目");
    }

    @Test
    @DisplayName("测试搜索项目 - 分页测试")
    void testSearchProjects_Pagination() throws Exception {
        // Given - 搜索请求
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(5);

        // When & Then
        mockMvc.perform(post("/meta/projects/search")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(5)));

        // 验证分页逻辑
        EntityCollection<ProjectMetaResponse> results = rpcProvider.search(searchRequest);
        assertNotNull(results);
        assertEquals(5, results.getPageSize());
    }

    @Test
    @DisplayName("测试搜索项目 - 不同分页参数")
    void testSearchProjects_DifferentPaginationParams() throws Exception {
        // Given
        searchRequest.setPageIndex(2);
        searchRequest.setPageSize(20);

        // When & Then
        mockMvc.perform(post("/meta/projects/search")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(20)));

        // 验证分页参数正确传递
        EntityCollection<ProjectMetaResponse> results = rpcProvider.search(searchRequest);
        assertEquals(20, results.getPageSize());
    }

    @Test
    @DisplayName("测试根据ID获取项目 - 成功")
    void testGetProjectById_Success() throws Exception {
        // Given - 先创建一个项目
        rpcProvider.create(projectRequest);

        // 搜索获取刚创建的项目ID
        ProjectMetaSearchRequest searchReq = new ProjectMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ProjectMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的项目");

        String projectId = ((java.util.List<ProjectMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(get("/meta/projects/{id}", projectId))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.id").exists())
                .andExpect(jsonPath("$.data.name", is(projectRequest.getName())))
                .andExpect(jsonPath("$.data.note", is(projectRequest.getNote())))
                .andExpect(jsonPath("$.data.enabled", is(true)));

        // 验证 RPC 调用
        ProjectMetaResponse response = rpcProvider.getById(projectId);
        assertNotNull(response);
        assertEquals(projectRequest.getName(), response.getName());
    }

    @Test
    @DisplayName("测试根据ID获取项目 - 项目不存在")
    void testGetProjectById_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String projectId = "999999999999";

        // When & Then
        mockMvc.perform(get("/meta/projects/{id}", projectId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证 RPC 调用返回 null
        ProjectMetaResponse response = rpcProvider.getById(projectId);
        assertNull(response, "不存在的项目应该返回 null");
    }

    @Test
    @DisplayName("测试创建多个项目 - 数据完整性")
    void testCreateMultipleProjects() throws Exception {
        // Given - 创建多个项目
        for (int i = 1; i <= 3; i++) {
            ProjectMetaRequest request = new ProjectMetaRequest();
            request.setName("批量测试项目" + i);
            request.setNote("备注" + i);
            request.setEnabled(true);
            rpcProvider.create(request);
        }

        // When - 搜索所有项目
        EntityCollection<ProjectMetaResponse> results = rpcProvider.search(searchRequest);

        // Then - 验证至少创建了3个项目
        assertTrue(results.getTotalCount() >= 3, "应该至少创建了3个项目");

        // 验证每个项目的数据
        ((java.util.List<ProjectMetaResponse>) results.getData()).forEach(response -> {
            assertNotNull(response.getId());
            assertNotNull(response.getName());
            assertNotNull(response.getCreatedTime());
        });
    }

    @Test
    @DisplayName("测试更新项目 - 完整字段测试")
    void testUpdateProject_AllFields() throws Exception {
        // Given - 先创建一个项目
        rpcProvider.create(projectRequest);

        // 搜索获取刚创建的项目ID
        ProjectMetaSearchRequest searchReq = new ProjectMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ProjectMetaResponse> searchResults = rpcProvider.search(searchReq);
        String projectId = ((java.util.List<ProjectMetaResponse>) searchResults.getData()).get(0).getId();

        ProjectMetaRequest fullRequest = new ProjectMetaRequest();
        fullRequest.setName("完整项目名称");
        fullRequest.setNote("详细的项目描述信息");
        fullRequest.setEnabled(false);

        // When & Then
        mockMvc.perform(put("/meta/projects/{id}", projectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(fullRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证所有字段都已更新
        ProjectMetaResponse updated = rpcProvider.getById(projectId);
        assertNotNull(updated);
        assertEquals("完整项目名称", updated.getName());
        assertEquals("详细的项目描述信息", updated.getNote());
        assertFalse(updated.isEnabled());
    }

    @Test
    @DisplayName("测试模块测试用例项目 - 验证预创建的项目")
    void testModuleTestProjects() throws Exception {
        List<String> ids = List.of(projectId1, projectId2);

        // Given - 搜索所有项目
        ProjectMetaSearchRequest searchReq = new ProjectMetaSearchRequest();
        searchReq.setIds(ids);
        searchReq.setPageIndex(1);
        searchReq.setPageSize(100);

        // When - 执行搜索
        EntityCollection<ProjectMetaResponse> results = rpcProvider.search(searchReq);

        // Then - 验证至少有两个模块测试项目
        assertTrue(results.getTotalCount() >= 2, "应该至少有两个模块测试项目");

        // 验证项目名称包含"模块测试项目"
        boolean foundProject1 = false;
        boolean foundProject2 = false;

        for (ProjectMetaResponse response : (java.util.List<ProjectMetaResponse>) results.getData()) {
            if ("模块测试项目1".equals(response.getName())) {
                foundProject1 = true;
                assertEquals("这是为模块测试用例创建的第一个项目", response.getNote());
                assertTrue(response.isEnabled());
            }
            if ("模块测试项目2".equals(response.getName())) {
                foundProject2 = true;
                assertEquals("这是为模块测试用例创建的第二个项目", response.getNote());
                assertTrue(response.isEnabled());
            }
        }

        assertTrue(foundProject1, "应该找到模块测试项目1");
        assertTrue(foundProject2, "应该找到模块测试项目2");

        // 验证项目ID变量已定义（供其他测试用例使用）
        assertNotNull(projectId1, "项目ID1应该已定义");
        assertNotNull(projectId2, "项目ID2应该已定义");
        assertEquals("1001", projectId1);
        assertEquals("1002", projectId2);
    }
}

