package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.enums.PageType;
import com.bcxin.tenant.bcx.jdks.PageMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.PageMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.PageMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.PageMetaResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.transaction.annotation.Transactional;

import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

/**
 * PageMetaController 集成测试
 * 
 * 测试覆盖：
 * 1. 创建页面 (POST /meta/modules/{moduleId}/pages)
 * 2. 更新页面 (PUT /meta/modules/{moduleId}/pages/{id})
 * 3. 删除页面 (DELETE /meta/modules/{moduleId}/pages/{id})
 * 4. 搜索页面 (POST /meta/modules/{moduleId}/pages/search)
 * 5. 获取单个页面 (GET /meta/modules/{moduleId}/pages/{id})
 * 
 * 注意：使用 @SpringBootTest 进行集成测试，会真实操作数据库
 * 使用 @Transactional 确保测试后数据回滚
 */
@SpringBootTest
@AutoConfigureMockMvc
@ActiveProfiles("test")
@Transactional
@DisplayName("PageMetaController 集成测试")
class PageMetaControllerTest {

    @Autowired
    private MockMvc mockMvc;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private PageMetaRpcProvider rpcProvider;

    private String testProjectId;
    private String testModuleId;
    private PageMetaRequest pageRequest;
    private PageMetaSearchRequest searchRequest;

    @BeforeEach
    void setUp() {
        // 初始化测试数据
        testProjectId = "1001";
        testModuleId = "module001";
        
        pageRequest = new PageMetaRequest();
        // projectId 由系统通过 moduleId 自动获取，无需设置
        pageRequest.setModuleId(testModuleId);
        pageRequest.setName("测试页面");
        pageRequest.setNote("测试页面备注");
        pageRequest.setFormJson("{\"fields\":[{\"name\":\"test\",\"type\":\"text\"}]}");
        pageRequest.setSlug("test-page");
        pageRequest.setPageType(PageType.Form);

        searchRequest = new PageMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(10);
        searchRequest.setProjectId(testProjectId);
        searchRequest.setModuleId(testModuleId);
    }

    @Test
    @DisplayName("测试创建页面 - 成功")
    void testCreatePage_Success() throws Exception {
        // When & Then - 真实创建页面到数据库
        mockMvc.perform(post("/meta/modules/{moduleId}/pages", testModuleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(pageRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据真正保存到数据库 - 通过搜索验证
        PageMetaSearchRequest searchRequest = new PageMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(100);
        searchRequest.setProjectId(testProjectId);
        searchRequest.setModuleId(testModuleId);
        EntityCollection<PageMetaResponse> results = rpcProvider.search(searchRequest);
        
        // 验证至少创建了一个页面
        assertTrue(results.getTotalCount() > 0, "应该至少有一个页面被创建");
    }

    @Test
    @DisplayName("测试创建页面 - 空参数")
    void testCreatePage_EmptyRequest() throws Exception {
        // Given - 创建一个无效的请求（没有必填字段）
        PageMetaRequest invalidRequest = new PageMetaRequest();

        // When & Then - 可能创建成功（取决于验证逻辑）或失败
        mockMvc.perform(post("/meta/modules/{moduleId}/pages", testModuleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(invalidRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest()); // 根据实际验证逻辑调整

        // 验证数据是否被创建（可能为null，取决于验证逻辑）
        // 这个测试主要用于验证接口不会崩溃
    }

    @Test
    @DisplayName("测试更新页面 - 成功")
    void testUpdatePage_Success() throws Exception {
        // Given - 先创建一个页面
        rpcProvider.create(pageRequest);
        
        // 搜索获取刚创建的页面ID
        PageMetaSearchRequest searchReq = new PageMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setProjectId(testProjectId);
        searchReq.setModuleId(testModuleId);
        EntityCollection<PageMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的页面");
        
        String pageId = ((java.util.List<PageMetaResponse>) searchResults.getData()).get(0).getId();
        
        // 修改请求
        PageMetaRequest updateRequest = new PageMetaRequest();
        // projectId 由系统通过 moduleId 自动获取，无需设置
        updateRequest.setModuleId(testModuleId);
        updateRequest.setName("更新后的页面名称");
        updateRequest.setNote("更新后的备注");
        updateRequest.setFormJson("{\"fields\":[{\"name\":\"updated\",\"type\":\"text\"}]}");

        // When & Then
        mockMvc.perform(put("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已更新
        PageMetaResponse updated = rpcProvider.getById(pageId);
        assertNotNull(updated);
        assertEquals("更新后的页面名称", updated.getName());
        assertEquals("更新后的备注", updated.getNote());
        assertEquals(testProjectId, updated.getProjectId());
        assertEquals(testModuleId, updated.getModuleId());
    }

    @Test
    @DisplayName("测试更新页面 - 页面不存在")
    void testUpdatePage_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String pageId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(put("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(pageRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试删除页面 - 成功")
    void testDeletePage_Success() throws Exception {
        // Given - 先创建一个页面
        rpcProvider.create(pageRequest);
        
        // 搜索获取刚创建的页面ID
        PageMetaSearchRequest searchReq = new PageMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setProjectId(testProjectId);
        searchReq.setModuleId(testModuleId);
        EntityCollection<PageMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的页面");
        
        String pageId = ((java.util.List<PageMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(delete("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已删除
        PageMetaResponse deleted = rpcProvider.getById(pageId);
        assertNull(deleted, "页面应该已被删除");
    }

    @Test
    @DisplayName("测试删除页面 - 页面不存在")
    void testDeletePage_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String pageId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(delete("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试搜索页面 - 成功返回列表")
    void testSearchPages_Success() throws Exception {
        // Given - 先创建两个页面
        PageMetaRequest request1 = new PageMetaRequest();
        request1.setModuleId(testModuleId);
        request1.setName("搜索测试页面1");
        request1.setSlug("search-page-1");
        request1.setPageType(PageType.Form);
        rpcProvider.create(request1);
        
        PageMetaRequest request2 = new PageMetaRequest();
        request2.setModuleId(testModuleId);
        request2.setName("搜索测试页面2");
        request2.setSlug("search-page-2");
        request2.setPageType(PageType.View);
        rpcProvider.create(request2);

        // When & Then
        mockMvc.perform(post("/meta/modules/{moduleId}/pages/search", testModuleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.totalCount").value(org.hamcrest.Matchers.greaterThanOrEqualTo(2)))
                .andExpect(jsonPath("$.data.pageSize", is(10)));

        // 验证 RPC 调用返回的结果
        EntityCollection<PageMetaResponse> results = rpcProvider.search(searchRequest);
        assertTrue(results.getTotalCount() >= 2, "应该至少有两个页面");
    }

    @Test
    @DisplayName("测试搜索页面 - 分页测试")
    void testSearchPages_Pagination() throws Exception {
        // Given - 搜索请求
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(5);

        // When & Then
        mockMvc.perform(post("/meta/modules/{moduleId}/pages/search", testModuleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(5)));

        // 验证分页逻辑
        EntityCollection<PageMetaResponse> results = rpcProvider.search(searchRequest);
        assertNotNull(results);
        assertEquals(5, results.getPageSize());
    }

    @Test
    @DisplayName("测试搜索页面 - 按名称搜索")
    void testSearchPages_ByName() throws Exception {
        // Given - 创建特定名称的页面
        PageMetaRequest request = new PageMetaRequest();
        request.setModuleId(testModuleId);
        request.setName("特定名称页面");
        request.setSlug("specific-page");
        request.setPageType(PageType.Form);
        rpcProvider.create(request);

        // 设置搜索条件
        PageMetaSearchRequest nameSearchRequest = new PageMetaSearchRequest();
        nameSearchRequest.setPageIndex(1);
        nameSearchRequest.setPageSize(10);
        nameSearchRequest.setProjectId(testProjectId);
        nameSearchRequest.setModuleId(testModuleId);
        nameSearchRequest.setName("特定名称");

        // When & Then
        mockMvc.perform(post("/meta/modules/{moduleId}/pages/search", testModuleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(nameSearchRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证搜索结果
        EntityCollection<PageMetaResponse> results = rpcProvider.search(nameSearchRequest);
        assertTrue(results.getTotalCount() >= 1, "应该找到至少一个匹配的页面");
    }

    @Test
    @DisplayName("测试搜索页面 - 按页面类型搜索")
    void testSearchPages_ByPageType() throws Exception {
        // Given - 创建特定类型的页面
        PageMetaRequest request = new PageMetaRequest();
        request.setModuleId(testModuleId);
        request.setName("表单页面");
        request.setSlug("form-page");
        request.setPageType(PageType.Form);
        rpcProvider.create(request);

        // 设置搜索条件
        PageMetaSearchRequest typeSearchRequest = new PageMetaSearchRequest();
        typeSearchRequest.setPageIndex(1);
        typeSearchRequest.setPageSize(10);
        typeSearchRequest.setProjectId(testProjectId);
        typeSearchRequest.setModuleId(testModuleId);
        typeSearchRequest.setPageType(PageType.Form);

        // When & Then
        mockMvc.perform(post("/meta/modules/{moduleId}/pages/search", testModuleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(typeSearchRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证搜索结果
        EntityCollection<PageMetaResponse> results = rpcProvider.search(typeSearchRequest);
        assertTrue(results.getTotalCount() >= 1, "应该找到至少一个匹配的页面");
    }

    @Test
    @DisplayName("测试根据ID获取页面 - 成功")
    void testGetPageById_Success() throws Exception {
        // Given - 先创建一个页面
        rpcProvider.create(pageRequest);
        
        // 搜索获取刚创建的页面ID
        PageMetaSearchRequest searchReq = new PageMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setProjectId(testProjectId);
        searchReq.setModuleId(testModuleId);
        EntityCollection<PageMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的页面");
        
        String pageId = ((java.util.List<PageMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(get("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.id").exists())
                .andExpect(jsonPath("$.data.name", is("测试页面")))
                .andExpect(jsonPath("$.data.projectId", is(testProjectId)))
                .andExpect(jsonPath("$.data.moduleId", is(testModuleId)))
                .andExpect(jsonPath("$.data.slug", is("test-page")))
                .andExpect(jsonPath("$.data.pageType", is("Form")));

        // 验证 RPC 调用
        PageMetaResponse response = rpcProvider.getById(pageId);
        assertNotNull(response);
        assertEquals("测试页面", response.getName());
        assertEquals(testProjectId, response.getProjectId());
        assertEquals(testModuleId, response.getModuleId());
        assertEquals("test-page", response.getSlug());
        assertEquals(PageType.Form, response.getPageType());
    }

    @Test
    @DisplayName("测试根据ID获取页面 - 页面不存在")
    void testGetPageById_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String pageId = "999999999999";

        // When & Then
        mockMvc.perform(get("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证 RPC 调用返回 null
        PageMetaResponse response = rpcProvider.getById(pageId);
        assertNull(response, "不存在的页面应该返回 null");
    }

    @Test
    @DisplayName("测试创建多个页面 - 数据完整性")
    void testCreateMultiplePages() throws Exception {
        // Given - 创建多个页面
        for (int i = 1; i <= 3; i++) {
            PageMetaRequest request = new PageMetaRequest();
            request.setModuleId(testModuleId);
            request.setName("批量测试页面" + i);
            request.setSlug("batch-page-" + i);
            request.setPageType(i % 2 == 0 ? PageType.View : PageType.Form);
            rpcProvider.create(request);
        }

        // When - 搜索所有页面
        EntityCollection<PageMetaResponse> results = rpcProvider.search(searchRequest);

        // Then - 验证至少创建了3个页面
        assertTrue(results.getTotalCount() >= 3, "应该至少创建了3个页面");
        
        // 验证每个页面的数据
        ((java.util.List<PageMetaResponse>) results.getData()).forEach(response -> {
            assertNotNull(response.getId());
            assertNotNull(response.getName());
            assertNotNull(response.getProjectId());
            assertNotNull(response.getModuleId());
            assertNotNull(response.getSlug());
            assertNotNull(response.getPageType());
            assertNotNull(response.getCreatedTime());
            assertTrue(response.getVersion() >= 0, "版本号应该大于等于0");
        });
    }

    @Test
    @DisplayName("测试更新页面 - 完整字段测试")
    void testUpdatePage_AllFields() throws Exception {
        // Given - 先创建一个页面
        rpcProvider.create(pageRequest);
        
        // 搜索获取刚创建的页面ID
        PageMetaSearchRequest searchReq = new PageMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setProjectId(testProjectId);
        searchReq.setModuleId(testModuleId);
        EntityCollection<PageMetaResponse> searchResults = rpcProvider.search(searchReq);
        String pageId = ((java.util.List<PageMetaResponse>) searchResults.getData()).get(0).getId();
        
        PageMetaRequest fullRequest = new PageMetaRequest();
        fullRequest.setModuleId(testModuleId);
        fullRequest.setName("完整页面名称");
        fullRequest.setNote("完整页面备注");
        fullRequest.setFormJson("{\"fields\":[{\"name\":\"full\",\"type\":\"text\"}]}");

        // When & Then
        mockMvc.perform(put("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(fullRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证所有字段都已更新
        PageMetaResponse updated = rpcProvider.getById(pageId);
        assertNotNull(updated);
        assertEquals("完整页面名称", updated.getName());
        assertEquals("完整页面备注", updated.getNote());
        assertEquals(testProjectId, updated.getProjectId());
        assertEquals(testModuleId, updated.getModuleId());
    }

    @Test
    @DisplayName("测试不同模块下的页面隔离")
    void testPageIsolationBetweenModules() throws Exception {
        // Given - 创建两个不同模块的页面
        String moduleId1 = "module001";
        String moduleId2 = "module002";
        
        PageMetaRequest request1 = new PageMetaRequest();
        request1.setModuleId(moduleId1);
        request1.setName("模块1的页面");
        request1.setSlug("module1-page");
        request1.setPageType(PageType.Form);
        rpcProvider.create(request1);
        
        PageMetaRequest request2 = new PageMetaRequest();
        request2.setModuleId(moduleId2);
        request2.setName("模块2的页面");
        request2.setSlug("module2-page");
        request2.setPageType(PageType.View);
        rpcProvider.create(request2);

        // When - 搜索模块1的页面
        PageMetaSearchRequest searchReq1 = new PageMetaSearchRequest();
        searchReq1.setPageIndex(1);
        searchReq1.setPageSize(10);
        searchReq1.setProjectId(testProjectId);
        searchReq1.setModuleId(moduleId1);
        EntityCollection<PageMetaResponse> results1 = rpcProvider.search(searchReq1);

        // Then - 验证页面隔离
        assertTrue(results1.getTotalCount() >= 1, "应该至少有一个页面");
        
        // 验证每个页面都属于正确的模块
        ((java.util.List<PageMetaResponse>) results1.getData()).forEach(response -> {
            assertTrue(response.getModuleId().equals(moduleId1), 
                      "页面应该属于正确的模块");
        });
    }

    @Test
    @DisplayName("测试页面版本管理")
    void testPageVersionManagement() throws Exception {
        // Given - 创建一个页面
        rpcProvider.create(pageRequest);
        
        // 搜索获取刚创建的页面ID
        PageMetaSearchRequest searchReq = new PageMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setProjectId(testProjectId);
        searchReq.setModuleId(testModuleId);
        EntityCollection<PageMetaResponse> searchResults = rpcProvider.search(searchReq);
        String pageId = ((java.util.List<PageMetaResponse>) searchResults.getData()).get(0).getId();

        // 获取初始版本
        PageMetaResponse initialPage = rpcProvider.getById(pageId);
        long initialVersion = initialPage.getVersion();

        // When - 更新页面（应该增加版本号）
        PageMetaRequest updateRequest = new PageMetaRequest();
        updateRequest.setModuleId(testModuleId);
        updateRequest.setName("更新后的页面");
        updateRequest.setFormJson("{\"fields\":[{\"name\":\"updated\",\"type\":\"text\"}]}");
        
        mockMvc.perform(put("/meta/modules/{moduleId}/pages/{id}", testModuleId, pageId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // Then - 验证版本号已更新
        PageMetaResponse updatedPage = rpcProvider.getById(pageId);
        assertNotNull(updatedPage);
        assertTrue(updatedPage.getVersion() >= initialVersion, "版本号应该增加或保持不变");
        assertEquals("更新后的页面", updatedPage.getName());
    }
}