package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.enums.MetaCreatorReferenceType;
import com.bcxin.tenant.bcx.jdks.OrganRegistryFormsRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.organregistry.OrganRegistryFormsRequest;
import com.bcxin.tenant.bcx.jdks.requests.organregistry.OrganRegistryFormsSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.organregistry.OrganRegistryFormsResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

/**
 * OrganRegistryFormsBusController 集成测试
 * 
 * 测试覆盖：
 * 1. 创建企业注册表单 (POST /bus/organ-registry-forms)
 * 2. 更新企业注册表单 (PUT /bus/organ-registry-forms/{id})
 * 3. 删除企业注册表单 (DELETE /bus/organ-registry-forms/{id})
 * 4. 批量删除企业注册表单 (DELETE /bus/organ-registry-forms/batch)
 * 5. 搜索企业注册表单 (POST /bus/organ-registry-forms/search)
 * 6. 获取单个企业注册表单 (GET /bus/organ-registry-forms/{id})
 * 7. 根据条件查找企业注册表单 (GET /bus/organ-registry-forms/find-by-condition)
 * 
 * 注意：使用 @SpringBootTest 进行集成测试，会真实操作数据库
 * 使用 @Transactional 确保测试后数据回滚
 */
@SpringBootTest
@AutoConfigureMockMvc
@ActiveProfiles("test")
@Transactional
@DisplayName("OrganRegistryFormsBusController 集成测试")
class OrganRegistryFormsBusControllerTest {

    @Autowired
    private MockMvc mockMvc;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private OrganRegistryFormsRpcProvider rpcProvider;

    private OrganRegistryFormsRequest testRequest;
    private OrganRegistryFormsSearchRequest searchRequest;

    @BeforeEach
    void setUp() {
        // 初始化测试数据
        testRequest = new OrganRegistryFormsRequest();
        testRequest.setInstitutionalTypeCode("0102");
        testRequest.setIndustryTypeCode("01");
        testRequest.setRegionCode("BJ");
        testRequest.setPageId("1001");

        searchRequest = new OrganRegistryFormsSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(10);
    }

    @Test
    @DisplayName("测试创建企业注册表单 - 成功")
    void testCreateOrganRegistryForm_Success() throws Exception {
        // When & Then - 真实创建企业注册表单到数据库
        mockMvc.perform(post("/bus/organ-registry-forms")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(testRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据真正保存到数据库 - 通过搜索验证
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(100);
        searchReq.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> results = rpcProvider.search(searchReq);
        
        // 验证至少创建了一个企业注册表单
        assertTrue(results.getTotalCount() > 0, "应该至少有一个企业注册表单被创建");
    }

    @Test
    @DisplayName("测试创建企业注册表单 - 空参数")
    void testCreateOrganRegistryForm_EmptyRequest() throws Exception {
        // Given - 创建一个无效的请求（没有必填字段）
        OrganRegistryFormsRequest invalidRequest = new OrganRegistryFormsRequest();

        // When & Then - 应该返回400错误
        mockMvc.perform(post("/bus/organ-registry-forms")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(invalidRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试更新企业注册表单 - 成功")
    void testUpdateOrganRegistryForm_Success() throws Exception {
        // Given - 先创建一个企业注册表单
        rpcProvider.create(testRequest);
        
        // 搜索获取刚创建的企业注册表单ID
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的企业注册表单");
        
        String formId = ((List<OrganRegistryFormsResponse>) searchResults.getData()).get(0).getId();
        
        // 修改请求
        OrganRegistryFormsRequest updateRequest = new OrganRegistryFormsRequest();
        updateRequest.setInstitutionalTypeCode("0102");
        updateRequest.setIndustryTypeCode("01");
        updateRequest.setRegionCode("SH");
        updateRequest.setPageId("1001");

        // When & Then
        mockMvc.perform(put("/bus/organ-registry-forms/{id}", formId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已更新
        OrganRegistryFormsResponse updated = rpcProvider.getById(formId);
        assertNotNull(updated);
        assertEquals("0102", updated.getInstitutionalTypeCode());
        assertEquals("01", updated.getIndustryTypeCode());
        assertEquals("SH", updated.getRegionCode());
        assertEquals("1001", updated.getPageId());
        assertEquals(MetaCreatorReferenceType.Tenant, updated.getReferenceType());
    }

    @Test
    @DisplayName("测试更新企业注册表单 - 表单不存在")
    void testUpdateOrganRegistryForm_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String formId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(put("/bus/organ-registry-forms/{id}", formId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(testRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试删除企业注册表单 - 成功")
    void testDeleteOrganRegistryForm_Success() throws Exception {
        // Given - 先创建一个企业注册表单
        rpcProvider.create(testRequest);
        
        // 搜索获取刚创建的企业注册表单ID
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的企业注册表单");
        
        String formId = ((List<OrganRegistryFormsResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(delete("/bus/organ-registry-forms/{id}", formId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已删除（逻辑删除）
        OrganRegistryFormsResponse deleted = rpcProvider.getById(formId);
//            判断是否逻辑删除
        if (deleted != null) {
            assertTrue(deleted.isDeleted(), "企业注册表单应该已被逻辑删除");
        }
//        assertNull(deleted, "企业注册表单应该已被删除");
    }

    @Test
    @DisplayName("测试删除企业注册表单 - 表单不存在")
    void testDeleteOrganRegistryForm_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String formId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(delete("/bus/organ-registry-forms/{id}", formId))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试批量删除企业注册表单 - 成功")
    void testBatchDeleteOrganRegistryForms_Success() throws Exception {
        // Given - 先创建两个企业注册表单
        rpcProvider.create(testRequest);
        
        OrganRegistryFormsRequest request2 = new OrganRegistryFormsRequest();
        request2.setInstitutionalTypeCode("0102");
        request2.setIndustryTypeCode("01");
        request2.setRegionCode("SH");
        request2.setPageId("1001");
        rpcProvider.create(request2);
        
        // 搜索获取刚创建的企业注册表单ID列表
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(10);
        EntityCollection<OrganRegistryFormsResponse> searchResults = rpcProvider.search(searchReq);
        assertTrue(searchResults.getTotalCount() >= 2, "应该至少有两个企业注册表单");
        
        List<String> formIds = ((List<OrganRegistryFormsResponse>) searchResults.getData())
                .stream()
                .map(OrganRegistryFormsResponse::getId)
                .limit(2)
                .toList();

        // When & Then
        mockMvc.perform(delete("/bus/organ-registry-forms/batch")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(formIds)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已删除
        for (String formId : formIds) {
            OrganRegistryFormsResponse deleted = rpcProvider.getById(formId);
//            判断是否逻辑删除
            if (deleted != null) {
                assertTrue(deleted.isDeleted(), "企业注册表单应该已被逻辑删除");
                continue;
            }
//            assertNull(deleted, "企业注册表单应该已被删除");
        }
    }

    @Test
    @DisplayName("测试搜索企业注册表单 - 成功返回列表")
    void testSearchOrganRegistryForms_Success() throws Exception {
        // Given - 先创建两个企业注册表单
        rpcProvider.create(testRequest);
        
        OrganRegistryFormsRequest request2 = new OrganRegistryFormsRequest();
        request2.setInstitutionalTypeCode("0102");
        request2.setIndustryTypeCode("01");
        request2.setRegionCode("SH");
        request2.setPageId("1001");
        rpcProvider.create(request2);

        // When & Then
        mockMvc.perform(post("/bus/organ-registry-forms/search")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.totalCount").value(org.hamcrest.Matchers.greaterThanOrEqualTo(2)))
                .andExpect(jsonPath("$.data.pageSize", is(10)));

        // 验证 RPC 调用返回的结果
        EntityCollection<OrganRegistryFormsResponse> results = rpcProvider.search(searchRequest);
        assertTrue(results.getTotalCount() >= 2, "应该至少有两个企业注册表单");
    }

    @Test
    @DisplayName("测试搜索企业注册表单 - 分页测试")
    void testSearchOrganRegistryForms_Pagination() throws Exception {
        // Given - 搜索请求
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(5);

        // When & Then
        mockMvc.perform(post("/bus/organ-registry-forms/search")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(5)));

        // 验证分页逻辑
        EntityCollection<OrganRegistryFormsResponse> results = rpcProvider.search(searchRequest);
        assertNotNull(results);
        assertEquals(5, results.getPageSize());
    }

    @Test
    @DisplayName("测试搜索企业注册表单 - 按机构类型搜索")
    void testSearchOrganRegistryForms_ByInstitutionalType() throws Exception {
        // Given - 创建特定机构类型的企业注册表单
        rpcProvider.create(testRequest);

        // 设置搜索条件
        OrganRegistryFormsSearchRequest typeSearchRequest = new OrganRegistryFormsSearchRequest();
        typeSearchRequest.setPageIndex(1);
        typeSearchRequest.setPageSize(10);
        typeSearchRequest.setInstitutionalTypeCode("0102");

        // When & Then
        mockMvc.perform(post("/bus/organ-registry-forms/search")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(typeSearchRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证搜索结果
        EntityCollection<OrganRegistryFormsResponse> results = rpcProvider.search(typeSearchRequest);
        assertTrue(results.getTotalCount() >= 1, "应该找到至少一个匹配的企业注册表单");
    }

    @Test
    @DisplayName("测试根据ID获取企业注册表单 - 成功")
    void testGetOrganRegistryFormById_Success() throws Exception {
        // Given - 先创建一个企业注册表单
        rpcProvider.create(testRequest);
        
        // 搜索获取刚创建的企业注册表单ID
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的企业注册表单");
        
        String formId = ((List<OrganRegistryFormsResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(get("/bus/organ-registry-forms/{id}", formId))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.id").exists())
                .andExpect(jsonPath("$.data.institutionalTypeCode", is("0102")))
                .andExpect(jsonPath("$.data.industryTypeCode", is("01")))
                .andExpect(jsonPath("$.data.regionCode", is("SH")))
                .andExpect(jsonPath("$.data.pageId", is("1001")))
                .andExpect(jsonPath("$.data.referenceType", is("Tenant")));

        // 验证 RPC 调用
        OrganRegistryFormsResponse response = rpcProvider.getById(formId);
        assertNotNull(response);
        assertEquals("0102", response.getInstitutionalTypeCode());
        assertEquals("01", response.getIndustryTypeCode());
        assertEquals("SH", response.getRegionCode());
        assertEquals("1001", response.getPageId());
//        assertEquals(MetaCreatorReferenceType.User, response.getReferenceType());
    }

    @Test
    @DisplayName("测试根据ID获取企业注册表单 - 表单不存在")
    void testGetOrganRegistryFormById_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String formId = "999999999999";

        // When & Then
        mockMvc.perform(get("/bus/organ-registry-forms/{id}", formId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证 RPC 调用返回 null
        OrganRegistryFormsResponse response = rpcProvider.getById(formId);
        assertNull(response, "不存在的企业注册表单应该返回 null");
    }

    @Test
    @DisplayName("测试根据条件查找企业注册表单 - 成功")
    void testFindByCondition_Success() throws Exception {
        // Given - 先创建一个企业注册表单
        rpcProvider.create(testRequest);

        // When & Then
        mockMvc.perform(get("/bus/organ-registry-forms/find-by-condition")
                        .param("institutionalTypeCode", "0102")
                        .param("industryTypeCode", "01")
                        .param("regionCode", "BJ"))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.institutionalTypeCode", is("0102")))
                .andExpect(jsonPath("$.data.industryTypeCode", is("01")))
                .andExpect(jsonPath("$.data.regionCode", is("BJ")));

        // 验证 RPC 调用
        OrganRegistryFormsResponse response = rpcProvider.findByCondition("0102", "01", "BJ");
        assertNotNull(response);
        assertEquals("0102", response.getInstitutionalTypeCode());
        assertEquals("01", response.getIndustryTypeCode());
        assertEquals("BJ", response.getRegionCode());
    }

    @Test
    @DisplayName("测试根据条件查找企业注册表单 - 不存在")
    void testFindByCondition_NotFound() throws Exception {
        // When & Then - 查找不存在的组合
        mockMvc.perform(get("/bus/organ-registry-forms/find-by-condition")
                        .param("institutionalTypeCode", "NONEXISTENT")
                        .param("industryTypeCode", "NONEXISTENT")
                        .param("regionCode", "NONEXISTENT"))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证 RPC 调用返回 null
        OrganRegistryFormsResponse response = rpcProvider.findByCondition("NONEXISTENT", "NONEXISTENT", "NONEXISTENT");
        assertNull(response, "不存在的条件组合应该返回 null");
    }

    @Test
    @DisplayName("测试创建多个企业注册表单 - 数据完整性")
    void testCreateMultipleOrganRegistryForms() throws Exception {
        // Given - 创建多个企业注册表单
        for (int i = 1; i <= 3; i++) {
            OrganRegistryFormsRequest request = new OrganRegistryFormsRequest();
            request.setInstitutionalTypeCode("0102");
            request.setIndustryTypeCode("01");
            request.setRegionCode("BJ" + i);
            request.setPageId("1001");
            rpcProvider.create(request);
        }

        // When - 搜索所有企业注册表单
        EntityCollection<OrganRegistryFormsResponse> results = rpcProvider.search(searchRequest);

        // Then - 验证至少创建了3个企业注册表单
        assertTrue(results.getTotalCount() >= 3, "应该至少创建了3个企业注册表单");
        
        // 验证每个企业注册表单的数据
        ((List<OrganRegistryFormsResponse>) results.getData()).forEach(response -> {
            assertNotNull(response.getId());
            assertNotNull(response.getInstitutionalTypeCode());
            assertNotNull(response.getIndustryTypeCode());
            assertNotNull(response.getRegionCode());
            assertNotNull(response.getPageId());
            assertNotNull(response.getReferenceType());
            assertNotNull(response.getReferenceNumber());
            assertNotNull(response.getCreatorId());
            assertNotNull(response.getCreatedTime());
            assertFalse(response.isDeleted(), "不应该被删除");
        });
    }

    @Test
    @DisplayName("测试更新企业注册表单 - 完整字段测试")
    void testUpdateOrganRegistryForm_AllFields() throws Exception {
        // Given - 先创建一个企业注册表单
        rpcProvider.create(testRequest);
        
        // 搜索获取刚创建的企业注册表单ID
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> searchResults = rpcProvider.search(searchReq);
        String formId = ((List<OrganRegistryFormsResponse>) searchResults.getData()).get(0).getId();
        
        OrganRegistryFormsRequest fullRequest = new OrganRegistryFormsRequest();
        fullRequest.setInstitutionalTypeCode("0102");
        fullRequest.setIndustryTypeCode("01");
        fullRequest.setRegionCode("SH");
        fullRequest.setPageId("1001");

        // When & Then
        mockMvc.perform(put("/bus/organ-registry-forms/{id}", formId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(fullRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证所有字段都已更新
        OrganRegistryFormsResponse updated = rpcProvider.getById(formId);
        assertNotNull(updated);
        assertEquals("0102", updated.getInstitutionalTypeCode());
        assertEquals("01", updated.getIndustryTypeCode());
        assertEquals("SH", updated.getRegionCode());
        assertEquals("1001", updated.getPageId());
        assertEquals(MetaCreatorReferenceType.Tenant, updated.getReferenceType());
        assertEquals("TENANT_001", updated.getReferenceNumber());
    }

    @Test
    @DisplayName("测试不同条件组合的企业注册表单隔离")
    void testOrganRegistryFormIsolationBetweenConditions() throws Exception {
        // Given - 创建两个不同条件的企业注册表单
        OrganRegistryFormsRequest request1 = new OrganRegistryFormsRequest();
        request1.setInstitutionalTypeCode("0102");
        request1.setIndustryTypeCode("01");
        request1.setRegionCode("BJ");
        request1.setPageId("1001");
        rpcProvider.create(request1);
        
        OrganRegistryFormsRequest request2 = new OrganRegistryFormsRequest();
        request2.setInstitutionalTypeCode("0102");
        request2.setIndustryTypeCode("01");
        request2.setRegionCode("SH");
        request2.setPageId("1001");
        rpcProvider.create(request2);

        // When - 搜索特定条件的企业注册表单
        OrganRegistryFormsSearchRequest searchReq1 = new OrganRegistryFormsSearchRequest();
        searchReq1.setPageIndex(1);
        searchReq1.setPageSize(10);
        searchReq1.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> results1 = rpcProvider.search(searchReq1);

        // Then - 验证条件隔离
        assertTrue(results1.getTotalCount() >= 1, "应该至少有一个企业注册表单");
        
        // 验证每个企业注册表单都属于正确的条件
        ((List<OrganRegistryFormsResponse>) results1.getData()).forEach(response -> {
            assertTrue(response.getInstitutionalTypeCode().equals("0102"), 
                      "企业注册表单应该属于正确的机构类型");
        });
    }

    @Test
    @DisplayName("测试企业注册表单的创建者信息管理")
    void testOrganRegistryFormCreatorManagement() throws Exception {
        // Given - 创建企业注册表单
        rpcProvider.create(testRequest);
        
        // 搜索获取刚创建的企业注册表单ID
        OrganRegistryFormsSearchRequest searchReq = new OrganRegistryFormsSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        searchReq.setInstitutionalTypeCode("0102");
        EntityCollection<OrganRegistryFormsResponse> searchResults = rpcProvider.search(searchReq);
        String formId = ((List<OrganRegistryFormsResponse>) searchResults.getData()).get(0).getId();

        // When - 更新企业注册表单（应该更新最后修改人）
        OrganRegistryFormsRequest updateRequest = new OrganRegistryFormsRequest();
        updateRequest.setInstitutionalTypeCode("0102");
        updateRequest.setIndustryTypeCode("01");
        updateRequest.setRegionCode("BJ");
        updateRequest.setPageId("1001");

        mockMvc.perform(put("/bus/organ-registry-forms/{id}", formId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // Then - 验证最后修改人已更新
        OrganRegistryFormsResponse updated = rpcProvider.getById(formId);
        assertNotNull(updated);
        assertEquals("UPDATER_002", updated.getLastUpdaterId());
        assertEquals("CREATOR_001", updated.getCreatorId()); // 创建者应该保持不变
    }
}
