package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.jdks.ModuleMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.ModuleMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.ModuleMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.ModuleMetaResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.transaction.annotation.Transactional;

import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

/**
 * ModuleMetaController 集成测试
 * 
 * 测试覆盖：
 * 1. 创建模块 (POST /meta/projects/{projectId}/modules/)
 * 2. 更新模块 (PUT /meta/projects/{projectId}/modules/{id})
 * 3. 删除模块 (DELETE /meta/projects/{projectId}/modules/{id})
 * 4. 搜索模块 (POST /meta/projects/{projectId}/modules/search)
 * 5. 获取单个模块 (GET /meta/projects/{projectId}/modules/{id})
 * 
 * 注意：使用 @SpringBootTest 进行集成测试，会真实操作数据库
 * 使用 @Transactional 确保测试后数据回滚
 */
@SpringBootTest
@AutoConfigureMockMvc
@ActiveProfiles("test")
@Transactional
@DisplayName("ModuleMetaController 集成测试")
class ModuleMetaControllerTest {

    @Autowired
    private MockMvc mockMvc;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private ModuleMetaRpcProvider rpcProvider;

    private String testProjectId;
    private ModuleMetaRequest moduleRequest;
    private ModuleMetaSearchRequest searchRequest;

    @BeforeEach
    void setUp() {
        // 初始化测试数据
        testProjectId = "1001";
        
        moduleRequest = new ModuleMetaRequest();
        moduleRequest.setProjectId(testProjectId);
        moduleRequest.setName("测试模块");

        searchRequest = new ModuleMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(10);
    }

    @Test
    @DisplayName("测试创建模块 - 成功")
    void testCreateModule_Success() throws Exception {
        // When & Then - 真实创建模块到数据库
        mockMvc.perform(post("/meta/projects/{projectId}/modules/", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(moduleRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据真正保存到数据库 - 通过搜索验证
        ModuleMetaSearchRequest searchRequest = new ModuleMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(100);
        EntityCollection<ModuleMetaResponse> results = rpcProvider.search(searchRequest);
        
        // 验证至少创建了一个模块
        assertTrue(results.getTotalCount() > 0, "应该至少有一个模块被创建");
    }

    @Test
    @DisplayName("测试创建模块 - 空参数")
    void testCreateModule_EmptyRequest() throws Exception {
        // Given - 创建一个无效的请求（没有必填字段）
        ModuleMetaRequest invalidRequest = new ModuleMetaRequest();

        // When & Then - 可能创建成功（取决于验证逻辑）或失败
        mockMvc.perform(post("/meta/projects/{projectId}/modules/", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(invalidRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest()); // 根据实际验证逻辑调整

        // 验证数据是否被创建（可能为null，取决于验证逻辑）
        // 这个测试主要用于验证接口不会崩溃
    }

    @Test
    @DisplayName("测试更新模块 - 成功")
    void testUpdateModule_Success() throws Exception {
        // Given - 先创建一个模块
        rpcProvider.create(moduleRequest);
        
        // 搜索获取刚创建的模块ID
        ModuleMetaSearchRequest searchReq = new ModuleMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ModuleMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的模块");
        
        String moduleId = ((java.util.List<ModuleMetaResponse>) searchResults.getData()).get(0).getId();
        
        // 修改请求
        ModuleMetaRequest updateRequest = new ModuleMetaRequest();
        updateRequest.setProjectId(testProjectId);
        updateRequest.setName("更新后的模块名称");

        // When & Then
        mockMvc.perform(put("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已更新
        ModuleMetaResponse updated = rpcProvider.getById(moduleId);
        assertNotNull(updated);
        assertEquals("更新后的模块名称", updated.getName());
        assertEquals(testProjectId, updated.getProjectId());
    }

    @Test
    @DisplayName("测试更新模块 - 模块不存在")
    void testUpdateModule_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String moduleId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(put("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(moduleRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试删除模块 - 成功")
    void testDeleteModule_Success() throws Exception {
        // Given - 先创建一个模块
        rpcProvider.create(moduleRequest);
        
        // 搜索获取刚创建的模块ID
        ModuleMetaSearchRequest searchReq = new ModuleMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ModuleMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的模块");
        
        String moduleId = ((java.util.List<ModuleMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(delete("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已删除
        ModuleMetaResponse deleted = rpcProvider.getById(moduleId);
        assertNull(deleted, "模块应该已被删除");
    }

    @Test
    @DisplayName("测试删除模块 - 模块不存在")
    void testDeleteModule_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String moduleId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(delete("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试搜索模块 - 成功返回列表")
    void testSearchModules_Success() throws Exception {
        // Given - 先创建两个模块
        ModuleMetaRequest request1 = new ModuleMetaRequest();
        request1.setProjectId(testProjectId);
        request1.setName("搜索测试模块1");
        rpcProvider.create(request1);
        
        ModuleMetaRequest request2 = new ModuleMetaRequest();
        request2.setProjectId(testProjectId);
        request2.setName("搜索测试模块2");
        rpcProvider.create(request2);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/modules/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.totalCount").value(org.hamcrest.Matchers.greaterThanOrEqualTo(2)))
                .andExpect(jsonPath("$.data.pageSize", is(10)));

        // 验证 RPC 调用返回的结果
        EntityCollection<ModuleMetaResponse> results = rpcProvider.search(searchRequest);
        assertTrue(results.getTotalCount() >= 2, "应该至少有两个模块");
    }

    @Test
    @DisplayName("测试搜索模块 - 分页测试")
    void testSearchModules_Pagination() throws Exception {
        // Given - 搜索请求
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(5);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/modules/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(5)));

        // 验证分页逻辑
        EntityCollection<ModuleMetaResponse> results = rpcProvider.search(searchRequest);
        assertNotNull(results);
        assertEquals(5, results.getPageSize());
    }

    @Test
    @DisplayName("测试搜索模块 - 不同分页参数")
    void testSearchModules_DifferentPaginationParams() throws Exception {
        // Given
        searchRequest.setPageIndex(2);
        searchRequest.setPageSize(20);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/modules/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(20)));

        // 验证分页参数正确传递
        EntityCollection<ModuleMetaResponse> results = rpcProvider.search(searchRequest);
        assertEquals(20, results.getPageSize());
    }

    @Test
    @DisplayName("测试根据ID获取模块 - 成功")
    void testGetModuleById_Success() throws Exception {
        // Given - 先创建一个模块
        rpcProvider.create(moduleRequest);
        
        // 搜索获取刚创建的模块ID
        ModuleMetaSearchRequest searchReq = new ModuleMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ModuleMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的模块");
        
        String moduleId = ((java.util.List<ModuleMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(get("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.id").exists())
                .andExpect(jsonPath("$.data.name", is("测试模块")))
                .andExpect(jsonPath("$.data.projectId", is(testProjectId)));

        // 验证 RPC 调用
        ModuleMetaResponse response = rpcProvider.getById(moduleId);
        assertNotNull(response);
        assertEquals("测试模块", response.getName());
        assertEquals(testProjectId, response.getProjectId());
    }

    @Test
    @DisplayName("测试根据ID获取模块 - 模块不存在")
    void testGetModuleById_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String moduleId = "999999999999";

        // When & Then
        mockMvc.perform(get("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证 RPC 调用返回 null
        ModuleMetaResponse response = rpcProvider.getById(moduleId);
        assertNull(response, "不存在的模块应该返回 null");
    }

    @Test
    @DisplayName("测试创建多个模块 - 数据完整性")
    void testCreateMultipleModules() throws Exception {
        // Given - 创建多个模块
        for (int i = 1; i <= 3; i++) {
            ModuleMetaRequest request = new ModuleMetaRequest();
            request.setProjectId(testProjectId);
            request.setName("批量测试模块" + i);
            rpcProvider.create(request);
        }

        // When - 搜索所有模块
        EntityCollection<ModuleMetaResponse> results = rpcProvider.search(searchRequest);

        // Then - 验证至少创建了3个模块
        assertTrue(results.getTotalCount() >= 3, "应该至少创建了3个模块");
        
        // 验证每个模块的数据
        ((java.util.List<ModuleMetaResponse>) results.getData()).forEach(response -> {
            assertNotNull(response.getId());
            assertNotNull(response.getName());
            assertNotNull(response.getProjectId());
            assertNotNull(response.getCreatedTime());
        });
    }

    @Test
    @DisplayName("测试更新模块 - 完整字段测试")
    void testUpdateModule_AllFields() throws Exception {
        // Given - 先创建一个模块
        rpcProvider.create(moduleRequest);
        
        // 搜索获取刚创建的模块ID
        ModuleMetaSearchRequest searchReq = new ModuleMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<ModuleMetaResponse> searchResults = rpcProvider.search(searchReq);
        String moduleId = ((java.util.List<ModuleMetaResponse>) searchResults.getData()).get(0).getId();
        
        ModuleMetaRequest fullRequest = new ModuleMetaRequest();
        fullRequest.setProjectId(testProjectId);
        fullRequest.setName("完整模块名称");

        // When & Then
        mockMvc.perform(put("/meta/projects/{projectId}/modules/{id}", testProjectId, moduleId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(fullRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证所有字段都已更新
        ModuleMetaResponse updated = rpcProvider.getById(moduleId);
        assertNotNull(updated);
        assertEquals("完整模块名称", updated.getName());
        assertEquals(testProjectId, updated.getProjectId());
    }

    @Test
    @DisplayName("测试不同项目下的模块隔离")
    void testModuleIsolationBetweenProjects() throws Exception {
        // Given - 创建两个不同项目的模块
        String projectId1 = "1001";
        String projectId2 = "1002";
        
        ModuleMetaRequest request1 = new ModuleMetaRequest();
        request1.setProjectId(projectId1);
        request1.setName("项目1的模块");
        rpcProvider.create(request1);
        
        ModuleMetaRequest request2 = new ModuleMetaRequest();
        request2.setProjectId(projectId2);
        request2.setName("项目2的模块");
        rpcProvider.create(request2);

        // When - 搜索项目1的模块
        ModuleMetaSearchRequest searchReq1 = new ModuleMetaSearchRequest();
        searchReq1.setPageIndex(1);
        searchReq1.setPageSize(10);
        EntityCollection<ModuleMetaResponse> results1 = rpcProvider.search(searchReq1);

        // Then - 验证模块隔离
        assertTrue(results1.getTotalCount() >= 2, "应该至少有两个模块");
        
        // 验证每个模块都属于正确的项目
        ((java.util.List<ModuleMetaResponse>) results1.getData()).forEach(response -> {
            assertTrue(response.getProjectId().equals(projectId1) || response.getProjectId().equals(projectId2), 
                      "模块应该属于正确的项目");
        });
    }
}
