package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.enums.MenuMediumType;
import com.bcxin.tenant.bcx.jdks.MenuMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.MenuMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.MenuMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.MenuMetaResponse;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.http.MediaType;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.transaction.annotation.Transactional;

import java.util.Set;

import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

/**
 * MenuMetaController 集成测试
 * 
 * 测试覆盖：
 * 1. 创建菜单 (POST /meta/projects/{projectId}/menu/)
 * 2. 更新菜单 (PUT /meta/projects/{projectId}/menu/{id})
 * 3. 删除菜单 (DELETE /meta/projects/{projectId}/menu/{id})
 * 4. 搜索菜单 (POST /meta/projects/{projectId}/menu/search)
 * 5. 获取单个菜单 (GET /meta/projects/{projectId}/menu/{id})
 * 
 * 注意：使用 @SpringBootTest 进行集成测试，会真实操作数据库
 * 使用 @Transactional 确保测试后数据回滚
 */
@SpringBootTest
@AutoConfigureMockMvc
@ActiveProfiles("test")
@Transactional
@DisplayName("MenuMetaController 集成测试")
class MenuMetaControllerTest {

    @Autowired
    private MockMvc mockMvc;

    @Autowired
    private ObjectMapper objectMapper;

    @Autowired
    private MenuMetaRpcProvider rpcProvider;

    private String testProjectId;
    private MenuMetaRequest menuRequest;
    private MenuMetaSearchRequest searchRequest;

    @BeforeEach
    void setUp() {
        // 初始化测试数据
        testProjectId = "1001";
        
        menuRequest = new MenuMetaRequest();
        menuRequest.setProjectId(testProjectId);
        menuRequest.setName("测试菜单");
        menuRequest.setPageId("page001");
        menuRequest.setParentId(null);
        menuRequest.setIndexTreeId("tree001");
        menuRequest.setSlug("test-menu");
        menuRequest.setMediumType(Set.of(MenuMediumType.PC));

        searchRequest = new MenuMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(10);
    }

    @Test
    @DisplayName("测试创建菜单 - 成功")
    void testCreateMenu_Success() throws Exception {
        // When & Then - 真实创建菜单到数据库
        mockMvc.perform(post("/meta/projects/{projectId}/menu/", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(menuRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据真正保存到数据库 - 通过搜索验证
        MenuMetaSearchRequest searchRequest = new MenuMetaSearchRequest();
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(100);
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(searchRequest);
        
        // 验证至少创建了一个菜单
        assertTrue(results.getTotalCount() > 0, "应该至少有一个菜单被创建");
    }

    @Test
    @DisplayName("测试创建菜单 - 空参数")
    void testCreateMenu_EmptyRequest() throws Exception {
        // Given - 创建一个无效的请求（没有必填字段）
        MenuMetaRequest invalidRequest = new MenuMetaRequest();

        // When & Then - 可能创建成功（取决于验证逻辑）或失败
        mockMvc.perform(post("/meta/projects/{projectId}/menu/", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(invalidRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest()); // 根据实际验证逻辑调整

        // 验证数据是否被创建（可能为null，取决于验证逻辑）
        // 这个测试主要用于验证接口不会崩溃
    }

    @Test
    @DisplayName("测试更新菜单 - 成功")
    void testUpdateMenu_Success() throws Exception {
        // Given - 先创建一个菜单
        rpcProvider.create(menuRequest);
        
        // 搜索获取刚创建的菜单ID
        MenuMetaSearchRequest searchReq = new MenuMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<MenuMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的菜单");
        
        String menuId = ((java.util.List<MenuMetaResponse>) searchResults.getData()).get(0).getId();
        
        // 修改请求
        MenuMetaRequest updateRequest = new MenuMetaRequest();
        updateRequest.setProjectId(testProjectId);
        updateRequest.setName("更新后的菜单名称");
        updateRequest.setPageId("page002");
        updateRequest.setParentId("parent001");
        updateRequest.setIndexTreeId("tree002");
        updateRequest.setSlug("updated-menu");
        updateRequest.setMediumType(Set.of(MenuMediumType.APP));

        // When & Then
        mockMvc.perform(put("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(updateRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已更新
        MenuMetaResponse updated = rpcProvider.getById(menuId);
        assertNotNull(updated);
        assertEquals("更新后的菜单名称", updated.getName());
        assertEquals(testProjectId, updated.getProjectId());
        assertEquals("page002", updated.getPageId());
        assertEquals("parent001", updated.getParentId());
        assertEquals("tree002", updated.getIndexTreeId());
        assertEquals("updated-menu", updated.getSlug());
        assertEquals(2, updated.getMediumType());
    }

    @Test
    @DisplayName("测试更新菜单 - 菜单不存在")
    void testUpdateMenu_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String menuId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(put("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(menuRequest)))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试删除菜单 - 成功")
    void testDeleteMenu_Success() throws Exception {
        // Given - 先创建一个菜单
        rpcProvider.create(menuRequest);
        
        // 搜索获取刚创建的菜单ID
        MenuMetaSearchRequest searchReq = new MenuMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<MenuMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的菜单");
        
        String menuId = ((java.util.List<MenuMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(delete("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证数据已删除
        MenuMetaResponse deleted = rpcProvider.getById(menuId);
        assertNull(deleted, "菜单应该已被删除");
    }

    @Test
    @DisplayName("测试删除菜单 - 菜单不存在")
    void testDeleteMenu_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String menuId = "999999999999";

        // When & Then - 应该抛出异常
        mockMvc.perform(delete("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId))
                .andDo(print())
                .andExpect(status().isBadRequest());
    }

    @Test
    @DisplayName("测试搜索菜单 - 成功返回列表")
    void testSearchMenus_Success() throws Exception {
        // Given - 先创建两个菜单
        MenuMetaRequest request1 = new MenuMetaRequest();
        request1.setProjectId(testProjectId);
        request1.setName("搜索测试菜单1");
        request1.setPageId("page001");
        request1.setSlug("search-menu-1");
        request1.setMediumType(Set.of(MenuMediumType.PC));
        rpcProvider.create(request1);
        
        MenuMetaRequest request2 = new MenuMetaRequest();
        request2.setProjectId(testProjectId);
        request2.setName("搜索测试菜单2");
        request2.setPageId("page002");
        request2.setSlug("search-menu-2");
        request2.setMediumType(Set.of(MenuMediumType.APP));
        rpcProvider.create(request2);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/menu/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.totalCount").value(org.hamcrest.Matchers.greaterThanOrEqualTo(2)))
                .andExpect(jsonPath("$.data.pageSize", is(10)));

        // 验证 RPC 调用返回的结果
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(searchRequest);
        assertTrue(results.getTotalCount() >= 2, "应该至少有两个菜单");
    }

    @Test
    @DisplayName("测试搜索菜单 - 分页测试")
    void testSearchMenus_Pagination() throws Exception {
        // Given - 搜索请求
        searchRequest.setPageIndex(1);
        searchRequest.setPageSize(5);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/menu/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(5)));

        // 验证分页逻辑
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(searchRequest);
        assertNotNull(results);
        assertEquals(5, results.getPageSize());
    }

    @Test
    @DisplayName("测试搜索菜单 - 不同分页参数")
    void testSearchMenus_DifferentPaginationParams() throws Exception {
        // Given
        searchRequest.setPageIndex(2);
        searchRequest.setPageSize(20);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/menu/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(searchRequest)))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.pageSize", is(20)));

        // 验证分页参数正确传递
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(searchRequest);
        assertEquals(20, results.getPageSize());
    }

    @Test
    @DisplayName("测试搜索菜单 - 按名称搜索")
    void testSearchMenus_ByName() throws Exception {
        // Given - 创建特定名称的菜单
        MenuMetaRequest request = new MenuMetaRequest();
        request.setProjectId(testProjectId);
        request.setName("特定名称菜单");
        request.setPageId("page001");
        request.setSlug("specific-menu");
        request.setMediumType(Set.of(MenuMediumType.PC));
        rpcProvider.create(request);

        // 设置搜索条件
        MenuMetaSearchRequest nameSearchRequest = new MenuMetaSearchRequest();
        nameSearchRequest.setPageIndex(1);
        nameSearchRequest.setPageSize(10);
        nameSearchRequest.setName("特定名称");

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/menu/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(nameSearchRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证搜索结果
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(nameSearchRequest);
        assertTrue(results.getTotalCount() >= 1, "应该找到至少一个匹配的菜单");
    }

    @Test
    @DisplayName("测试搜索菜单 - 按页面ID搜索")
    void testSearchMenus_ByPageId() throws Exception {
        // Given - 创建特定页面ID的菜单
        MenuMetaRequest request = new MenuMetaRequest();
        request.setProjectId(testProjectId);
        request.setName("页面菜单");
        request.setPageId("specific-page-001");
        request.setSlug("page-menu");
        request.setMediumType(Set.of(MenuMediumType.PC));
        rpcProvider.create(request);

        // 设置搜索条件
        MenuMetaSearchRequest pageSearchRequest = new MenuMetaSearchRequest();
        pageSearchRequest.setPageIndex(1);
        pageSearchRequest.setPageSize(10);
        pageSearchRequest.setPageId("specific-page-001");

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/menu/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(pageSearchRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证搜索结果
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(pageSearchRequest);
        assertTrue(results.getTotalCount() >= 1, "应该找到至少一个匹配的菜单");
    }

    @Test
    @DisplayName("测试搜索菜单 - 按设备类型搜索")
    void testSearchMenus_ByMediumType() throws Exception {
        // Given - 创建特定设备类型的菜单
        MenuMetaRequest request = new MenuMetaRequest();
        request.setProjectId(testProjectId);
        request.setName("移动端菜单");
        request.setPageId("page001");
        request.setSlug("mobile-menu");
        request.setMediumType(Set.of(MenuMediumType.APP)); // 移动端
        rpcProvider.create(request);

        // 设置搜索条件
        MenuMetaSearchRequest mediumSearchRequest = new MenuMetaSearchRequest();
        mediumSearchRequest.setPageIndex(1);
        mediumSearchRequest.setPageSize(10);
        mediumSearchRequest.setMediumType(2);

        // When & Then
        mockMvc.perform(post("/meta/projects/{projectId}/menu/search", testProjectId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(mediumSearchRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证搜索结果
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(mediumSearchRequest);
        assertTrue(results.getTotalCount() >= 1, "应该找到至少一个匹配的菜单");
    }

    @Test
    @DisplayName("测试根据ID获取菜单 - 成功")
    void testGetMenuById_Success() throws Exception {
        // Given - 先创建一个菜单
        rpcProvider.create(menuRequest);
        
        // 搜索获取刚创建的菜单ID
        MenuMetaSearchRequest searchReq = new MenuMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<MenuMetaResponse> searchResults = rpcProvider.search(searchReq);
        assertFalse(searchResults.getData().isEmpty(), "应该能找到刚创建的菜单");
        
        String menuId = ((java.util.List<MenuMetaResponse>) searchResults.getData()).get(0).getId();

        // When & Then
        mockMvc.perform(get("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId))
                .andDo(print())
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.data.id").exists())
                .andExpect(jsonPath("$.data.name", is("测试菜单")))
                .andExpect(jsonPath("$.data.projectId", is(testProjectId)))
                .andExpect(jsonPath("$.data.pageId", is("page001")))
                .andExpect(jsonPath("$.data.slug", is("test-menu")))
                .andExpect(jsonPath("$.data.mediumType", is(1)));

        // 验证 RPC 调用
        MenuMetaResponse response = rpcProvider.getById(menuId);
        assertNotNull(response);
        assertEquals("测试菜单", response.getName());
        assertEquals(testProjectId, response.getProjectId());
        assertEquals("page001", response.getPageId());
        assertEquals("test-menu", response.getSlug());
        assertEquals(1, response.getMediumType());
    }

    @Test
    @DisplayName("测试根据ID获取菜单 - 菜单不存在")
    void testGetMenuById_NotFound() throws Exception {
        // Given - 使用一个不存在的ID
        String menuId = "999999999999";

        // When & Then
        mockMvc.perform(get("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证 RPC 调用返回 null
        MenuMetaResponse response = rpcProvider.getById(menuId);
        assertNull(response, "不存在的菜单应该返回 null");
    }

    @Test
    @DisplayName("测试创建多个菜单 - 数据完整性")
    void testCreateMultipleMenus() throws Exception {
        // Given - 创建多个菜单
        for (int i = 1; i <= 3; i++) {
            MenuMetaRequest request = new MenuMetaRequest();
            request.setProjectId(testProjectId);
            request.setName("批量测试菜单" + i);
            request.setPageId("page00" + i);
            request.setSlug("batch-menu-" + i);
            request.setMediumType(i % 2 == 0 ? Set.of(MenuMediumType.APP) : Set.of(MenuMediumType.PC));
            rpcProvider.create(request);
        }

        // When - 搜索所有菜单
        EntityCollection<MenuMetaResponse> results = rpcProvider.search(searchRequest);

        // Then - 验证至少创建了3个菜单
        assertTrue(results.getTotalCount() >= 3, "应该至少创建了3个菜单");
        
        // 验证每个菜单的数据
        ((java.util.List<MenuMetaResponse>) results.getData()).forEach(response -> {
            assertNotNull(response.getId());
            assertNotNull(response.getName());
            assertNotNull(response.getProjectId());
            assertNotNull(response.getPageId());
            assertNotNull(response.getSlug());
            assertNotNull(response.getCreatedTime());
            assertTrue(response.getMediumType() > 0, "设备类型应该大于0");
        });
    }

    @Test
    @DisplayName("测试更新菜单 - 完整字段测试")
    void testUpdateMenu_AllFields() throws Exception {
        // Given - 先创建一个菜单
        rpcProvider.create(menuRequest);
        
        // 搜索获取刚创建的菜单ID
        MenuMetaSearchRequest searchReq = new MenuMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<MenuMetaResponse> searchResults = rpcProvider.search(searchReq);
        String menuId = ((java.util.List<MenuMetaResponse>) searchResults.getData()).get(0).getId();
        
        MenuMetaRequest fullRequest = new MenuMetaRequest();
        fullRequest.setProjectId(testProjectId);
        fullRequest.setName("完整菜单名称");
        fullRequest.setPageId("full-page-001");
        fullRequest.setParentId("parent-001");
        fullRequest.setIndexTreeId("full-tree-001");
        fullRequest.setSlug("full-menu-slug");
        fullRequest.setMediumType(Set.of(MenuMediumType.APP));

        // When & Then
        mockMvc.perform(put("/meta/projects/{projectId}/menu/{id}", testProjectId, menuId)
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(fullRequest)))
                .andDo(print())
                .andExpect(status().isOk());

        // 验证所有字段都已更新
        MenuMetaResponse updated = rpcProvider.getById(menuId);
        assertNotNull(updated);
        assertEquals("完整菜单名称", updated.getName());
        assertEquals(testProjectId, updated.getProjectId());
        assertEquals("full-page-001", updated.getPageId());
        assertEquals("parent-001", updated.getParentId());
        assertEquals("full-tree-001", updated.getIndexTreeId());
        assertEquals("full-menu-slug", updated.getSlug());
        assertEquals(2, updated.getMediumType());
    }

    @Test
    @DisplayName("测试不同项目下的菜单隔离")
    void testMenuIsolationBetweenProjects() throws Exception {
        // Given - 创建两个不同项目的菜单
        String projectId1 = "1001";
        String projectId2 = "1002";
        
        MenuMetaRequest request1 = new MenuMetaRequest();
        request1.setProjectId(projectId1);
        request1.setName("项目1的菜单");
        request1.setPageId("page001");
        request1.setSlug("project1-menu");
        request1.setMediumType(Set.of(MenuMediumType.PC));
        rpcProvider.create(request1);
        
        MenuMetaRequest request2 = new MenuMetaRequest();
        request2.setProjectId(projectId2);
        request2.setName("项目2的菜单");
        request2.setPageId("page002");
        request2.setSlug("project2-menu");
        request2.setMediumType(Set.of(MenuMediumType.APP));
        rpcProvider.create(request2);

        // When - 搜索项目1的菜单
        MenuMetaSearchRequest searchReq1 = new MenuMetaSearchRequest();
        searchReq1.setPageIndex(1);
        searchReq1.setPageSize(10);
        EntityCollection<MenuMetaResponse> results1 = rpcProvider.search(searchReq1);

        // Then - 验证菜单隔离
        assertTrue(results1.getTotalCount() >= 2, "应该至少有两个菜单");
        
        // 验证每个菜单都属于正确的项目
        ((java.util.List<MenuMetaResponse>) results1.getData()).forEach(response -> {
            assertTrue(response.getProjectId().equals(projectId1) || response.getProjectId().equals(projectId2), 
                      "菜单应该属于正确的项目");
        });
    }

    @Test
    @DisplayName("测试菜单层级关系")
    void testMenuHierarchy() throws Exception {
        // Given - 创建父菜单
        MenuMetaRequest parentRequest = new MenuMetaRequest();
        parentRequest.setProjectId(testProjectId);
        parentRequest.setName("父菜单");
        parentRequest.setPageId("parent-page");
        parentRequest.setSlug("parent-menu");
        parentRequest.setMediumType(Set.of(MenuMediumType.PC));
        rpcProvider.create(parentRequest);

        // 获取父菜单ID
        MenuMetaSearchRequest searchReq = new MenuMetaSearchRequest();
        searchReq.setPageIndex(1);
        searchReq.setPageSize(1);
        EntityCollection<MenuMetaResponse> searchResults = rpcProvider.search(searchReq);
        String parentId = ((java.util.List<MenuMetaResponse>) searchResults.getData()).get(0).getId();

        // 创建子菜单
        MenuMetaRequest childRequest = new MenuMetaRequest();
        childRequest.setProjectId(testProjectId);
        childRequest.setName("子菜单");
        childRequest.setPageId("child-page");
        childRequest.setParentId(parentId);
        childRequest.setSlug("child-menu");
        childRequest.setMediumType(Set.of(MenuMediumType.PC));
        rpcProvider.create(childRequest);

        // When - 搜索子菜单
        MenuMetaSearchRequest childSearchRequest = new MenuMetaSearchRequest();
        childSearchRequest.setPageIndex(1);
        childSearchRequest.setPageSize(10);
        childSearchRequest.setParentId(parentId);
        EntityCollection<MenuMetaResponse> childResults = rpcProvider.search(childSearchRequest);

        // Then - 验证层级关系
        assertTrue(childResults.getTotalCount() >= 1, "应该找到至少一个子菜单");
        
        // 验证子菜单的父ID正确
        ((java.util.List<MenuMetaResponse>) childResults.getData()).forEach(response -> {
            if (response.getName().equals("子菜单")) {
                assertEquals(parentId, response.getParentId(), "子菜单的父ID应该正确");
            }
        });
    }
}
