package com.bcxin.tenant.bcx.rest.apis.filters;

import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.UserDetailResponse;
import com.bcxin.tenant.bcx.infrastructures.components.JsonProvider;
import com.bcxin.tenant.bcx.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ForbidTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.bcx.infrastructures.utils.ExceptionUtil;
import com.bcxin.tenant.bcx.infrastructures.valueTypes.TrafficTagValueType;
import com.bcxin.tenant.bcx.jdks.IdentityRpcProvider;
import com.bcxin.tenant.bcx.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.bcx.rest.apis.controllers.responses.ResponseBuilder;
import com.bcxin.tenant.bcx.rest.apis.utils.DubboCommonUtils;
import com.bcxin.tenant.bcx.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.bcx.rest.apis.utils.ServletRequestUtil;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Component
@Order(Ordered.HIGHEST_PRECEDENCE-1)
public class AuthFilter extends OncePerRequestFilter {
    private static final Logger logger = LoggerFactory.getLogger(AuthFilter.class);
    private final static Collection<String> excludedUrls = new ArrayList<>();
    private final static Collection<String> ignoredDeskCheckUrls = new ArrayList<>();

    static {
        excludedUrls.add("/identity/");
        //excludedUrls.add("/security-stations/station-types");
        /**
         * API Docs
         */
        excludedUrls.add("/v3/");
        excludedUrls.add("/swagger-ui");
        excludedUrls.add("/static");
        excludedUrls.add("/html");
        excludedUrls.add("/alive-connect");


        excludedUrls.add("/json");
        /**
         * 允许匿名访问页面表单数据接口，用于检查页面是否启用匿名访问
         */
        excludedUrls.add("/meta/pages/\\w+/form-json");
        /**
         * 允许匿名访问页面设计器全局配置接口
         */
        excludedUrls.add("/meta/page-settings/search");
        /**
         * websocket链接
         */
        ignoredDeskCheckUrls.add("/websocket/connect");
    }

    /**
     * todo 统一修复序列化方式
     */
    private final JsonProvider jsonProvider;
    private final IdentityRpcProvider identityRpcProvider;
    private final HotCacheProvider hotCacheProvider;

    private final String tenantNode;

    public AuthFilter(JsonProvider jsonProvider,
                      IdentityRpcProvider identityRpcProvider,
                      HotCacheProvider hotCacheProvider,
                      @Value("${tenant.node:null}") String tenantNode) {
        this.jsonProvider = jsonProvider;
        this.identityRpcProvider = identityRpcProvider;
        this.hotCacheProvider = hotCacheProvider;
        this.tenantNode = tenantNode;
    }


    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        boolean isAfterParseRequestToken = false;
        TenantContext tenantContext = TenantContext.getInstance();
        try {
            String requestUri = request.getRequestURI();
            boolean ignoredDeskCheck = ignoredDeskCheckUrls.stream()
                    .anyMatch(ix -> requestUri.startsWith(String.format("%s%s", request.getContextPath(), ix)));

            parseRequestToken(request, ignoredDeskCheck);
            isAfterParseRequestToken = true;


            TenantEmployeeContext userContext = tenantContext.getUserContext();
            if (userContext.get() == null) {
                throw new UnAuthorizedTenantException("当前无有效dispatchToken信息");
            }

            /**
             * 无调度台权限
             */
            TenantEmployeeContext.TenantUserModel userModel = userContext.get();

            responseWithFlowTag(response,tenantContext);
            filterChain.doFilter(request, response);
        } catch (Exception ex) {
            HttpStatus status = HttpStatus.BAD_REQUEST;
            String trace = ExceptionUtil.getStackMessage(ex);
            String message = "系统异常, 请联系管理员";

            if (ex instanceof TenantExceptionAbstract) {
                status = HttpStatus.UNAUTHORIZED;
                message = ex.getMessage();
                if (ex instanceof UnAuthorizedTenantException atE) {
                    trace = String.format("系统认证异常token =%s", atE.getData());
                } else {
                    trace = message;
                }
            } else if (!isAfterParseRequestToken) {
                message = "当前操作无效V2, 请联系系统管理员" + ex.getMessage() + ";";
            }

            ResponseEntity responseEntity =
                    ResponseBuilder.build(status, trace, message);

            response.setStatus(responseEntity.getStatusCode().value());

            String error = this.jsonProvider.getJson(responseEntity.getBody());
            response.setHeader(HttpHeaders.CONTENT_TYPE, "application/json");
            responseWithFlowTag(response,tenantContext);

            response.getOutputStream().write(error.getBytes(StandardCharsets.UTF_8));
            response.flushBuffer();
        }
    }

    @Override
    protected boolean shouldNotFilter(HttpServletRequest request) throws ServletException {
        /**
         * 重置当前用户上下文信息
         */
        TenantContext tenantContext = TenantContext.getInstance();
        tenantContext.getUserContext().reset();
        String requestUri = request.getRequestURI();

        boolean matchedResult =
                excludedUrls.stream()
                        .anyMatch(ii -> {
                                    String excludeUrl = String.format("%s%s", request.getContextPath(), ii);
                                    boolean isMatch = requestUri.startsWith(excludeUrl) || requestUri.contains("/do-confirm");
                                    if (!isMatch && ii.contains("\\w+")) {
                                        Pattern pattern = Pattern.compile(excludeUrl);
                                        Matcher matcher = pattern.matcher(requestUri);
                                        isMatch = matcher.matches();
                                    }
                                    return isMatch;
                                }
                        );
        if (matchedResult) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }


        if (requestUri.contains("/rooms") && "get".equalsIgnoreCase(request.getMethod())) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }

        /**
         * 针对点名的回复, 先按照这种方式进行处理
         */
        if ("PUT".equalsIgnoreCase(request.getMethod()) && requestUri.startsWith(String.format("%s/roll-calls", request.getContextPath()))) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }

        if (requestUri.contains(String.format("%s/swagger", request.getContextPath())) ||
                requestUri.contains("/download/resources")) {
            executeAddProviderTag(request, tenantContext, null);
            return true;
        }

        return false;
    }

    private void parseRequestToken(HttpServletRequest request,
                                   boolean ignoredDeskCheck) {
        String dispatchToken = null;
        String content = null;
        try {
            dispatchToken = ServletRequestUtil.getDispatchToken(request);
            if (!StringUtils.hasLength(dispatchToken)) {
                throw new UnAuthorizedTenantException("当前无有效dispatchToken(token不能为空)信息");
            }

            content = JwtUtil.getContentFromPaasToken(dispatchToken);

            /**
             * 如果通过token的方式来保持权限
             *
             */
            UserDetailResponse userDetailResponse =
                    this.jsonProvider.toObject(UserDetailResponse.class, content);
            if (userDetailResponse == null) {
                logger.error(String.format("当前无有效paasToken信息=%s;content=%s", dispatchToken, content));
                throw new UnAuthorizedTenantException("当前无有效paasToken信息, 请联系管理员");
            }

            /**
             * todo: 先限制仅组织用户可访问
             */
            /*
            if(!StringUtils.hasLength(userDetailResponse.getOrganizationId())) {
                throw new ForbidTenantException(String.format("非组织用户无法访问此请求:%s", userDetailResponse.getOrganizationId()));
            }

             */

            /**
             * 初始化当前token和对应的tenant user id
             * 获取当前用户的权限
             */
            TenantEmployeeContext.PermissionModel permissionModel =
                    TenantEmployeeContext.PermissionModel.create(false,null,null,null,"");

            TenantContext tenantContext = TenantContext.getInstance();
            TenantEmployeeContext.TenantUserModel userModel = tenantContext.getUserContext()
                            .init(userDetailResponse,permissionModel);

            executeAddProviderTag(request,tenantContext,userModel.getTrafficTag());

            try {
                TenantContext.getInstance().getUserContext().init(userModel);
            } catch (Exception ex) {
                if (ex.getClass().isAssignableFrom(ForbidTenantException.class) || ex.getClass().isAssignableFrom(UnAuthorizedTenantException.class)) {
                    throw ex;
                }

                throw new BadTenantException("解析用户信息无效:" + ex.getMessage(), ex);
            }
        } catch (UnAuthorizedTenantException ex) {
            logger.error("当前Token异常:token={};content={}", dispatchToken, content);
            throw ex;
        } catch (Exception ex) {
            logger.error("当前无有效dispatchToken信息={};content={};", dispatchToken, content, ex);
            throw ex;
        }
    }

    private void responseWithFlowTag(HttpServletResponse response,TenantContext tenantContext) {
        try {
            if (tenantContext.getAttachValue(TenantContext.PROVIDER_FLOW_TAG_NAME) instanceof TrafficTagValueType tagValueType) {
                response.addHeader(TenantContext.PROVIDER_FLOW_TAG_NAME, tagValueType.getTagStringValue());
            }

            response.addHeader(TenantContext.REST_REQUEST_FLOW_TAG_NAME, String.valueOf(tenantContext.getAttachValue(TenantContext.FLOW_TAG_NAME)));
            response.addHeader(TenantContext.REST_VERSION, "1.5");
            response.addHeader("t_node", this.tenantNode);
        } catch (Exception ex) {
            //todo: 增值的功能, 仅仅只是补充
        }
    }

    private void executeAddProviderTag(ServletRequest request,TenantContext tenantContext, String fallBackTrafficTag){
        DubboCommonUtils.addContextTag(request,tenantContext,fallBackTrafficTag);
    }
}
