package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.jdks.PageMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.PageMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.PageMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.PageMetaStructureRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.MetaPageStructureSettingResponse;
import com.bcxin.tenant.bcx.jdks.responses.metas.PageFormDataResponse;
import com.bcxin.tenant.bcx.jdks.responses.metas.PageMetaResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

/**
 * 页面元数据控制器
 */
@Tag(name = "PageMetaController", description = "页面元数据管理")
@RestController
@RequestMapping("/meta")
public class PageMetaController extends ControllerAbstract{
    private final PageMetaRpcProvider rpcProvider;

    public PageMetaController(PageMetaRpcProvider rpcProvider) {
        this.rpcProvider = rpcProvider;
    }

    @Operation(
            summary = "新增页面",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/modules/{moduleId}/pages")
    public void doPost(
            @PathVariable String moduleId,
            @RequestBody PageMetaRequest request) {
        request.setModuleId(moduleId);
        this.rpcProvider.create(request);
    }

    @Operation(
            summary = "编辑页面",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PutMapping("/modules/{moduleId}/pages/{id}")
    public void doPut(
            @PathVariable String moduleId,
            @PathVariable String id,
            @RequestBody PageMetaRequest request) {
        request.setModuleId(moduleId);
        this.rpcProvider.update(id, request);
    }

    @Operation(
            summary = "删除页面",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/modules/{moduleId}/pages/{id}")
    public void doDelete(
            @PathVariable String moduleId,
            @PathVariable String id) {
        this.rpcProvider.delete(id);
    }

    @Operation(
            summary = "批量删除页面",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/modules/{moduleId}/pages/batch")
    public void doBatchDelete(
            @PathVariable String moduleId,
            @RequestBody java.util.List<String> ids) {
        this.rpcProvider.batchDelete(ids);
    }

    @Operation(
            summary = "高级搜索",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/modules/{moduleId}/pages/search")
    public ResponseEntity<EntityCollection<PageMetaResponse>> doSearch(
            @PathVariable String moduleId,
            @RequestBody PageMetaSearchRequest request) {
        // 设置模块ID到搜索请求中，确保只搜索指定模块的页面
        request.setModuleId(moduleId);
        var data = this.rpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(
            summary = "获取详情",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @GetMapping("/modules/{moduleId}/pages/{id}")
    public ResponseEntity<PageMetaResponse> doGet(
            @PathVariable String moduleId,
            @PathVariable String id) {
        var data = this.rpcProvider.getById(id);

        return this.ok(data);
    }

    @Operation(
            summary = "获取表单数据",
            description = "获取指定页面的表单数据，包含id、name、formJson、settingJson",
            responses = {
                    @ApiResponse(responseCode = "200", description = "获取成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
                    @ApiResponse(responseCode = "404", description = "页面不存在返回404.")
            }
    )
    @GetMapping("/pages/{id}/form-json")
    public ResponseEntity<PageFormDataResponse> getFormJson(
            @PathVariable String id,
            @RequestHeader(value = "x-bcx-for-mobile",required = false,defaultValue = "false")boolean forMobile) {
        // 获取表单数据
        PageFormDataResponse data = this.rpcProvider.getFormJson(id,forMobile);
        return this.ok(data);
    }

    @Operation(
            summary = "保存表单JSON",
            description = "保存或更新页面的表单JSON结构",
            responses = {
                    @ApiResponse(responseCode = "200", description = "保存成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
                    @ApiResponse(responseCode = "404", description = "页面不存在返回404.")
            }
    )
    @PatchMapping("/modules/{moduleId}/pages/{id}/form-json")
    public ResponseEntity<PageMetaResponse> saveFormJson(
            @PathVariable String moduleId,
            @PathVariable String id,
            @RequestBody PageMetaStructureRequest request) {
        // 保存表单JSON
        this.rpcProvider.saveFormJson(id, request);
        
        // 返回更新后的页面信息
        var data = this.rpcProvider.getById(id);
        return this.ok(data);
    }

    @Operation(
            summary = "获取页面的配置内容(查询)",
            responses = {
                    @ApiResponse(responseCode = "200", description = "获取成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
                    @ApiResponse(responseCode = "404", description = "页面不存在返回404.")
            }
    )
    @GetMapping("/pages/{id}/setting")
    public ResponseEntity<MetaPageStructureSettingResponse> getPageSetting(@PathVariable String id,
                                                                           @RequestHeader(value = "x-bcx-for-mobile",required = false,defaultValue = "false")
                                                                           boolean forMobile) {
        // 获取表单数据
        MetaPageStructureSettingResponse data = this.rpcProvider.getPageSetting(id,forMobile);

        return this.ok(data);
    }
}
