package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.jdks.OrganRegistryFormsRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.organregistry.OrganRegistryFormsRequest;
import com.bcxin.tenant.bcx.jdks.requests.organregistry.OrganRegistryFormsSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.organregistry.OrganRegistryFormsResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 企业注册表单控制器
 */
@Tag(name = "OrganRegistryFormsBusController", description = "企业注册表单管理")
@RestController
@RequestMapping("/bus/organ-registry-forms")
public class OrganRegistryFormsBusController extends ControllerAbstract {
    private final OrganRegistryFormsRpcProvider rpcProvider;

    public OrganRegistryFormsBusController(OrganRegistryFormsRpcProvider rpcProvider) {
        this.rpcProvider = rpcProvider;
    }

    @Operation(
            summary = "新增企业注册表单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "创建成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("")
    public void doPost(@RequestBody OrganRegistryFormsRequest request) {
        this.rpcProvider.create(request);
    }

    @Operation(
            summary = "编辑企业注册表单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "更新成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PutMapping("/{id}")
    public void doPut(
            @PathVariable String id,
            @RequestBody OrganRegistryFormsRequest request) {
        this.rpcProvider.update(id, request);
    }

    @Operation(
            summary = "删除企业注册表单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "删除成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/{id}")
    public void doDelete(@PathVariable String id) {
        this.rpcProvider.delete(id);
    }

    @Operation(
            summary = "批量删除企业注册表单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "批量删除成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/batch")
    public void doBatchDelete(@RequestBody List<String> ids) {
        this.rpcProvider.batchDelete(ids);
    }

    @Operation(
            summary = "高级搜索",
            responses = {
                    @ApiResponse(responseCode = "200", description = "搜索成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/search")
    public ResponseEntity<EntityCollection<OrganRegistryFormsResponse>> doSearch(
            @RequestBody OrganRegistryFormsSearchRequest request) {
        var data = this.rpcProvider.search(request);
        return this.ok(data);
    }

    @Operation(
            summary = "获取详情",
            responses = {
                    @ApiResponse(responseCode = "200", description = "获取成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<OrganRegistryFormsResponse> doGet(@PathVariable String id) {
        var data = this.rpcProvider.getById(id);
        return this.ok(data);
    }

    @Operation(
            summary = "根据条件查找企业注册表单",
            description = "根据机构类型、行业类型、监管区域查找对应的注册表单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "查找成功"),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
                    @ApiResponse(responseCode = "404", description = "未找到匹配的表单")
            }
    )
    @GetMapping("/find-by-condition")
    public ResponseEntity<OrganRegistryFormsResponse> findByCondition(
            @RequestParam String institutionalTypeCode,
            @RequestParam String industryTypeCode,
            @RequestParam String regionCode) {
        var data = this.rpcProvider.findByCondition(institutionalTypeCode, industryTypeCode, regionCode);
        return this.ok(data);
    }
}
