package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.jdks.ModuleMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.ModuleMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.ModuleMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.ModuleMetaResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

/**
 * 模块元数据控制器
 */
@Tag(name = "ModuleMetaController", description = "模块元数据管理")
@RestController
@RequestMapping("/meta/projects/{projectId}/modules")
public class ModuleMetaController extends ControllerAbstract{
    private final ModuleMetaRpcProvider rpcProvider;

    public ModuleMetaController(ModuleMetaRpcProvider rpcProvider) {
        this.rpcProvider = rpcProvider;
    }

    @Operation(
            summary = "新增模块",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("")
    public void doPost(
            @PathVariable String projectId,
            @RequestBody ModuleMetaRequest request) {
        request.setProjectId(projectId);
        this.rpcProvider.create(request);
    }

    @Operation(
            summary = "编辑模块",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PutMapping("/{id}")
    public void doPut(
            @PathVariable String projectId,
            @PathVariable String id,
            @RequestBody ModuleMetaRequest request) {
        request.setProjectId(projectId);
        this.rpcProvider.update(id, request);
    }

    @Operation(
            summary = "删除模块",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/{id}")
    public void doDelete(
            @PathVariable String projectId,
            @PathVariable String id) {
        this.rpcProvider.delete(id);
    }

    @Operation(
            summary = "批量删除模块",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/batch")
    public void doBatchDelete(
            @PathVariable String projectId,
            @RequestBody java.util.List<String> ids) {
        this.rpcProvider.batchDelete(ids);
    }

    @Operation(
            summary = "高级搜索",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/search")
    public ResponseEntity<EntityCollection<ModuleMetaResponse>> doSearch(
            @PathVariable String projectId,
            @RequestBody ModuleMetaSearchRequest request) {
        // 设置项目ID到搜索请求中，确保只搜索指定项目的模块
        request.setProjectId(projectId);
        var data = this.rpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(
            summary = "获取详情",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<ModuleMetaResponse> doGet(
            @PathVariable String projectId,
            @PathVariable String id) {
        var data = this.rpcProvider.getById(id);

        return this.ok(data);
    }
}
