package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.jdks.MenuMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.metas.MenuMetaRequest;
import com.bcxin.tenant.bcx.jdks.requests.metas.MenuMetaSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.MenuMetaResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

/**
 * 菜单元数据控制器
 */
@Tag(name = "MenuMetaController", description = "菜单元数据管理")
@RestController
@RequestMapping("/meta/projects/{projectId}/menus")
public class MenuMetaController extends ControllerAbstract{
    private final MenuMetaRpcProvider rpcProvider;

    public MenuMetaController(MenuMetaRpcProvider rpcProvider) {
        this.rpcProvider = rpcProvider;
    }

    @Operation(
            summary = "新增菜单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("")
    public void doPost(
            @PathVariable String projectId,
            @RequestBody MenuMetaRequest request) {
        request.setProjectId(projectId);
        this.rpcProvider.create(request);
    }

    @Operation(
            summary = "编辑菜单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PutMapping("/{id}")
    public void doPut(
            @PathVariable String projectId,
            @PathVariable String id,
            @RequestBody MenuMetaRequest request) {
        request.setProjectId(projectId);
        this.rpcProvider.update(id, request);
    }

    @Operation(
            summary = "删除菜单",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/{id}")
    public void doDelete(
            @PathVariable String projectId,
            @PathVariable String id) {
        this.rpcProvider.delete(id);
    }

    @Operation(
            summary = "高级搜索",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/search")
    public ResponseEntity<EntityCollection<MenuMetaResponse>> doSearch(
            @PathVariable String projectId,
            @RequestBody MenuMetaSearchRequest request) {
        // 设置项目ID到搜索请求中，确保只搜索指定项目的菜单
        request.setProjectId(projectId);
        var data = this.rpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(
            summary = "获取详情",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<MenuMetaResponse> doGet(
            @PathVariable String projectId,
            @PathVariable String id) {
        var data = this.rpcProvider.getById(id);

        return this.ok(data);
    }
}
