package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.bcxin.tenant.bcx.infrastructures.UserDetailResponse;
import com.bcxin.tenant.bcx.infrastructures.components.JsonProvider;
import com.bcxin.tenant.bcx.jdks.IdentityRpcProvider;
import com.bcxin.tenant.bcx.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.bcx.rest.apis.controllers.responses.DispatchResponse;
import com.bcxin.tenant.bcx.rest.apis.utils.JwtUtil;
import com.bcxin.tenant.bcx.rest.apis.utils.ServletRequestUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

@Tag(name = "IdentityController", description = "身份认证")
@RestController
@RequestMapping("/identity")
public class IdentityController extends ControllerAbstract {
    private final IdentityRpcProvider identityRpcProvider;
    private final JsonProvider jsonProvider;

    private final HotCacheProvider hotCacheProvider;

    public IdentityController(
            IdentityRpcProvider identityRpcProvider,
            JsonProvider jsonProvider,
            HotCacheProvider hotCacheProvider) {
        this.identityRpcProvider = identityRpcProvider;
        this.jsonProvider = jsonProvider;
        this.hotCacheProvider = hotCacheProvider;
    }

    @Operation(
            summary = "单点登入调度系统", description = "单点登入调度系统-获取调度系统 dispatchToken",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回调度系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken",
                            description = "来自V5的accessToken"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "accountType", description = "账号类型: 使用百保盾APP的时候使用APP(现有方式); 集成第三方设备的时候的使用Device;进入GIS的调度台(WEB及后期APP的调度界面)进行调度的时候使用Desk")
            }
    )
    @PostMapping("/auto-login")
    public ResponseEntity<DispatchResponse> autoLogin(HttpServletRequest servletRequest,
                                                      @RequestParam(value = "org_id",required = false) String organizationId
                                                      ) {
        String bearerToken = ServletRequestUtil.getBearerToken(servletRequest);

        if (!StringUtils.hasLength(bearerToken)) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }
        String tenantUserId =
                JwtUtil.getSaasUserIdFromToken(bearerToken);

        UserDetailResponse response =
                this.identityRpcProvider.getByIdAndOrganizationId(tenantUserId, organizationId);

        if(response==null) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }

        String content = this.jsonProvider.getJson(response);
        String paasToken = JwtUtil.getPaasTokenFromIdentityData(content);

        return this.ok(DispatchResponse.create(paasToken, response));
    }
}