package com.bcxin.tenant.bcx.rest.apis.controllers;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.handler.SheetWriteHandler;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.holder.WriteSheetHolder;
import com.alibaba.excel.write.metadata.holder.WriteWorkbookHolder;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ForbidTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.bcx.jdks.PageDataRpcProvider;
import com.bcxin.tenant.bcx.jdks.PageMetaRpcProvider;
import com.bcxin.tenant.bcx.jdks.requests.pages.BatchPageDataRequest;
import com.bcxin.tenant.bcx.jdks.requests.pages.PageDataActionRequest;
import com.bcxin.tenant.bcx.jdks.requests.pages.PageDataRequest;
import com.bcxin.tenant.bcx.jdks.requests.pages.PageDataSearchRequest;
import com.bcxin.tenant.bcx.jdks.responses.metas.MetaPageStructureSettingResponse;
import com.bcxin.tenant.bcx.jdks.responses.pages.PageDataResponse;
import com.bcxin.tenant.bcx.rest.apis.components.ExtractDataComponent;
import com.bcxin.tenant.bcx.rest.apis.controllers.requests.DynamicDataImportRequest;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.apache.poi.ss.usermodel.Sheet;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

@Tag(name = "DynamicPageDataController", description = "动态表单数据")
@RestController
@RequestMapping("/dynamic-pages/{pageId}/data")
public class DynamicPageDataController extends ControllerAbstract {
    private final PageDataRpcProvider rpcProvider;
    private final PageMetaRpcProvider pageMetaRpcProvider;
    private final ExtractDataComponent extractDataComponent;

    public DynamicPageDataController(PageDataRpcProvider rpcProvider,
                                     PageMetaRpcProvider pageMetaRpcProvider,
                                     ExtractDataComponent extractDataComponent) {
        this.rpcProvider = rpcProvider;
        this.pageMetaRpcProvider = pageMetaRpcProvider;
        this.extractDataComponent = extractDataComponent;
    }

    @Operation(
            summary = "新增通用表单数据",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping
    public ResponseEntity<String> doPost(
            @PathVariable String pageId,
            @RequestBody PageDataRequest request) {
        request.setPageId(pageId);
        String id = this.rpcProvider.create(request);

        return this.ok(id);
    }

    @Operation(
            summary = "编辑通用表单数据",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @Deprecated
    @PutMapping("/{id}")
    public ResponseEntity doPut(
            @PathVariable String pageId,
            @PathVariable String id, @RequestBody PageDataRequest request) {
        request.setPageId(pageId);
        this.rpcProvider.update(id, request);

        return this.ok();
    }

    @Operation(
            summary = "对数据执行Action(修改, 确认, 审批)操作",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PutMapping("/do-action")
    public ResponseEntity doAction(
            @PathVariable String pageId,
            @RequestBody PageDataActionRequest request) {
        request.setPageId(pageId);
        this.rpcProvider.doAction(request);

        return this.ok();
    }

    @Operation(
            summary = "删除通用表单数据",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @DeleteMapping("/{id}")
    public ResponseEntity doDelete(
            @PathVariable String pageId,
            @PathVariable String id) {
        this.rpcProvider.delete(pageId, id);

        return this.ok();
    }

    @Operation(
            summary = "获取通用表单数据",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity doGet(
            @PathVariable String pageId,
            @PathVariable String id) {
        PageDataResponse rp = this.rpcProvider.getById(pageId,id);
        if (rp == null) {
            return this.notFound();
        }

        return this.ok(rp);
    }

    @Operation(
            summary = "搜索通用表单数据",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/search")
    public ResponseEntity doSearch(
            @PathVariable String pageId,
            @RequestBody PageDataSearchRequest request) {
        request.setPageId(pageId);
        EntityCollection<PageDataResponse> rps = this.rpcProvider.search(request);

        return this.ok(rps);
    }

    @Operation(summary = "导出-通用表单数据", description = "导出-通用表单数据 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回签到信息列表及驻勤点信息列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/export")
    public void export(
            @PathVariable String pageId,
            @RequestBody PageDataSearchRequest request,
                       HttpServletResponse response) throws IOException {
        request.setPageId(pageId);
        MetaPageStructureSettingResponse setting =
                this.pageMetaRpcProvider.getPageSetting(
                        request.getPageId(),
                        request.isForMobile());
        if (setting == null) {
            throw new NoFoundTenantException();
        }

        String rqName = "通用数据";
        if (StringUtils.hasLength(setting.getPageName())) {
            rqName = setting.getPageName();
        }

        String name = String.format("%s-%s", rqName, new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        EntityCollection<PageDataResponse> entityCollection = this.rpcProvider.search(request);
        Collection<PageDataResponse> dataList = entityCollection.getData();

        /**
         * 保证顺序
         */
        Set<String> headerKeys = new HashSet<>();
        Map<String, String> headerSet = new HashMap<>();
        setting.getFields().forEach(ix -> {
            headerSet.put(ix.getField(), ix.getLabel());
            headerKeys.add(ix.getField());
        });

        String key_createdTime = "createdTime";
        String key_lastUpdatedTime = "lastUpdatedTime";
        headerSet.put(key_createdTime, "创建时间");
        headerSet.put(key_lastUpdatedTime, "最后修改时间");
        headerKeys.add(key_createdTime);
        headerKeys.add(key_lastUpdatedTime);


        List<List<String>> head = new ArrayList<>();
        for (String key : headerKeys) {
            head.add(Collections.singletonList(headerSet.get(key)));
        }

        List<List<Object>> dataToWrite = new ArrayList<>();
        for (PageDataResponse data : dataList) {
            List<Object> rowData = new ArrayList<>();
            Map<String, Object> items = data.getItems();
            for (String key : headerKeys) {
                if (key_createdTime.equals(key) || key_lastUpdatedTime.equals(key)) {
                    if (key_createdTime.equals(key)) {
                        rowData.add(data.getCreatedTime());
                    } else {
                        rowData.add(data.getLastUpdatedTime());
                    }
                } else {
                    rowData.add(items.getOrDefault(key, ""));
                }
            }

            dataToWrite.add(rowData);
        }

        response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
        response.setCharacterEncoding("utf-8");
        // 这里URLEncoder.encode可以防止中文乱码 当然和easyexcel没有关系
        String fileName = URLEncoder.encode(name, "UTF-8").replaceAll("\\+", "%20");
        response.setHeader("Content-disposition", "attachment;filename*=utf-8''" + URLEncoder.encode(name, "UTF-8") + ".xlsx");


        WriteSheet writeSheet = EasyExcel.writerSheet(fileName).build();
        ExcelWriterBuilder builder = EasyExcel.write(response.getOutputStream());
        builder.head(head)
                .sheet(writeSheet.getSheetName())
                .registerWriteHandler(new SheetWriteHandler() {
                    @Override
                    public void afterSheetCreate(WriteWorkbookHolder writeWorkbookHolder,
                                                 WriteSheetHolder writeSheetHolder) {
                        Sheet sheet = writeSheetHolder.getSheet();
                        // 为每一列设置宽度
                        for (int i = 0; i < head.size(); i++) {
                            if (i >= head.size() - 2) {
                                sheet.setColumnWidth(i, 20 * 256);  // 第一列20个字符宽
                            } else {
                                sheet.setColumnWidth(i, 30 * 256);  // 其他列25个字符宽
                            }
                        }
                    }
                })
                .doWrite(dataToWrite);
    }

    @Operation(
            summary = "执行批量导入",
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回系统的 dispatchToken."),
                    @ApiResponse(responseCode = "401", description = "无效用户返回401."),
                    @ApiResponse(responseCode = "403", description = "禁止使用的企业, 该接口返回403."),
            }
    )
    @PostMapping("/do-import")
    public ResponseEntity doPost(
            @PathVariable String pageId,
            @RequestBody DynamicDataImportRequest request,
            HttpServletRequest servletRequest) throws IOException {
        if(!StringUtils.hasLength(request.getPath())) {
            throw new ArgumentTenantException("导入的附件不能为空");
        }

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }

        if(!StringUtils.hasLength(userModel.getOrganizationId())) {
            throw new ForbidTenantException("非租户用户不允许此操作");
        }

        Collection<Map<String, Object>> batchImportContractRequests =
                extractDataComponent.extract(1, request.getPath(), excelMapValues -> {
                    return excelMapValues.stream().map(ii -> {
                        Map<String, Object> rt = new HashMap<>();
                        if(!CollectionUtils.isEmpty(request.getDefaultParams())) {
                            rt.putAll(request.getDefaultParams());
                        }

                        rt.put("credential_type", "0");
                        rt.put("name", ii.get(0));
                        rt.put("credential_number", ii.get(1));
                        rt.put("telephone", ii.get(2));
                        return rt;
                    }).collect(Collectors.toList());
                });
        BatchPageDataRequest batchPageDataRequest = BatchPageDataRequest.create(
                pageId,
                batchImportContractRequests
        );

        this.rpcProvider.batchCreate(batchPageDataRequest);

        return this.ok();
    }
}
